"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const errors_1 = require("../core/errors");
const payment_service_1 = require("../modules/payments/payment.service");
const paystack_1 = require("../integrations/paystack");
const flutterwave_1 = require("../integrations/flutterwave");
const env_1 = require("../config/env");
const router = (0, express_1.Router)();
router.post('/paystack', (0, errors_1.asyncHandler)(async (req, res) => {
    const signature = req.headers['x-paystack-signature'];
    const rawBody = JSON.stringify(req.body);
    console.log('[Webhook] Paystack webhook received:', {
        signature: signature ? 'present' : 'missing',
        bodyKeys: Object.keys(req.body),
        event: req.body.event,
    });
    try {
        const isValid = paystack_1.paystackAdapter.verifyWebhookSignature(signature, rawBody);
        if (!isValid) {
            console.error('[Webhook] Invalid Paystack webhook signature');
            return res.status(401).json({
                success: false,
                message: 'Invalid webhook signature',
            });
        }
        const result = await payment_service_1.paymentService.processWebhook({
            provider: 'paystack',
            signature,
            rawBody,
            event: req.body,
        });
        console.log('[Webhook] Paystack webhook processed successfully:', result);
        return res.status(200).json({
            success: true,
            message: 'Webhook processed successfully',
            data: result,
        });
    }
    catch (error) {
        console.error('[Webhook] Paystack webhook processing failed:', error);
        return res.status(500).json({
            success: false,
            message: 'Webhook processing failed',
            error: error instanceof Error ? error.message : 'Unknown error',
        });
    }
}));
router.post('/flutterwave', (0, errors_1.asyncHandler)(async (req, res) => {
    const signature = req.headers['verif-hash'];
    const rawBody = JSON.stringify(req.body);
    console.log('[Webhook] Flutterwave webhook received:', {
        signature: signature ? 'present' : 'missing',
        bodyKeys: Object.keys(req.body),
        event: req.body.event,
    });
    try {
        const isValid = flutterwave_1.flutterwaveAdapter.verifyWebhookSignature(signature, rawBody);
        if (!isValid) {
            console.error('[Webhook] Invalid Flutterwave webhook signature');
            return res.status(401).json({
                success: false,
                message: 'Invalid webhook signature',
            });
        }
        const result = await payment_service_1.paymentService.processWebhook({
            provider: 'flutterwave',
            signature,
            rawBody,
            event: req.body,
        });
        console.log('[Webhook] Flutterwave webhook processed successfully:', result);
        return res.status(200).json({
            success: true,
            message: 'Webhook processed successfully',
            data: result,
        });
    }
    catch (error) {
        console.error('[Webhook] Flutterwave webhook processing failed:', error);
        return res.status(500).json({
            success: false,
            message: 'Webhook processing failed',
            error: error instanceof Error ? error.message : 'Unknown error',
        });
    }
}));
router.post('/aspfiy', (0, errors_1.asyncHandler)(async (req, res) => {
    const requestId = req.requestId || `webhook-${Date.now()}`;
    const startTime = Date.now();
    console.log(`[Webhook-${requestId}] Aspfiy webhook received at ${new Date().toISOString()}`);
    const signature = (req.headers['x-wiaxy-signature'] ||
        req.headers['X-Wiaxy-signature'] ||
        req.headers['x-webhook-signature'] ||
        req.headers['x-aspfiy-signature'] ||
        req.headers['x-signature'] ||
        req.headers['signature']);
    const rawBody = req.rawBody || JSON.stringify(req.body);
    console.log(`[Webhook-${requestId}] Request details:`, {
        signature: signature ? 'present' : 'missing',
        signatureValue: signature ? `${signature.substring(0, 10)}...` : 'none',
        signatureHeaders: Object.keys(req.headers).filter(h => h.toLowerCase().includes('signature')),
        bodyKeys: Object.keys(req.body),
        event: req.body.event,
        bodyLength: rawBody.length,
        rawBodyAvailable: !!req.rawBody,
        userAgent: req.headers['user-agent'],
        contentType: req.headers['content-type'],
        ip: req.ip || req.connection.remoteAddress,
        timestamp: new Date().toISOString()
    });
    try {
        if (!env_1.env.ASPFIY_WEBHOOK_SECRET) {
            console.log(`[Webhook-${requestId}] ASPFIY_WEBHOOK_SECRET not configured — skipping signature verification`);
        }
        else {
            console.log(`[Webhook-${requestId}] ASPFIY_WEBHOOK_SECRET configured — signature verification required`);
            if (!signature) {
                console.error(`[Webhook-${requestId}] Signature missing but secret configured - rejecting webhook`);
                return res.status(401).json({
                    success: false,
                    message: 'Invalid webhook signature',
                    requestId,
                    timestamp: new Date().toISOString()
                });
            }
            console.log(`[Webhook-${requestId}] Verifying webhook signature...`);
        }
        console.log(`[Webhook-${requestId}] Processing webhook event: ${req.body.event}`);
        const result = await payment_service_1.paymentService.processWebhook({
            provider: 'aspfiy',
            signature: signature || 'no-signature',
            rawBody,
            event: req.body,
        });
        const processingTime = Date.now() - startTime;
        console.log(`[Webhook-${requestId}] Webhook processed successfully in ${processingTime}ms:`, {
            success: result.success,
            transactionId: result.transactionId,
            message: result.message
        });
        return res.status(200).json({
            success: true,
            message: 'Webhook processed successfully',
            data: result,
            requestId,
            processingTime,
            timestamp: new Date().toISOString()
        });
    }
    catch (error) {
        const processingTime = Date.now() - startTime;
        console.error(`[Webhook-${requestId}] Webhook processing failed after ${processingTime}ms:`, {
            error: error instanceof Error ? error.message : 'Unknown error',
            stack: error instanceof Error ? error.stack : undefined,
            event: req.body.event,
            bodyKeys: Object.keys(req.body)
        });
        return res.status(500).json({
            success: false,
            message: 'Webhook processing failed',
            error: error instanceof Error ? error.message : 'Unknown error',
            requestId,
            processingTime,
            timestamp: new Date().toISOString()
        });
    }
}));
router.get('/health', (_req, res) => {
    res.status(200).json({
        success: true,
        message: 'Webhook service is healthy',
        timestamp: new Date().toISOString(),
        environment: env_1.env.NODE_ENV,
        server: {
            uptime: process.uptime(),
            memory: process.memoryUsage(),
            nodeVersion: process.version,
        },
        providers: {
            paystack: {
                enabled: !!env_1.env.PAYSTACK_SECRET_KEY,
                webhookSecret: !!env_1.env.PAYSTACK_WEBHOOK_SECRET,
                baseUrl: 'https://api.paystack.co',
            },
            flutterwave: {
                enabled: !!env_1.env.FLUTTERWAVE_SECRET_KEY,
                baseUrl: 'https://api.flutterwave.com',
            },
            aspfiy: {
                enabled: !!env_1.env.ASPFIY_SECRET_KEY,
                webhookSecret: !!env_1.env.ASPFIY_WEBHOOK_SECRET,
                baseUrl: env_1.env.ASPFIY_BASE_URL || 'Not configured',
                forceStub: !!env_1.env.ASPFIY_FORCE_STUB,
            },
        },
        webhook: {
            rawBodyMiddleware: 'enabled',
            signatureVerification: 'enhanced',
            logging: 'enhanced',
        },
    });
});
router.get('/monitor', (0, errors_1.asyncHandler)(async (_req, res) => {
    try {
        res.status(200).json({
            success: true,
            message: 'Webhook monitoring data',
            timestamp: new Date().toISOString(),
            monitoring: {
                status: 'active',
                lastCheck: new Date().toISOString(),
                webhookEndpoints: [
                    {
                        provider: 'aspfiy',
                        endpoint: '/api/v1/webhooks/aspfiy',
                        status: 'active',
                        lastActivity: 'No recent activity',
                    },
                    {
                        provider: 'paystack',
                        endpoint: '/api/v1/webhooks/paystack',
                        status: 'active',
                        lastActivity: 'No recent activity',
                    },
                    {
                        provider: 'flutterwave',
                        endpoint: '/api/v1/webhooks/flutterwave',
                        status: 'active',
                        lastActivity: 'No recent activity',
                    },
                ],
            },
            recommendations: [
                'Monitor server logs for webhook processing',
                'Check Aspfiy dashboard for webhook delivery status',
                'Verify webhook URL configuration in Aspfiy',
                'Test webhook endpoints using diagnostic tools',
            ],
        });
    }
    catch (error) {
        res.status(500).json({
            success: false,
            message: 'Failed to get monitoring data',
            error: error instanceof Error ? error.message : 'Unknown error',
        });
    }
}));
if (env_1.env.NODE_ENV === 'development') {
    router.post('/debug/aspfiy', (0, errors_1.asyncHandler)(async (req, res) => {
        const signature = (req.headers['x-aspfiy-signature'] ||
            req.headers['x-signature'] ||
            req.headers['signature'] ||
            req.headers['x-webhook-signature']);
        const rawBody = req.rawBody || JSON.stringify(req.body);
        res.status(200).json({
            success: true,
            message: 'Webhook debug info',
            data: {
                headers: {
                    signature: signature || 'missing',
                    allSignatureHeaders: Object.keys(req.headers).filter(h => h.toLowerCase().includes('signature')),
                    contentType: req.headers['content-type'],
                    userAgent: req.headers['user-agent'],
                },
                body: {
                    parsed: req.body,
                    raw: rawBody,
                    rawLength: rawBody.length,
                },
                environment: {
                    webhookSecret: env_1.env.ASPFIY_WEBHOOK_SECRET ? 'configured' : 'missing',
                    nodeEnv: env_1.env.NODE_ENV,
                },
                signatureVerification: {
                    expected: env_1.env.ASPFIY_WEBHOOK_SECRET
                        ? 'calculated'
                        : 'not configured',
                    received: signature || 'missing',
                },
            },
        });
    }));
    router.get('/debug/wallet/:userId', (0, errors_1.asyncHandler)(async (req, res) => {
        const { userId } = req.params;
        try {
            const { walletService } = await Promise.resolve().then(() => __importStar(require('../modules/wallet/wallet.service')));
            const balances = await walletService.getUserBalances(userId);
            res.status(200).json({
                success: true,
                message: 'Wallet debug info',
                data: {
                    userId,
                    balances,
                    timestamp: new Date().toISOString(),
                },
            });
        }
        catch (error) {
            res.status(500).json({
                success: false,
                message: 'Failed to get wallet info',
                error: error instanceof Error ? error.message : 'Unknown error',
            });
        }
    }));
    router.get('/debug/transactions/:userId', (0, errors_1.asyncHandler)(async (req, res) => {
        const { userId } = req.params;
        const limit = parseInt(req.query.limit) || 10;
        try {
            const { Transaction } = await Promise.resolve().then(() => __importStar(require('../database/models/transactions')));
            const transactions = await Transaction.find({ userId })
                .sort({ createdAt: -1 })
                .limit(limit)
                .lean();
            res.status(200).json({
                success: true,
                message: 'Recent transactions',
                data: {
                    userId,
                    transactions: transactions.map(tx => ({
                        id: tx._id,
                        reference: tx.reference,
                        type: tx.type,
                        amount: tx.amount,
                        currency: tx.currency,
                        status: tx.status,
                        description: tx.description,
                        balanceBefore: tx.balanceBefore,
                        balanceAfter: tx.balanceAfter,
                        createdAt: tx.createdAt,
                        metadata: tx.metadata,
                    })),
                    count: transactions.length,
                    timestamp: new Date().toISOString(),
                },
            });
        }
        catch (error) {
            res.status(500).json({
                success: false,
                message: 'Failed to get transactions',
                error: error instanceof Error ? error.message : 'Unknown error',
            });
        }
    }));
    router.get('/debug/email/:email', (0, errors_1.asyncHandler)(async (req, res) => {
        const { email } = req.params;
        try {
            const { User } = await Promise.resolve().then(() => __importStar(require('../modules/users/user.model')));
            const user = await User.findOne({ email: email.toLowerCase() });
            res.status(200).json({
                success: true,
                message: 'Email check result',
                data: {
                    email,
                    exists: !!user,
                    user: user
                        ? {
                            id: user._id,
                            email: user.email,
                            firstName: user.firstName,
                            lastName: user.lastName,
                            isEmailVerified: user.isEmailVerified,
                            isDeleted: user.isDeleted,
                            createdAt: user.createdAt,
                        }
                        : null,
                    timestamp: new Date().toISOString(),
                },
            });
        }
        catch (error) {
            res.status(500).json({
                success: false,
                message: 'Failed to check email',
                error: error instanceof Error ? error.message : 'Unknown error',
            });
        }
    }));
    router.delete('/debug/cleanup-user/:email', (0, errors_1.asyncHandler)(async (req, res) => {
        const { email } = req.params;
        try {
            const { User } = await Promise.resolve().then(() => __importStar(require('../modules/users/user.model')));
            const user = await User.findOne({
                email: email.toLowerCase(),
                isEmailVerified: false,
                isDeleted: { $ne: true },
            });
            if (!user) {
                return res.status(404).json({
                    success: false,
                    message: 'User not found or already verified/deleted',
                    data: { email },
                });
            }
            await User.findByIdAndUpdate(user._id, {
                $set: {
                    isDeleted: true,
                    deletedAt: new Date(),
                    email: `deleted_${Date.now()}_${user.email}`,
                },
            });
            res.status(200).json({
                success: true,
                message: 'User cleaned up successfully',
                data: {
                    email,
                    userId: user._id,
                    action: 'soft_deleted',
                },
            });
        }
        catch (error) {
            res.status(500).json({
                success: false,
                message: 'Failed to cleanup user',
                error: error instanceof Error ? error.message : 'Unknown error',
            });
        }
    }));
    router.delete('/debug/cleanup-habud', (0, errors_1.asyncHandler)(async (req, res) => {
        try {
            const { User } = await Promise.resolve().then(() => __importStar(require('../modules/users/user.model')));
            const user = await User.findOne({
                email: 'habud@gmail.com',
                isDeleted: { $ne: true },
            });
            if (!user) {
                return res.status(404).json({
                    success: false,
                    message: 'User not found',
                    data: { email: 'habud@gmail.com' },
                });
            }
            await User.findByIdAndUpdate(user._id, {
                $set: {
                    isDeleted: true,
                    deletedAt: new Date(),
                    email: `deleted_${Date.now()}_${user.email}`,
                },
            });
            res.status(200).json({
                success: true,
                message: 'User cleaned up successfully',
                data: {
                    email: 'habud@gmail.com',
                    userId: user._id,
                    action: 'soft_deleted',
                },
            });
        }
        catch (error) {
            res.status(500).json({
                success: false,
                message: 'Failed to cleanup user',
                error: error instanceof Error ? error.message : 'Unknown error',
            });
        }
    }));
    router.post('/debug/credit-wallet/:userId', (0, errors_1.asyncHandler)(async (req, res) => {
        const { userId } = req.params;
        const { amount = 10, reference = `TEST-${Date.now()}` } = req.body;
        try {
            const { walletService } = await Promise.resolve().then(() => __importStar(require('../modules/wallet/wallet.service')));
            const result = await walletService.credit({
                userId,
                currency: 'NGN',
                amount,
                reference,
                category: 'funding',
                description: `Test credit - ${reference}`,
                metadata: { source: 'debug_endpoint' },
            });
            res.status(200).json({
                success: true,
                message: 'Wallet credited successfully',
                data: {
                    userId,
                    amount,
                    reference,
                    transactionId: result.transaction.id,
                    newBalance: result.wallet.availableBalance,
                    walletId: result.wallet.id,
                },
            });
        }
        catch (error) {
            res.status(500).json({
                success: false,
                message: 'Failed to credit wallet',
                error: error instanceof Error ? error.message : 'Unknown error',
            });
        }
    }));
    router.post('/debug/fix-transaction/:reference', (0, errors_1.asyncHandler)(async (req, res) => {
        const { reference } = req.params;
        try {
            const { Transaction } = await Promise.resolve().then(() => __importStar(require('../database/models/transactions')));
            const { walletService } = await Promise.resolve().then(() => __importStar(require('../modules/wallet/wallet.service')));
            const transaction = await Transaction.findOne({ reference });
            if (!transaction) {
                return res.status(404).json({
                    success: false,
                    message: 'Transaction not found',
                    data: { reference },
                });
            }
            if (transaction.status !== 'successful') {
                return res.status(400).json({
                    success: false,
                    message: 'Transaction is not successful',
                    data: { reference, status: transaction.status },
                });
            }
            if (!transaction.userId) {
                return res.status(400).json({
                    success: false,
                    message: 'Transaction missing userId',
                    data: { reference },
                });
            }
            const result = await walletService.credit({
                userId: transaction.userId.toString(),
                currency: transaction.currency,
                amount: transaction.amount,
                reference: transaction.reference,
                category: transaction.category,
                description: transaction.description,
                metadata: transaction.metadata,
            });
            res.status(200).json({
                success: true,
                message: 'Transaction fixed successfully',
                data: {
                    reference,
                    transactionId: result.transaction.id,
                    newBalance: result.wallet.availableBalance,
                    walletId: result.wallet.id,
                    balanceBefore: result.transaction.balanceBefore,
                    balanceAfter: result.transaction.balanceAfter,
                },
            });
        }
        catch (error) {
            res.status(500).json({
                success: false,
                message: 'Failed to fix transaction',
                error: error instanceof Error ? error.message : 'Unknown error',
            });
        }
    }));
    router.post('/debug/fix-aspfiy-transaction', (0, errors_1.asyncHandler)(async (req, res) => {
        const { reference = 'R-FHLEFVUQJG' } = req.body;
        try {
            const { Transaction } = await Promise.resolve().then(() => __importStar(require('../database/models/transactions')));
            const { walletService } = await Promise.resolve().then(() => __importStar(require('../modules/wallet/wallet.service')));
            const transaction = await Transaction.findOne({ reference });
            if (!transaction) {
                return res.status(404).json({
                    success: false,
                    message: 'Transaction not found',
                    data: { reference },
                });
            }
            console.log('[Debug] Found transaction:', {
                reference: transaction.reference,
                amount: transaction.amount,
                balanceBefore: transaction.balanceBefore,
                balanceAfter: transaction.balanceAfter,
                status: transaction.status,
                userId: transaction.userId,
            });
            if (transaction.status !== 'successful') {
                return res.status(400).json({
                    success: false,
                    message: 'Transaction is not successful',
                    data: { reference, status: transaction.status },
                });
            }
            if (!transaction.userId) {
                return res.status(400).json({
                    success: false,
                    message: 'Transaction missing userId',
                    data: { reference },
                });
            }
            const wallet = await walletService.getWallet(transaction.userId.toString(), transaction.currency);
            console.log('[Debug] Current wallet balance:', {
                walletId: wallet.id,
                currentBalance: wallet.availableBalance,
                expectedBalance: transaction.balanceAfter,
            });
            const result = await walletService.credit({
                userId: transaction.userId.toString(),
                currency: transaction.currency,
                amount: transaction.amount,
                reference: transaction.reference,
                category: transaction.category,
                description: transaction.description,
                metadata: transaction.metadata,
            });
            res.status(200).json({
                success: true,
                message: 'Aspfiy transaction fixed successfully',
                data: {
                    reference,
                    transactionId: result.transaction.id,
                    oldBalance: wallet.availableBalance,
                    newBalance: result.wallet.availableBalance,
                    walletId: result.wallet.id,
                    balanceBefore: result.transaction.balanceBefore,
                    balanceAfter: result.transaction.balanceAfter,
                    amount: transaction.amount,
                    correction: result.wallet.availableBalance - wallet.availableBalance,
                },
            });
        }
        catch (error) {
            console.error('[Debug] Failed to fix Aspfiy transaction:', error);
            res.status(500).json({
                success: false,
                message: 'Failed to fix Aspfiy transaction',
                error: error instanceof Error ? error.message : 'Unknown error',
            });
        }
    }));
}
if (env_1.env.NODE_ENV === 'development') {
    router.post('/test/paystack', (0, errors_1.asyncHandler)(async (req, res) => {
        const testEvent = {
            event: 'charge.success',
            data: {
                id: 123456789,
                domain: 'test',
                amount: 100000,
                currency: 'NGN',
                source: 'api',
                reason: 'Test payment',
                recipient: 0,
                status: 'success',
                transfer_code: 'TRF-TEST-123',
                created_at: new Date().toISOString(),
                updated_at: new Date().toISOString(),
                metadata: {
                    custom_fields: [],
                    referrer: 'test',
                },
                reference: req.body.reference || 'TEST-REF-123',
                fees: 0,
                customer: {
                    id: 1,
                    first_name: 'Test',
                    last_name: 'Customer',
                    email: 'test@example.com',
                    customer_code: 'CUS_TEST',
                    phone: '+2348012345678',
                    metadata: null,
                    risk_action: 'default',
                    international_format_phone: '+2348012345678',
                },
            },
        };
        const result = await payment_service_1.paymentService.processWebhook({
            provider: 'paystack',
            signature: 'test-signature',
            rawBody: JSON.stringify(testEvent),
            event: testEvent,
        });
        res.status(200).json({
            success: true,
            message: 'Test webhook processed',
            data: result,
        });
    }));
    router.post('/test/flutterwave', (0, errors_1.asyncHandler)(async (req, res) => {
        const testEvent = {
            event: 'charge.completed',
            data: {
                id: 123456789,
                tx_ref: req.body.tx_ref || 'TEST-TX-REF-123',
                flw_ref: 'FLW-TEST-123',
                device_fingerprint: 'test-fingerprint',
                amount: 1000,
                currency: 'NGN',
                charged_amount: 1000,
                app_fee: 10,
                merchant_fee: 0,
                processor_response: 'Approved',
                auth_model: 'PIN',
                ip: '127.0.0.1',
                narration: 'Test transaction',
                status: 'successful',
                payment_type: 'card',
                created_at: new Date().toISOString(),
                updated_at: new Date().toISOString(),
                account_id: 1,
                customer: {
                    id: 1,
                    phone_number: '+2348012345678',
                    name: 'Test Customer',
                    email: 'test@example.com',
                    created_at: new Date().toISOString(),
                },
                meta: {},
            },
        };
        const result = await payment_service_1.paymentService.processWebhook({
            provider: 'flutterwave',
            signature: 'test-signature',
            rawBody: JSON.stringify(testEvent),
            event: testEvent,
        });
        res.status(200).json({
            success: true,
            message: 'Test webhook processed',
            data: result,
        });
    }));
    router.post('/test/aspfiy', (0, errors_1.asyncHandler)(async (req, res) => {
        const testEvent = {
            event: 'payment.successful',
            data: {
                id: `ASP-${Date.now()}`,
                reference: req.body.reference || 'TEST-ASP-REF-123',
                amount: 1000,
                currency: 'NGN',
                status: 'successful',
                payment_type: 'bank_transfer',
                account_number: '1234567890',
                account_name: 'Test Account Name',
                bank_name: 'Test Bank',
                customer_email: 'test@example.com',
                customer_name: 'Test Customer',
                customer_phone: '+2348012345678',
                created_at: new Date().toISOString(),
                updated_at: new Date().toISOString(),
                metadata: {
                    test: true,
                },
                fees: 10,
                net_amount: 990,
            },
        };
        const result = await payment_service_1.paymentService.processWebhook({
            provider: 'aspfiy',
            signature: 'test-signature',
            rawBody: JSON.stringify(testEvent),
            event: testEvent,
        });
        res.status(200).json({
            success: true,
            message: 'Test webhook processed',
            data: result,
        });
    }));
}
exports.default = router;
