"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PlanSyncService = void 0;
class PlanSyncService {
    servicePlanRepo;
    serviceProviderRepo;
    asbDataProvider;
    constructor(servicePlanRepo, serviceProviderRepo, asbDataProvider) {
        this.servicePlanRepo = servicePlanRepo;
        this.serviceProviderRepo = serviceProviderRepo;
        this.asbDataProvider = asbDataProvider;
    }
    async syncPlansFromASBData() {
        const results = {
            synced: 0,
            updated: 0,
            errors: []
        };
        try {
            const asbDataProvider = await this.serviceProviderRepo.findBySlug('asbdata');
            if (!asbDataProvider) {
                throw new Error('ASBData provider not found');
            }
            const networks = ['mtn', 'airtel', 'glo', '9mobile'];
            for (const network of networks) {
                try {
                    console.log(`[PlanSync] Syncing plans for ${network}...`);
                    const asbDataPlans = await this.asbDataProvider.getDataPlans(network);
                    for (const asbPlan of asbDataPlans) {
                        try {
                            const existingPlan = await this.servicePlanRepo.findByPlanId(asbPlan.planCode, asbDataProvider._id.toString());
                            const planData = {
                                providerId: asbDataProvider._id.toString(),
                                planId: asbPlan.planCode,
                                name: this.formatPlanName(asbPlan),
                                type: 'data',
                                network: network,
                                dataPlanType: this.categorizePlan(asbPlan),
                                size: asbPlan.size,
                                validity: asbPlan.validity,
                                amount: asbPlan.price,
                                sellingPrice: this.calculateSellingPrice(asbPlan.price),
                                vendorPlanId: asbPlan.planCode,
                                status: 'active',
                                isVisible: true,
                                totalSales: existingPlan?.totalSales || 0,
                                successRate: existingPlan?.successRate || 100,
                                metadata: {
                                    asbDataId: asbPlan.id,
                                    description: asbPlan.description,
                                    originalName: asbPlan.name,
                                    lastSynced: new Date().toISOString()
                                }
                            };
                            if (existingPlan) {
                                await this.servicePlanRepo.updateById(existingPlan._id.toString(), {
                                    amount: planData.amount,
                                    metadata: planData.metadata,
                                    lastSynced: new Date()
                                });
                                results.updated++;
                                console.log(`[PlanSync] Updated plan: ${planData.name}`);
                            }
                            else {
                                await this.servicePlanRepo.create(planData);
                                results.synced++;
                                console.log(`[PlanSync] Created plan: ${planData.name}`);
                            }
                        }
                        catch (planError) {
                            const errorMsg = `Failed to sync plan ${asbPlan.planCode} for ${network}: ${planError}`;
                            console.error(`[PlanSync] ${errorMsg}`);
                            results.errors.push(errorMsg);
                        }
                    }
                }
                catch (networkError) {
                    const errorMsg = `Failed to sync plans for ${network}: ${networkError}`;
                    console.error(`[PlanSync] ${errorMsg}`);
                    results.errors.push(errorMsg);
                }
            }
            console.log(`[PlanSync] Sync completed: ${results.synced} created, ${results.updated} updated, ${results.errors.length} errors`);
            return results;
        }
        catch (error) {
            console.error('[PlanSync] Sync failed:', error);
            results.errors.push(`Sync failed: ${error}`);
            return results;
        }
    }
    formatPlanName(plan) {
        const size = plan.size || plan.name;
        const validity = plan.validity || '30 days';
        const price = plan.price || 0;
        return `${size} - ${validity} N${price}`;
    }
    categorizePlan(plan) {
        const description = (plan.description || '').toLowerCase();
        if (description.includes('sme')) {
            return 'sme';
        }
        else if (description.includes('corporate') || description.includes('gifting')) {
            return 'corporate';
        }
        else if (description.includes('awoof')) {
            return 'awoof';
        }
        else if (description.includes('gifting')) {
            return 'gifting';
        }
        else {
            return 'sme';
        }
    }
    calculateSellingPrice(buyingPrice) {
        const markupPercentage = 5;
        return Math.round(buyingPrice * (1 + markupPercentage / 100));
    }
    async getPlansFromDatabase(network, planType) {
        try {
            console.log(`[PlanSync] Getting plans from database for network: ${network}, planType: ${planType}`);
            const asbDataProvider = await this.serviceProviderRepo.findBySlug('asbdata');
            if (!asbDataProvider) {
                console.log('[PlanSync] ASBData provider not found');
                throw new Error('ASBData provider not found');
            }
            console.log(`[PlanSync] Found ASBData provider: ${asbDataProvider._id}`);
            const filters = {
                providerId: asbDataProvider._id.toString(),
                type: 'data',
                network: network,
                status: 'active',
                isVisible: true
            };
            if (planType) {
                filters.dataPlanType = planType;
            }
            console.log(`[PlanSync] Query filters:`, filters);
            const plans = await this.servicePlanRepo.find(filters);
            console.log(`[PlanSync] Found ${plans.length} plans in database`);
            return plans.map(plan => ({
                id: plan.planId,
                name: plan.name,
                network: plan.network,
                size: plan.size,
                validity: plan.validity,
                price: plan.sellingPrice,
                planCode: plan.planId,
                description: plan.dataPlanType,
                amount: plan.sellingPrice,
                dataPlanType: plan.dataPlanType,
                metadata: plan.metadata
            }));
        }
        catch (error) {
            console.error('[PlanSync] Failed to get plans from database:', error);
            throw error;
        }
    }
    async getPlanTypesFromDatabase() {
        try {
            const asbDataProvider = await this.serviceProviderRepo.findBySlug('asbdata');
            if (!asbDataProvider) {
                throw new Error('ASBData provider not found');
            }
            const plans = await this.servicePlanRepo.find({
                providerId: asbDataProvider._id.toString(),
                type: 'data',
                status: 'active',
                isVisible: true
            });
            const planTypes = [...new Set(plans.map(plan => plan.dataPlanType).filter(Boolean))];
            return planTypes.map((type) => ({
                value: type,
                label: type.toUpperCase(),
                description: `${type} data plans`
            }));
        }
        catch (error) {
            console.error('[PlanSync] Failed to get plan types from database:', error);
            throw error;
        }
    }
}
exports.PlanSyncService = PlanSyncService;
