"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.configurationService = exports.ConfigurationService = void 0;
const service_configuration_repo_1 = require("../database/repositories/service-configuration.repo");
class ConfigurationService {
    static instance;
    cache = new Map();
    CACHE_TTL = 5 * 60 * 1000;
    constructor() { }
    static getInstance() {
        if (!ConfigurationService.instance) {
            ConfigurationService.instance = new ConfigurationService();
        }
        return ConfigurationService.instance;
    }
    async getConfigValue(key, scope = 'global', opts) {
        const cacheKey = this.buildCacheKey(key, scope, opts);
        if (this.cache.has(cacheKey)) {
            const cached = this.cache.get(cacheKey);
            if (Date.now() < cached.expiry) {
                return cached.value;
            }
            this.cache.delete(cacheKey);
        }
        const config = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(key, scope, opts);
        const value = config?.value || config?.defaultValue || null;
        this.cache.set(cacheKey, {
            value,
            expiry: Date.now() + this.CACHE_TTL
        });
        return value;
    }
    async getConfigValues(keys, scope = 'global', opts) {
        const results = {};
        const promises = keys.map(async (key) => {
            const value = await this.getConfigValue(key, scope, opts);
            return { key, value };
        });
        const configs = await Promise.all(promises);
        configs.forEach(({ key, value }) => {
            results[key] = value;
        });
        return results;
    }
    async getConfigsByCategory(category) {
        const configs = await service_configuration_repo_1.serviceConfigurationRepository.listByCategory(category);
        const results = {};
        configs.forEach(config => {
            results[config.key] = config.value || config.defaultValue;
        });
        return results;
    }
    async setConfigValue(key, value, scope = 'global', opts) {
        const configData = {
            key,
            scope,
            value,
            type: opts?.type || 'string',
            name: opts?.name || key,
            description: opts?.description,
            category: opts?.category,
            serviceType: opts?.serviceType,
            providerId: opts?.providerId,
            planId: opts?.planId,
            isPublic: false,
            isEditable: true,
            isDeleted: false,
        };
        const result = await service_configuration_repo_1.serviceConfigurationRepository.upsert(configData);
        const cacheKey = this.buildCacheKey(key, scope, opts);
        this.cache.delete(cacheKey);
        return result;
    }
    clearCache(key, scope, opts) {
        if (key) {
            const cacheKey = this.buildCacheKey(key, scope, opts);
            this.cache.delete(cacheKey);
        }
        else {
            this.cache.clear();
        }
    }
    buildCacheKey(key, scope, opts) {
        let cacheKey = `${key}:${scope}`;
        if (opts?.serviceType)
            cacheKey += `:${opts.serviceType}`;
        if (opts?.providerId)
            cacheKey += `:${opts.providerId}`;
        if (opts?.planId)
            cacheKey += `:${opts.planId}`;
        return cacheKey;
    }
    async getServiceConfigs(serviceType) {
        const configs = await service_configuration_repo_1.serviceConfigurationRepository.find({
            serviceType,
            isDeleted: { $ne: true }
        });
        const results = {};
        configs.forEach(config => {
            results[config.key] = config.value || config.defaultValue;
        });
        return results;
    }
    async hasConfig(key, scope = 'global', opts) {
        const config = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(key, scope, opts);
        return config !== null;
    }
}
exports.ConfigurationService = ConfigurationService;
exports.configurationService = ConfigurationService.getInstance();
