"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = __importDefault(require("express"));
const compression_1 = __importDefault(require("compression"));
const morgan_1 = __importDefault(require("morgan"));
const mongoose_1 = __importDefault(require("mongoose"));
const env_1 = require("./config/env");
const index_1 = __importDefault(require("./routes/index"));
const docs_routes_1 = __importDefault(require("./routes/docs.routes"));
const schedulers_1 = require("./jobs/schedulers");
const errors_1 = require("./core/errors");
const request_lifecycle_1 = require("./core/middleware/request-lifecycle");
const webhook_raw_body_1 = require("./core/middleware/webhook-raw-body");
const rate_limit_1 = require("./config/rate-limit");
const redis_1 = require("./config/redis");
const security_config_1 = require("./core/security/security-config");
const secrets_manager_1 = require("./core/security/secrets-manager");
const backup_manager_1 = require("./core/disaster-recovery/backup-manager");
const app = (0, express_1.default)();
const PORT = process.env['PORT'] || 3000;
const NODE_ENV = process.env['NODE_ENV'] || 'development';
if (NODE_ENV === 'production') {
    app.set('trust proxy', 1);
}
else {
    app.set('trust proxy', false);
}
app.use(errors_1.requestIdMiddleware);
app.use(request_lifecycle_1.initializeContext);
app.use(request_lifecycle_1.logRequest);
app.use(request_lifecycle_1.logResponse);
const securityMiddleware = (0, security_config_1.createSecurityMiddleware)();
app.use(securityMiddleware.helmet);
app.use(securityMiddleware.cors);
app.use(securityMiddleware.hpp);
app.use(request_lifecycle_1.securityHeaders);
app.use(request_lifecycle_1.sanitizeRequest);
app.use((0, request_lifecycle_1.sizeLimit)('10mb'));
app.use((0, request_lifecycle_1.timeout)(30000));
app.use(webhook_raw_body_1.webhookRawBodyMiddleware);
app.use(express_1.default.json({ limit: '10mb' }));
app.use(express_1.default.urlencoded({ extended: true, limit: '10mb' }));
app.use((0, compression_1.default)());
app.use(rate_limit_1.defaultRateLimit);
app.use((0, morgan_1.default)('combined'));
app.get('/health', (_req, res) => {
    res.status(200).json({
        status: 'OK',
        timestamp: new Date().toISOString(),
        uptime: process.uptime(),
        environment: NODE_ENV,
    });
});
app.get("/", (req, res) => {
    res.json({ status: "ok", service: "what are you doing here?" });
});
app.use('/api', index_1.default);
app.use('/', docs_routes_1.default);
app.use('*', (req, res) => {
    res.status(404).json({
        error: 'Route not found',
        path: req.originalUrl,
    });
});
app.use((0, errors_1.errorHandler)());
const startServer = async () => {
    try {
        try {
            if (env_1.env.REDIS_OPTIONAL && (env_1.env.NODE_ENV === 'development' || env_1.env.NODE_ENV === 'test')) {
                if (!env_1.env.REDIS_URL) {
                    console.log('⚠️ Redis URL not provided. Using in-memory fallback because REDIS_OPTIONAL is true.');
                }
                await redis_1.redisManager.getClient('main');
                if (redis_1.redisManager.isUsingFallback()) {
                    console.log('✅ Using in-memory fallback for Redis');
                }
                else {
                    console.log('✅ Connected to Redis');
                }
            }
            else if (env_1.env.REDIS_URL) {
                await redis_1.redisManager.getClient('main');
                console.log('✅ Connected to Redis');
            }
            else {
                throw new Error('Redis URL is required but not provided. Set REDIS_URL or REDIS_OPTIONAL=true');
            }
        }
        catch (redisError) {
            if (env_1.env.REDIS_OPTIONAL && env_1.env.NODE_ENV !== 'production') {
                console.warn('⚠️ Redis unavailable. Continuing without Redis because REDIS_OPTIONAL is true.');
            }
            else {
                throw redisError;
            }
        }
        await mongoose_1.default.connect((0, env_1.getMongoUri)());
        console.log('✅ Connected to MongoDB');
        await secrets_manager_1.secretsManager.initializeDefaultSecrets();
        console.log('✅ Secrets management initialized');
        secrets_manager_1.keyRotationScheduler.startRotationSchedule(24);
        console.log('✅ Key rotation scheduler started');
        app.listen(PORT, () => {
            console.log(`🚀 Server is running on port ${PORT}`);
            console.log(`🌍 Environment: ${NODE_ENV}`);
            console.log(`📊 Health check available at http://localhost:${PORT}/health`);
            console.log(`🔒 Rate limiting enabled`);
            if (env_1.env.REDIS_OPTIONAL && env_1.env.NODE_ENV !== 'production') {
                console.log(`💾 Caching layer: disabled (Redis optional)`);
            }
            else {
                console.log(`💾 Caching layer enabled`);
            }
            console.log(`📝 Request lifecycle tracking enabled`);
            console.log(`🛡️ Enhanced security features enabled`);
            console.log(`📋 NDPR/CBN compliance enabled`);
            console.log(`💾 Backup and DR system enabled`);
            if (NODE_ENV !== 'test') {
                (0, schedulers_1.startAllSchedulers)();
            }
        });
    }
    catch (error) {
        console.error('❌ Failed to start server:', error);
        process.exit(1);
    }
};
const gracefulShutdown = async (signal) => {
    console.log(`\n🛑 Received ${signal}. Starting graceful shutdown...`);
    try {
        secrets_manager_1.keyRotationScheduler.stopRotationSchedule();
        console.log('✅ Schedulers stopped');
        try {
            await backup_manager_1.backupManager.createBackup('db-daily');
            console.log('✅ Emergency backup created');
        }
        catch (error) {
            console.warn('⚠️ Emergency backup failed:', error);
        }
        await redis_1.redisManager.closeAll();
        console.log('✅ Redis connections closed');
        await mongoose_1.default.connection.close();
        console.log('✅ MongoDB connection closed');
        console.log('✅ Graceful shutdown completed');
        process.exit(0);
    }
    catch (error) {
        console.error('❌ Error during graceful shutdown:', error);
        process.exit(1);
    }
};
process.on('SIGTERM', () => gracefulShutdown('SIGTERM'));
process.on('SIGINT', () => gracefulShutdown('SIGINT'));
console.log('[Aspfiy] Using adapter:', env_1.env.ASPFIY_FORCE_STUB || !env_1.env.ASPFIY_SECRET_KEY ? 'STUB' : 'REAL');
console.log('[Redis] REDIS_OPTIONAL:', env_1.env.REDIS_OPTIONAL);
startServer();
exports.default = app;
