"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const errors_1 = require("../core/errors");
const verification_service_1 = require("../modules/verification/verification.service");
const uuid_1 = require("uuid");
const router = (0, express_1.Router)();
router.post('/nin/by-number', (0, errors_1.asyncHandler)(async (_req, res) => {
    const idempotencyKey = `example_nin_${(0, uuid_1.v4)()}`;
    const result = await verification_service_1.verificationService.verifyNin({
        nin: '12345678901',
    }, idempotencyKey);
    res.json({
        success: true,
        message: 'NIN verification by number example',
        data: result,
        idempotencyKey,
    });
}));
router.post('/nin/by-name', (0, errors_1.asyncHandler)(async (_req, res) => {
    const idempotencyKey = `example_nin_name_${(0, uuid_1.v4)()}`;
    const result = await verification_service_1.verificationService.verifyNin({
        first_name: 'John',
        last_name: 'Doe',
        gender: 'male',
        dob: '15-06-1990',
    }, idempotencyKey);
    res.json({
        success: true,
        message: 'NIN verification by name example',
        data: result,
        idempotencyKey,
    });
}));
router.post('/nin/by-phone', (0, errors_1.asyncHandler)(async (_req, res) => {
    const idempotencyKey = `example_nin_phone_${(0, uuid_1.v4)()}`;
    const result = await verification_service_1.verificationService.verifyNin({
        phone: '08012345678',
    }, idempotencyKey);
    res.json({
        success: true,
        message: 'NIN verification by phone example',
        data: result,
        idempotencyKey,
    });
}));
router.post('/bvn/verify', (0, errors_1.asyncHandler)(async (_req, res) => {
    const idempotencyKey = `example_bvn_${(0, uuid_1.v4)()}`;
    const result = await verification_service_1.verificationService.verifyBvn({
        BVN: '12345678901',
    }, idempotencyKey);
    res.json({
        success: true,
        message: 'BVN verification example',
        data: result,
        idempotencyKey,
    });
}));
router.post('/ipe/clearance', (0, errors_1.asyncHandler)(async (_req, res) => {
    const idempotencyKey = `example_ipe_${(0, uuid_1.v4)()}`;
    const result = await verification_service_1.verificationService.createIpeClearance({
        type: 'Normal IPE Clearance',
        tracking_ids: ['123456789012345'],
    }, idempotencyKey);
    res.json({
        success: true,
        message: 'IPE clearance example',
        data: result,
        idempotencyKey,
    });
}));
router.post('/personalization', (0, errors_1.asyncHandler)(async (_req, res) => {
    const idempotencyKey = `example_personalization_${(0, uuid_1.v4)()}`;
    const result = await verification_service_1.verificationService.createPersonalization({
        type: 'Instant Personalization',
        tracking_id: '123456789012345',
    }, idempotencyKey);
    res.json({
        success: true,
        message: 'Personalization example',
        data: result,
        idempotencyKey,
    });
}));
router.post('/sandbox/enable', (0, errors_1.asyncHandler)(async (_req, res) => {
    verification_service_1.verificationService.enableSandboxMode();
    res.json({
        success: true,
        message: 'Sandbox mode enabled for testing',
        sandboxMode: true,
    });
}));
router.post('/sandbox/disable', (0, errors_1.asyncHandler)(async (_req, res) => {
    verification_service_1.verificationService.disableSandboxMode();
    res.json({
        success: true,
        message: 'Sandbox mode disabled',
        sandboxMode: false,
    });
}));
router.put('/sandbox/config', (0, errors_1.asyncHandler)(async (_req, res) => {
    const config = {
        mockDelay: 2000,
        simulateErrors: true,
        errorRate: 0.2,
    };
    verification_service_1.verificationService.updateSandboxConfig(config);
    res.json({
        success: true,
        message: 'Sandbox configuration updated',
        config: verification_service_1.verificationService.getSandboxConfig(),
    });
}));
router.get('/providers/status', (0, errors_1.asyncHandler)(async (_req, res) => {
    const status = await verification_service_1.verificationService.getProviderStatus();
    res.json({
        success: true,
        message: 'Provider status retrieved',
        data: status,
        currentProvider: verification_service_1.verificationService.isSandboxMode() ? 'mock' : 'unify',
    });
}));
router.get('/transactions/:type', (0, errors_1.asyncHandler)(async (req, res) => {
    const { type } = req.params;
    const limit = parseInt(req.query['limit'], 10) || 10;
    const transactions = await verification_service_1.verificationService.getTransactionsByType(type, limit);
    res.json({
        success: true,
        message: `Transactions for type: ${type}`,
        data: transactions,
        count: transactions.length,
    });
}));
router.post('/test-idempotency', (0, errors_1.asyncHandler)(async (_req, res) => {
    const idempotencyKey = `test_idempotency_${(0, uuid_1.v4)()}`;
    const result1 = await verification_service_1.verificationService.verifyNin({
        nin: '12345678901',
    }, idempotencyKey);
    const result2 = await verification_service_1.verificationService.verifyNin({
        nin: '12345678901',
    }, idempotencyKey);
    res.json({
        success: true,
        message: 'Idempotency test completed',
        data: {
            firstRequest: result1,
            secondRequest: result2,
            isIdempotent: result1.idempotencyKey === result2.idempotencyKey,
            idempotencyKey,
        },
    });
}));
router.get('/health', (0, errors_1.asyncHandler)(async (_req, res) => {
    const isHealthy = await verification_service_1.verificationService.healthCheck();
    const providerStatus = await verification_service_1.verificationService.getProviderStatus();
    res.json({
        success: true,
        message: 'Verification service health check',
        data: {
            healthy: isHealthy,
            providerStatus,
            sandboxMode: verification_service_1.verificationService.isSandboxMode(),
            timestamp: new Date().toISOString(),
        },
    });
}));
router.post('/simulate-webhook', (0, errors_1.asyncHandler)(async (req, res) => {
    const idempotencyKey = `webhook_${(0, uuid_1.v4)()}`;
    const timestamp = Date.now().toString();
    const eventType = req.body.eventType || 'nin.verification.completed';
    const webhookPayload = {
        event: eventType,
        data: {
            nin: '12345678901',
            firstname: 'JOHN',
            surname: 'DOE',
            trackingId: 'S' + Math.random().toString(36).substring(2, 15),
            status: 'completed',
        },
        timestamp: parseInt(timestamp),
        idempotencyKey,
    };
    res.json({
        success: true,
        message: 'Webhook simulation payload',
        data: {
            webhookPayload,
            headers: {
                'x-webhook-signature': 'mock_signature',
                'x-webhook-timestamp': timestamp,
                'x-webhook-event': eventType,
                'x-idempotency-key': idempotencyKey,
            },
        },
    });
}));
exports.default = router;
