"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const errors_1 = require("../core/errors");
const security_config_1 = require("../core/security/security-config");
const secrets_manager_1 = require("../core/security/secrets-manager");
const compliance_manager_1 = require("../core/compliance/compliance-manager");
const backup_manager_1 = require("../core/disaster-recovery/backup-manager");
const auth_1 = require("../core/middleware/auth");
const router = (0, express_1.Router)();
router.post('/validate-password', (0, errors_1.asyncHandler)(async (req, res) => {
    const { password } = req.body;
    const validation = security_config_1.securityUtils.validatePasswordStrength(password);
    res.json({
        success: true,
        data: {
            isValid: validation.isValid,
            errors: validation.errors,
            strength: validation.isValid ? 'strong' : 'weak',
        },
    });
}));
router.post('/validate-api-key', (0, errors_1.asyncHandler)(async (req, res) => {
    const { apiKey } = req.body;
    const isValid = security_config_1.securityUtils.validateApiKey(apiKey);
    res.json({
        success: true,
        data: {
            isValid,
            format: '32-64 alphanumeric characters',
        },
    });
}));
router.post('/secrets/create', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (req, res) => {
    const { name, value, type, rotationPeriod } = req.body;
    const secret = await (0, secrets_manager_1.createSecret)(name, value, type, 'development', rotationPeriod);
    res.json({
        success: true,
        data: {
            id: secret.id,
            name: secret.name,
            type: secret.type,
            version: secret.version,
            createdAt: secret.createdAt,
        },
    });
}));
router.post('/secrets/rotate/:name', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (req, res) => {
    const { name } = req.params;
    const rotatedSecret = await (0, secrets_manager_1.rotateSecret)(name);
    res.json({
        success: true,
        data: {
            name: rotatedSecret.name,
            version: rotatedSecret.version,
            lastRotated: rotatedSecret.lastRotated,
        },
    });
}));
router.get('/secrets/list', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (_req, res) => {
    const secrets = await secrets_manager_1.secretsManager.listSecrets();
    res.json({
        success: true,
        data: secrets.map(secret => ({
            id: secret.id,
            name: secret.name,
            type: secret.type,
            version: secret.version,
            lastRotated: secret.lastRotated,
            isActive: secret.isActive,
        })),
    });
}));
router.post('/compliance/detect-pii', (0, errors_1.asyncHandler)(async (req, res) => {
    const { data } = req.body;
    const processedData = (0, compliance_manager_1.detectPII)(data);
    res.json({
        success: true,
        data: {
            original: data,
            processed: processedData,
            piiDetected: Object.keys(data).filter(key => ['email', 'phone', 'bvn', 'nin', 'address', 'bank_account'].includes(key)),
        },
    });
}));
router.post('/compliance/scan-dlp', (0, errors_1.asyncHandler)(async (req, res) => {
    const { content } = req.body;
    const dlpRules = (0, compliance_manager_1.scanDLP)(content);
    res.json({
        success: true,
        data: {
            contentLength: content.length,
            rulesTriggered: dlpRules.map(rule => ({
                name: rule.name,
                severity: rule.severity,
                action: rule.action,
            })),
            totalRules: dlpRules.length,
        },
    });
}));
router.post('/compliance/consent', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (req, res) => {
    const { userId } = req.user;
    const { purpose, consentType, dataUsage, expiresAt } = req.body;
    const consent = await (0, compliance_manager_1.recordConsent)(userId, purpose, consentType, dataUsage, expiresAt ? new Date(expiresAt) : undefined);
    res.json({
        success: true,
        data: {
            id: consent.id,
            purpose: consent.purpose,
            consentType: consent.consentType,
            grantedAt: consent.grantedAt,
            expiresAt: consent.expiresAt,
        },
    });
}));
router.get('/compliance/consent/:purpose', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (req, res) => {
    const { userId } = req.user;
    const { purpose } = req.params;
    const hasConsent = await (0, compliance_manager_1.checkConsent)(userId, purpose);
    res.json({
        success: true,
        data: {
            purpose,
            hasConsent,
            userId,
        },
    });
}));
router.get('/compliance/report', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (_req, res) => {
    const report = await (0, compliance_manager_1.generateComplianceReport)();
    res.json({
        success: true,
        data: report,
    });
}));
router.post('/backup/create/:configId', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (req, res) => {
    const { configId } = req.params;
    const backup = await (0, backup_manager_1.createBackup)(configId);
    res.json({
        success: true,
        data: {
            id: backup.id,
            configId: backup.configId,
            status: backup.status,
            size: backup.size,
            duration: backup.duration,
            timestamp: backup.timestamp,
        },
    });
}));
router.get('/backup/list', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (req, res) => {
    const { configId } = req.query;
    const backups = await (0, backup_manager_1.listBackups)(configId);
    res.json({
        success: true,
        data: backups.map(backup => ({
            id: backup.id,
            configId: backup.configId,
            status: backup.status,
            size: backup.size,
            timestamp: backup.timestamp,
            duration: backup.duration,
        })),
    });
}));
router.post('/dr/execute/:runbookId', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (req, res) => {
    const { runbookId } = req.params;
    await (0, backup_manager_1.executeDRRunbook)(runbookId);
    res.json({
        success: true,
        data: {
            runbookId,
            status: 'executed',
            message: 'DR runbook executed successfully',
        },
    });
}));
router.get('/dr/runbooks', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (_req, res) => {
    const runbooks = backup_manager_1.backupManager.getDRRunbooks();
    res.json({
        success: true,
        data: runbooks.map(runbook => ({
            id: runbook.id,
            name: runbook.name,
            description: runbook.description,
            severity: runbook.severity,
            estimatedTime: runbook.estimatedTime,
            stepsCount: runbook.steps.length,
        })),
    });
}));
router.post('/security/track-event', (0, errors_1.asyncHandler)(async (req, res) => {
    const { event, details } = req.body;
    security_config_1.securityMonitoring.trackSecurityEvent(event, details);
    res.json({
        success: true,
        data: {
            event,
            tracked: true,
            timestamp: new Date().toISOString(),
        },
    });
}));
router.post('/security/track-failed-auth', (0, errors_1.asyncHandler)(async (req, res) => {
    const { ip, userAgent, reason } = req.body;
    security_config_1.securityMonitoring.trackFailedAuth(ip, userAgent, reason);
    res.json({
        success: true,
        data: {
            tracked: true,
            ip,
            reason,
            timestamp: new Date().toISOString(),
        },
    });
}));
router.get('/compliance/retention/:dataType', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (req, res) => {
    const { dataType } = req.params;
    const { createdAt } = req.query;
    const compliance = await compliance_manager_1.complianceManager.checkRetentionCompliance(dataType, new Date(createdAt));
    res.json({
        success: true,
        data: {
            dataType,
            compliant: compliance.compliant,
            daysUntilDisposal: compliance.daysUntilDisposal,
            disposalMethod: compliance.disposalMethod,
            createdAt: createdAt,
        },
    });
}));
router.post('/compliance/dispose/:dataType', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (req, res) => {
    const { dataType } = req.params;
    const { data } = req.body;
    await compliance_manager_1.complianceManager.disposeData(dataType, data);
    res.json({
        success: true,
        data: {
            dataType,
            disposed: true,
            timestamp: new Date().toISOString(),
        },
    });
}));
exports.default = router;
