"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const swagger_ui_express_1 = __importDefault(require("swagger-ui-express"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const yaml_1 = __importDefault(require("yaml"));
const router = (0, express_1.Router)();
function readFileText(filePath) {
    const absolutePath = path_1.default.isAbsolute(filePath)
        ? filePath
        : path_1.default.join(process.cwd(), filePath);
    return fs_1.default.readFileSync(absolutePath, 'utf8');
}
function parseYaml(content) {
    return yaml_1.default.parse(content);
}
function normalizeYamlFragment(content) {
    const lines = content.split(/\r?\n/);
    const indents = lines
        .filter((l) => l.trim().length > 0 && !l.trimStart().startsWith('#'))
        .map((l) => l.match(/^\s*/)?.[0].length || 0);
    const common = indents.length ? Math.min(...indents) : 0;
    if (common > 0) {
        return lines.map((l) => (l.length >= common ? l.slice(common) : l)).join('\n');
    }
    return content;
}
function mergeArraysUnique(a = [], b = []) {
    const byName = new Map();
    for (const item of [...a, ...b]) {
        if (item && typeof item === 'object') {
            const key = item.name ?? JSON.stringify(item);
            if (!byName.has(key))
                byName.set(key, item);
        }
    }
    return Array.from(byName.values());
}
function deepMergeOpenApi(baseDoc, extraDoc) {
    const result = { ...baseDoc };
    let extraPaths = {};
    if (extraDoc && typeof extraDoc === 'object') {
        if (extraDoc.paths && typeof extraDoc.paths === 'object') {
            extraPaths = extraDoc.paths;
        }
        else {
            extraPaths = Object.fromEntries(Object.entries(extraDoc)
                .filter(([key]) => typeof key === 'string' && key.startsWith('/')));
        }
    }
    result.paths = {
        ...(baseDoc.paths || {}),
        ...(extraPaths || {}),
    };
    result.tags = mergeArraysUnique(baseDoc.tags || [], extraDoc.tags || []);
    if (baseDoc.components || extraDoc.components) {
        result.components = {
            ...(baseDoc.components || {}),
            ...(extraDoc.components || {}),
            schemas: {
                ...((baseDoc.components && baseDoc.components.schemas) || {}),
                ...((extraDoc.components && extraDoc.components.schemas) || {}),
            },
            responses: {
                ...((baseDoc.components && baseDoc.components.responses) || {}),
                ...((extraDoc.components && extraDoc.components.responses) || {}),
            },
            securitySchemes: {
                ...((baseDoc.components && baseDoc.components.securitySchemes) || {}),
                ...((extraDoc.components && extraDoc.components.securitySchemes) || {}),
            },
            parameters: {
                ...((baseDoc.components && baseDoc.components.parameters) || {}),
                ...((extraDoc.components && extraDoc.components.parameters) || {}),
            },
        };
    }
    result.servers = mergeArraysUnique(baseDoc.servers || [], extraDoc.servers || []);
    result.info = baseDoc.info || extraDoc.info;
    result.openapi = baseDoc.openapi || extraDoc.openapi || '3.0.3';
    return result;
}
const mainSpec = parseYaml(readFileText('openapi.yaml'));
let mergedSpec = mainSpec;
const additionalSpecPath = path_1.default.join(process.cwd(), 'backend', 'additional-endpoints.yaml');
if (fs_1.default.existsSync(additionalSpecPath)) {
    const raw = readFileText(additionalSpecPath);
    let extraSpec;
    try {
        extraSpec = parseYaml(raw);
    }
    catch {
        extraSpec = parseYaml(normalizeYamlFragment(raw));
    }
    const hasDirectPaths = extraSpec && Object.keys(extraSpec || {}).some((k) => k.startsWith('/'));
    const hasPathsObject = extraSpec && extraSpec.paths && typeof extraSpec.paths === 'object';
    if (!hasDirectPaths && !hasPathsObject) {
        extraSpec = parseYaml(normalizeYamlFragment(raw));
    }
    mergedSpec = deepMergeOpenApi(mainSpec, extraSpec);
}
router.get('/openapi.json', (_req, res) => {
    res.type('application/json').send(mergedSpec);
});
router.use('/docs', swagger_ui_express_1.default.serve, swagger_ui_express_1.default.setup(mergedSpec));
exports.default = router;
