"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const errors_1 = require("../core/errors");
const cache_1 = require("../core/middleware/cache");
const errors_2 = require("../core/errors");
const auth_1 = require("../core/middleware/auth");
const router = (0, express_1.Router)();
router.get('/profile/:userId', auth_1.requireAuth, (0, cache_1.userCache)('user-profile'), (0, errors_1.asyncHandler)(async (req, res) => {
    const { userId } = req.params;
    const user = await getUserFromDatabase(userId || '');
    if (!user) {
        throw new errors_2.NotFoundError(`User ${userId} not found`);
    }
    res.json({
        success: true,
        data: user,
        cached: true,
    });
}));
router.get('/config', (0, cache_1.staticCache)('app-config'), (0, errors_1.asyncHandler)(async (_req, res) => {
    const config = {
        version: '1.0.0',
        features: ['auth', 'payments', 'notifications'],
        limits: {
            maxFileSize: '10MB',
            maxRequestsPerMinute: 100,
        },
    };
    res.json({
        success: true,
        data: config,
        cached: true,
    });
}));
router.get('/stats', (0, cache_1.mediumCache)('app-stats'), (0, errors_1.asyncHandler)(async (_req, res) => {
    const stats = await calculateAppStats();
    res.json({
        success: true,
        data: stats,
        cached: true,
    });
}));
router.post('/profile/update', auth_1.requireAuth, (0, cache_1.cacheInvalidation)(['user:profile', 'user:stats']), (0, errors_1.asyncHandler)(async (req, res) => {
    const { userId } = req.user;
    const updateData = req.body;
    if (!updateData.name && !updateData.email) {
        throw new errors_2.ValidationError('At least one field must be provided for update');
    }
    const updatedUser = await updateUserInDatabase(userId, updateData);
    res.json({
        success: true,
        data: updatedUser,
        message: 'Profile updated successfully',
    });
}));
router.get('/error-demo/:type', (0, errors_1.asyncHandler)(async (req, res) => {
    const { type } = req.params;
    switch (type) {
        case 'validation':
            throw new errors_2.ValidationError('Invalid input data', { field: 'email', value: 'invalid-email' });
        case 'not-found':
            throw new errors_2.NotFoundError('Resource not found');
        case 'database':
            throw new errors_2.DatabaseError('Database connection failed');
        case 'external':
            throw new Error('External service unavailable');
        default:
            res.json({
                success: true,
                message: 'No error thrown',
            });
    }
}));
router.get('/users/search', (0, errors_1.asyncHandler)(async (req, res) => {
    const { query, page = 1, limit = 10 } = req.query;
    const cacheKey = cache_1.queryCache.generateKey('users', { query, page, limit });
    let results = await cache_1.queryCache.get(cacheKey);
    if (!results) {
        results = await searchUsersFromDatabase(query, Number(page), Number(limit));
        await cache_1.queryCache.set(cacheKey, results, 300);
    }
    res.json({
        success: true,
        data: results,
        cached: !!results,
    });
}));
router.get('/cache/stats', auth_1.requireAuth, (0, errors_1.asyncHandler)(async (_req, res) => {
    const stats = {
        redis: await getRedisStats(),
        cache: await getCacheStats(),
        performance: await getPerformanceStats(),
    };
    res.json({
        success: true,
        data: stats,
    });
}));
async function getUserFromDatabase(userId) {
    await new Promise(resolve => setTimeout(resolve, 100));
    if (userId === 'nonexistent') {
        return null;
    }
    return {
        id: userId,
        name: 'John Doe',
        email: 'john@example.com',
        createdAt: new Date().toISOString(),
    };
}
async function updateUserInDatabase(userId, updateData) {
    await new Promise(resolve => setTimeout(resolve, 200));
    return {
        id: userId,
        ...updateData,
        updatedAt: new Date().toISOString(),
    };
}
async function calculateAppStats() {
    await new Promise(resolve => setTimeout(resolve, 500));
    return {
        totalUsers: 15000,
        activeUsers: 8500,
        totalTransactions: 45000,
        revenue: 125000,
        lastUpdated: new Date().toISOString(),
    };
}
async function searchUsersFromDatabase(_query, page, limit) {
    await new Promise(resolve => setTimeout(resolve, 300));
    return {
        users: [
            { id: '1', name: 'John Doe', email: 'john@example.com' },
            { id: '2', name: 'Jane Smith', email: 'jane@example.com' },
        ],
        pagination: {
            page,
            limit,
            total: 2,
            pages: 1,
        },
    };
}
async function getRedisStats() {
    return {
        connected: true,
        memory: '2.5MB',
        keys: 150,
        hitRate: 85.5,
    };
}
async function getCacheStats() {
    return {
        hits: 1250,
        misses: 150,
        hitRate: 89.3,
    };
}
async function getPerformanceStats() {
    return {
        avgResponseTime: 45.2,
        requestsPerSecond: 125,
        errorRate: 0.5,
    };
}
exports.default = router;
