"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const auth_1 = require("../core/middleware/auth");
const aspfiy_1 = require("../integrations/aspfiy");
const env_1 = require("../config/env");
const router = (0, express_1.Router)();
const requireAdmin = (_req, res, next) => {
    if (_req.user?.role !== 'admin') {
        return res.status(403).json({
            success: false,
            message: 'Admin access required'
        });
    }
    next();
};
router.get('/aspfiy/status', auth_1.requireAuth, requireAdmin, (_req, res) => {
    const currentMode = (0, aspfiy_1.getAspfiyAdapterMode)();
    const useStub = env_1.env.ASPFIY_FORCE_STUB ||
        (currentMode === 'stub') ||
        (!env_1.env.ASPFIY_SECRET_KEY && currentMode !== 'real');
    return res.json({
        success: true,
        data: {
            currentMode,
            activeAdapter: useStub ? 'stub' : 'real',
            environment: {
                ASPFIY_FORCE_STUB: env_1.env.ASPFIY_FORCE_STUB,
                ASPFIY_SECRET_KEY_PRESENT: !!env_1.env.ASPFIY_SECRET_KEY,
                ASPFIY_BASE_URL: env_1.env.ASPFIY_BASE_URL
            }
        }
    });
});
router.post('/aspfiy/toggle', auth_1.requireAuth, requireAdmin, (req, res) => {
    const { mode } = req.body;
    if (!mode || !['stub', 'real'].includes(mode)) {
        return res.status(400).json({
            success: false,
            message: 'Mode must be either "stub" or "real"'
        });
    }
    (0, aspfiy_1.setAspfiyAdapterMode)(mode);
    return res.json({
        success: true,
        message: `Aspfiy adapter switched to ${mode.toUpperCase()} mode`,
        data: {
            currentMode: (0, aspfiy_1.getAspfiyAdapterMode)(),
            activeAdapter: mode
        }
    });
});
router.post('/aspfiy/force-stub', auth_1.requireAuth, requireAdmin, (req, res) => {
    const { force } = req.body;
    if (typeof force !== 'boolean') {
        return res.status(400).json({
            success: false,
            message: 'Force parameter must be a boolean'
        });
    }
    if (force) {
        (0, aspfiy_1.setAspfiyAdapterMode)('stub');
    }
    else {
        (0, aspfiy_1.setAspfiyAdapterMode)('real');
    }
    return res.json({
        success: true,
        message: `Aspfiy adapter ${force ? 'forced to STUB' : 'switched to REAL'} mode`,
        data: {
            currentMode: (0, aspfiy_1.getAspfiyAdapterMode)(),
            activeAdapter: force ? 'stub' : 'real',
            note: 'For permanent change, set ASPFIY_FORCE_STUB in .env and restart server'
        }
    });
});
exports.default = router;
