"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.walletService = exports.WalletService = void 0;
const wallet_1 = require("../../database/repositories/wallet");
const transactions_1 = require("../../database/repositories/transactions");
const mongoose_1 = __importDefault(require("mongoose"));
const configuration_service_1 = require("../../services/configuration.service");
class WalletService {
    async validateTransactionParams(params) {
        if (!params.userId || typeof params.userId !== 'string') {
            throw new Error('Invalid userId provided');
        }
        if (!params.currency || !['NGN', 'USD'].includes(params.currency)) {
            throw new Error('Invalid currency provided');
        }
        if (!params.reference || typeof params.reference !== 'string') {
            throw new Error('Invalid reference provided');
        }
        if (!params.category || typeof params.category !== 'string') {
            throw new Error('Invalid category provided');
        }
        if (params.amount <= 0 || !Number.isFinite(params.amount)) {
            throw new Error('Invalid amount provided');
        }
        const transactionLimits = await configuration_service_1.configurationService.getConfigValue('wallet_transaction_limits', 'global', { serviceType: 'wallet' });
        const limits = transactionLimits || {
            maxSingleTransaction: 1000000,
            maxDailyTransaction: 5000000,
            maxMonthlyTransaction: 10000000
        };
        if (params.amount > limits.maxSingleTransaction) {
            throw new Error(`Transaction amount exceeds maximum limit of ${limits.maxSingleTransaction}`);
        }
    }
    logTransaction(type, params, transactionId, balanceBefore, balanceAfter, success, error) {
        const auditLog = {
            timestamp: new Date().toISOString(),
            type,
            transactionId,
            userId: params.userId,
            currency: params.currency,
            amount: params.amount,
            reference: params.reference,
            category: params.category,
            balanceBefore,
            balanceAfter,
            success,
            error: error || null,
            metadata: params.metadata || {}
        };
        console.log(`[WalletService] ${type.toUpperCase()} AUDIT:`, auditLog);
    }
    async calculateFees(amount, category, metadata) {
        let feeAmount = 0;
        let vatAmount = 0;
        const feeConfig = await configuration_service_1.configurationService.getConfigValue('wallet_fees', 'global', { serviceType: 'wallet' });
        const fees = feeConfig || {
            vtu: { percentage: 0, maxAmount: 100 },
            bill_payment: { percentage: 0, maxAmount: 150 },
            transfer: { percentage: 0.02, maxAmount: 200 },
            withdrawal: { percentage: 0.025, maxAmount: 250 },
            default: { percentage: 0, maxAmount: 0 }
        };
        const categoryFees = fees[category] || fees.default;
        feeAmount = Math.min(amount * categoryFees.percentage, categoryFees.maxAmount);
        const serviceCategory = (metadata?.billType ||
            metadata?.serviceCategory ||
            (category === 'vtu' ? metadata?.vtuType : undefined) ||
            (category === 'bill_payment' ? metadata?.billType : undefined) ||
            category ||
            'general')
            .toString()
            .toLowerCase();
        const vatConfig = await configuration_service_1.configurationService.getConfigValue('vat_rate', 'service', { serviceType: serviceCategory });
        const { rate = 0, enabled = false } = (vatConfig || {});
        vatAmount = enabled ? feeAmount * Number(rate || 0) : 0;
        return {
            feeAmount: Math.round(feeAmount * 100) / 100,
            vatAmount: Math.round(vatAmount * 100) / 100,
            netAmount: amount + feeAmount + vatAmount,
        };
    }
    generateJournalEntryId() {
        return `JE-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
    }
    async ensureIdempotency(reference) {
        const existing = await transactions_1.transactionRepository.findByReference(reference);
        if (existing) {
            console.log('[WalletService] Found existing transaction for reference:', {
                reference,
                transactionId: existing.id,
                status: existing.status,
                amount: existing.amount,
                balanceBefore: existing.balanceBefore,
                balanceAfter: existing.balanceAfter,
            });
        }
        return existing;
    }
    async getOrCreateWallet(userId, currency) {
        console.log('[WalletService] Looking for wallet:', { userId, currency });
        let wallet = await wallet_1.walletRepository.findByUserCurrency(userId, currency);
        if (!wallet) {
            console.log('[WalletService] Wallet not found, creating new wallet...');
            try {
                wallet = await wallet_1.walletRepository.create({
                    userId: new mongoose_1.default.Types.ObjectId(userId),
                    currency,
                    availableBalance: 0,
                    ledgerBalance: 0,
                    status: 'active',
                });
                console.log('[WalletService] New wallet created:', {
                    walletId: wallet.id,
                    userId: wallet.userId,
                    currency: wallet.currency,
                });
            }
            catch (error) {
                console.error('[WalletService] Failed to create wallet:', error);
                throw new Error(`Failed to create wallet: ${error instanceof Error ? error.message : 'Unknown error'}`);
            }
        }
        else {
            console.log('[WalletService] Existing wallet found:', {
                walletId: wallet.id,
                userId: wallet.userId,
                currency: wallet.currency,
                balance: wallet.availableBalance,
            });
        }
        return wallet;
    }
    async debit(params) {
        console.log('[WalletService] Starting debit process:', {
            userId: params.userId,
            amount: params.amount,
            currency: params.currency,
            reference: params.reference,
            category: params.category,
        });
        await this.validateTransactionParams(params);
        try {
            const existingTransaction = await this.ensureIdempotency(params.reference);
            if (existingTransaction) {
                console.log('[WalletService] Found existing debit transaction, returning idempotent result');
                return {
                    transaction: existingTransaction,
                    wallet: await this.getOrCreateWallet(params.userId, params.currency),
                };
            }
            const wallet = await this.getOrCreateWallet(params.userId, params.currency);
            console.log('[WalletService] Wallet found for debit:', {
                walletId: wallet.id,
                currentBalance: wallet.availableBalance,
            });
            const fees = await this.calculateFees(params.amount, params.category, params.metadata);
            const totalDebitAmount = fees.netAmount;
            console.log('[WalletService] Fee calculation:', {
                baseAmount: params.amount,
                feeAmount: fees.feeAmount,
                vatAmount: fees.vatAmount,
                totalDebitAmount,
            });
            if (wallet.availableBalance < totalDebitAmount) {
                throw new Error(`Insufficient balance. Required: ${totalDebitAmount}, Available: ${wallet.availableBalance}`);
            }
            const balanceBefore = wallet.availableBalance;
            const balanceAfter = balanceBefore - totalDebitAmount;
            const journalEntryId = this.generateJournalEntryId();
            console.log('[WalletService] Creating debit transaction...');
            const transaction = await transactions_1.transactionRepository.create({
                userId: new mongoose_1.default.Types.ObjectId(wallet.userId),
                walletId: new mongoose_1.default.Types.ObjectId(wallet.id),
                reference: params.reference,
                type: 'debit',
                category: params.category,
                amount: params.amount,
                currency: params.currency,
                balanceBefore,
                balanceAfter,
                feeAmount: fees.feeAmount,
                vatAmount: fees.vatAmount,
                netAmount: fees.netAmount,
                journalEntryId,
                status: 'processing',
                description: params.description,
                metadata: { ...params.metadata, feeCalculation: fees },
            });
            console.log('[WalletService] Debit transaction created:', {
                transactionId: transaction.id,
                reference: transaction.reference,
                amount: transaction.amount,
                balanceBefore,
                balanceAfter,
                totalDebitAmount,
            });
            if (fees.feeAmount > 0) {
                await transactions_1.transactionRepository.create({
                    userId: new mongoose_1.default.Types.ObjectId(wallet.userId),
                    walletId: new mongoose_1.default.Types.ObjectId(wallet.id),
                    reference: `${params.reference}-fee`,
                    type: 'debit',
                    category: 'fee',
                    amount: fees.feeAmount,
                    currency: params.currency,
                    balanceBefore: balanceBefore - params.amount,
                    balanceAfter: balanceBefore - params.amount - fees.feeAmount,
                    journalEntryId,
                    status: 'successful',
                    description: `Transaction fee for ${params.reference}`,
                    metadata: {
                        originalTransaction: params.reference,
                        feeType: 'transaction',
                    },
                    relatedTransactionId: transaction.id,
                });
            }
            if (fees.vatAmount > 0) {
                await transactions_1.transactionRepository.create({
                    userId: wallet.userId,
                    walletId: wallet.id,
                    reference: `${params.reference}-vat`,
                    type: 'debit',
                    category: 'fee',
                    amount: fees.vatAmount,
                    currency: params.currency,
                    balanceBefore: balanceBefore - params.amount - fees.feeAmount,
                    balanceAfter: balanceBefore - totalDebitAmount,
                    journalEntryId,
                    status: 'successful',
                    description: `VAT on transaction fee for ${params.reference}`,
                    metadata: { originalTransaction: params.reference, feeType: 'vat' },
                    relatedTransactionId: transaction.id,
                });
            }
            console.log('[WalletService] Adjusting wallet balance...');
            const updatedWallet = await wallet_1.walletRepository.adjustBalance(wallet.id, -totalDebitAmount);
            if (!updatedWallet) {
                console.error('[WalletService] Failed to update wallet balance');
                throw new Error('Failed to update wallet');
            }
            console.log('[WalletService] Wallet balance updated:', {
                walletId: updatedWallet.id,
                oldBalance: balanceBefore,
                newBalance: updatedWallet.availableBalance,
                expectedBalance: balanceAfter,
                match: updatedWallet.availableBalance === balanceAfter,
            });
            if (updatedWallet.availableBalance !== balanceAfter) {
                console.error('[WalletService] Balance inconsistency detected:', {
                    expected: balanceAfter,
                    actual: updatedWallet.availableBalance,
                    difference: balanceAfter - updatedWallet.availableBalance,
                });
                throw new Error('Balance inconsistency detected after debit');
            }
            await wallet_1.walletRepository.setLastTransactionAt(wallet.id, new Date());
            await transactions_1.transactionRepository.markStatus(transaction.id, 'successful');
            this.logTransaction('debit', params, transaction.id, balanceBefore, balanceAfter, true);
            console.log('[WalletService] Debit process completed successfully');
            return {
                transaction: {
                    ...transaction.toObject(),
                    status: 'successful',
                },
                wallet: updatedWallet,
            };
        }
        catch (error) {
            const errorMessage = error instanceof Error ? error.message : 'Unknown error';
            this.logTransaction('debit', params, 'failed', 0, 0, false, errorMessage);
            throw error;
        }
    }
    async credit(params) {
        console.log('[WalletService] Starting credit process:', {
            userId: params.userId,
            amount: params.amount,
            currency: params.currency,
            reference: params.reference,
        });
        await this.validateTransactionParams(params);
        try {
            const existingTransaction = await this.ensureIdempotency(params.reference);
            if (existingTransaction) {
                console.log('[WalletService] Found existing transaction, returning idempotent result');
                const wallet = await this.getOrCreateWallet(params.userId, params.currency);
                console.log('[WalletService] Idempotent transaction details:', {
                    transactionId: existingTransaction.id,
                    reference: existingTransaction.reference,
                    amount: existingTransaction.amount,
                    status: existingTransaction.status,
                    walletBalance: wallet.availableBalance,
                    transactionBalanceAfter: existingTransaction.balanceAfter,
                });
                if (existingTransaction.status === 'successful' &&
                    (existingTransaction.balanceBefore === undefined || existingTransaction.balanceAfter === undefined)) {
                    console.warn('[WalletService] Idempotent path: existing successful tx missing balances; leaving unchanged to avoid double-credit', {
                        transactionId: existingTransaction.id,
                    });
                }
                if (existingTransaction.status === 'successful' && existingTransaction.balanceAfter !== undefined && wallet.availableBalance !== existingTransaction.balanceAfter) {
                    console.warn('[WalletService] Idempotent path: wallet balance differs from recorded transaction balance; not auto-correcting', {
                        walletBalance: wallet.availableBalance,
                        txBalanceAfter: existingTransaction.balanceAfter,
                    });
                }
                return { transaction: existingTransaction, wallet };
            }
            console.log('[WalletService] Getting or creating wallet...');
            const wallet = await this.getOrCreateWallet(params.userId, params.currency);
            console.log('[WalletService] Wallet found/created:', {
                walletId: wallet.id,
                userId: wallet.userId,
                currency: wallet.currency,
                currentBalance: wallet.availableBalance,
            });
            const balanceBefore = wallet.availableBalance;
            const balanceAfter = balanceBefore + params.amount;
            const journalEntryId = this.generateJournalEntryId();
            console.log('[WalletService] Creating transaction...');
            const transaction = await transactions_1.transactionRepository.create({
                userId: new mongoose_1.default.Types.ObjectId(wallet.userId),
                walletId: new mongoose_1.default.Types.ObjectId(wallet.id),
                reference: params.reference,
                type: 'credit',
                category: params.category,
                amount: params.amount,
                currency: params.currency,
                balanceBefore,
                balanceAfter,
                journalEntryId,
                status: 'processing',
                description: params.description,
                metadata: params.metadata,
            });
            console.log('[WalletService] Transaction created:', {
                transactionId: transaction.id,
                reference: transaction.reference,
                amount: transaction.amount,
                balanceBefore,
                balanceAfter,
                status: transaction.status,
            });
            console.log('[WalletService] Adjusting wallet balance...');
            const updatedWallet = await wallet_1.walletRepository.adjustBalance(wallet.id, params.amount);
            if (!updatedWallet) {
                console.error('[WalletService] Failed to update wallet balance');
                throw new Error('Failed to update wallet');
            }
            console.log('[WalletService] Wallet balance updated:', {
                walletId: updatedWallet.id,
                oldBalance: balanceBefore,
                newBalance: updatedWallet.availableBalance,
                expectedBalance: balanceAfter,
                match: updatedWallet.availableBalance === balanceAfter,
            });
            if (updatedWallet.availableBalance !== balanceAfter) {
                console.error('[WalletService] Balance inconsistency detected:', {
                    expected: balanceAfter,
                    actual: updatedWallet.availableBalance,
                    difference: balanceAfter - updatedWallet.availableBalance,
                });
                throw new Error('Balance inconsistency detected after credit');
            }
            await wallet_1.walletRepository.setLastTransactionAt(wallet.id, new Date());
            await transactions_1.transactionRepository.markStatus(transaction.id, 'successful');
            this.logTransaction('credit', params, transaction.id, balanceBefore, balanceAfter, true);
            console.log('[WalletService] Credit process completed successfully');
            return {
                transaction: {
                    ...transaction.toObject(),
                    status: 'successful',
                },
                wallet: updatedWallet,
            };
        }
        catch (error) {
            const errorMessage = error instanceof Error ? error.message : 'Unknown error';
            this.logTransaction('credit', params, 'failed', 0, 0, false, errorMessage);
            throw error;
        }
    }
    async createReversal(originalTransactionId, reason) {
        const originalTx = await transactions_1.transactionRepository.findById(originalTransactionId);
        if (!originalTx)
            throw new Error('Original transaction not found');
        const reversalRef = `${originalTx.reference}-reversal`;
        const existingReversal = await transactions_1.transactionRepository.findByReference(reversalRef);
        if (existingReversal)
            return existingReversal;
        const wallet = await wallet_1.walletRepository.findById(originalTx.walletId?.toString() || '');
        if (!wallet)
            throw new Error('Wallet not found');
        const balanceBefore = wallet.availableBalance;
        const journalEntryId = this.generateJournalEntryId();
        const reversal = await transactions_1.transactionRepository.create({
            userId: originalTx.userId,
            walletId: originalTx.walletId,
            reference: reversalRef,
            type: originalTx.type === 'debit' ? 'credit' : 'debit',
            category: 'reversal',
            amount: originalTx.amount,
            currency: originalTx.currency,
            balanceBefore,
            balanceAfter: originalTx.type === 'debit'
                ? balanceBefore + originalTx.amount
                : balanceBefore - originalTx.amount,
            journalEntryId,
            status: 'successful',
            description: `Reversal: ${reason}`,
            metadata: {
                originalTransaction: originalTx.reference,
                reason,
                originalAmount: originalTx.amount,
                originalCategory: originalTx.category,
            },
            relatedTransactionId: originalTx.id,
            isReversal: true,
        });
        const adjustment = originalTx.type === 'debit' ? originalTx.amount : -originalTx.amount;
        await wallet_1.walletRepository.adjustBalance(wallet.id, adjustment);
        await wallet_1.walletRepository.setLastTransactionAt(wallet.id, new Date());
        return reversal;
    }
    async getWallet(userId, currency) {
        return this.getOrCreateWallet(userId, currency);
    }
    async getUserBalances(userId) {
        const currencies = ['NGN', 'USD'];
        const balances = [];
        for (const currency of currencies) {
            const wallet = await this.getOrCreateWallet(userId, currency);
            balances.push({
                currency,
                availableBalance: wallet.availableBalance,
                ledgerBalance: wallet.ledgerBalance,
            });
        }
        return balances;
    }
    async getUserBalanceByCurrency(userId, currency) {
        const wallet = await this.getOrCreateWallet(userId, currency);
        return {
            currency,
            availableBalance: wallet.availableBalance,
            ledgerBalance: wallet.ledgerBalance,
        };
    }
}
exports.WalletService = WalletService;
exports.walletService = new WalletService();
