"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const auth_1 = require("../../core/middleware/auth");
const validate_1 = require("../../core/validate");
const wallet_service_1 = require("./wallet.service");
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const user_model_1 = require("../users/user.model");
const router = (0, express_1.Router)();
router.get('/balance', auth_1.requireAuth, async (req, res) => {
    try {
        const balances = await wallet_service_1.walletService.getUserBalances(req.user.userId);
        res.status(200).json({ success: true, data: balances });
    }
    catch (e) {
        res.status(400).json({ success: false, message: e.message || 'Failed to get balances' });
    }
});
router.get('/balance/:currency', auth_1.requireAuth, async (req, res) => {
    try {
        const currencyParam = (req.params['currency'] || '').toUpperCase();
        if (currencyParam !== 'NGN' && currencyParam !== 'USD') {
            res.status(400).json({ success: false, message: 'Invalid currency. Use NGN or USD' });
            return;
        }
        const balance = await wallet_service_1.walletService.getUserBalanceByCurrency(req.user.userId, currencyParam);
        res.status(200).json({ success: true, data: balance });
    }
    catch (e) {
        res.status(400).json({ success: false, message: e.message || 'Failed to get balance' });
    }
});
router.post('/credit', auth_1.requireAuth, (0, validate_1.validate)(validate_1.walletCreditSchema), async (req, res) => {
    try {
        const { amount, currency, reference, category, description, metadata } = req.body || {};
        if (!amount || !reference) {
            res.status(400).json({ error: 'amount and reference are required' });
            return;
        }
        const result = await wallet_service_1.walletService.credit({ userId: req.user.userId, amount, currency, reference, category, description, metadata });
        res.status(200).json({ transaction: result.transaction, wallet: result.wallet });
    }
    catch (e) {
        res.status(400).json({ error: e.message || 'Failed to credit wallet' });
    }
});
router.post('/debit', auth_1.requireAuth, (0, validate_1.validate)(validate_1.walletDebitSchema), async (req, res) => {
    try {
        const { amount, currency, reference, category, description, metadata, transactionPin } = req.body || {};
        if (!amount || !reference) {
            res.status(400).json({ error: 'amount and reference are required' });
            return;
        }
        const user = await user_model_1.User.findById(req.user.userId).select('+transactionPinHash');
        if (!user) {
            res.status(404).json({ error: 'User not found' });
            return;
        }
        if (user['transactionPinHash']) {
            if (!transactionPin) {
                res.status(400).json({ error: 'Transaction PIN required' });
                return;
            }
            const ok = await bcryptjs_1.default.compare(String(transactionPin), String(user['transactionPinHash']));
            if (!ok) {
                res.status(400).json({ error: 'Invalid transaction PIN' });
                return;
            }
        }
        const result = await wallet_service_1.walletService.debit({ userId: req.user.userId, amount, currency, reference, category, description, metadata });
        res.status(200).json({ transaction: result.transaction, wallet: result.wallet });
    }
    catch (e) {
        res.status(400).json({ error: e.message || 'Failed to debit wallet' });
    }
});
exports.default = router;
