"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.vtuService = exports.VtuService = void 0;
const wallet_service_1 = require("../wallet/wallet.service");
const vtu_1 = require("../../database/repositories/vtu");
const transactions_1 = require("../../database/repositories/transactions");
const registry_1 = require("../../integrations/registry");
class VtuService {
    async callProvider(order) {
        const adapter = (0, registry_1.getVtuAdapter)();
        const payload = {
            amount: order.amount,
            type: order.type,
            network: order.network,
            phone: order.phone,
            reference: String(order.id),
        };
        if (order.planCode !== undefined) {
            payload.planCode = order.planCode;
        }
        return adapter.buyVtu(payload);
    }
    async buyVtu(params) {
        const existingTx = await transactions_1.transactionRepository.findByReference(params.reference);
        if (existingTx) {
            const existingOrder = await vtu_1.vtuRepository.findOne({ transactionId: existingTx.id });
            if (existingOrder)
                return { order: existingOrder };
        }
        const order = await vtu_1.vtuRepository.create({
            userId: params.userId,
            type: params.type,
            network: params.network,
            phone: params.phone,
            amount: params.amount,
            planCode: params.planCode,
            status: 'pending',
            provider: 'others'
        });
        const { transaction } = await wallet_service_1.walletService.debit({
            userId: params.userId,
            currency: params.currency,
            amount: params.amount,
            reference: params.reference,
            category: 'vtu',
            description: `${params.type} purchase on ${params.network} for ${params.phone}`,
            metadata: { orderId: order.id }
        });
        const updatedOrder = await vtu_1.vtuRepository.updateById(order.id, { transactionId: transaction.id, status: 'processing' });
        if (!updatedOrder)
            throw new Error('Failed to update VTU order');
        const provider = await this.callProvider(updatedOrder);
        if (provider.success) {
            await vtu_1.vtuRepository.updateById(updatedOrder.id, {
                status: 'successful',
                providerReference: provider.providerReference,
                metadata: provider.meta,
            });
        }
        else {
            await vtu_1.vtuRepository.updateById(updatedOrder.id, { status: 'failed' });
            const reversalRef = `${params.reference}-reversal`;
            await wallet_service_1.walletService.credit({
                userId: params.userId,
                currency: params.currency,
                amount: params.amount,
                reference: reversalRef,
                category: 'reversal',
                description: `Reversal for failed vtu ${params.reference}`,
                metadata: { originalReference: params.reference, orderId: updatedOrder.id }
            });
        }
        const finalOrder = await vtu_1.vtuRepository.findById(updatedOrder.id);
        if (!finalOrder)
            throw new Error('VTU order not found after update');
        return { order: finalOrder };
    }
}
exports.VtuService = VtuService;
exports.vtuService = new VtuService();
