"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.vtuBillsService = exports.VtuBillsService = void 0;
const wallet_service_1 = require("../wallet/wallet.service");
const vtu_1 = require("../../database/repositories/vtu");
const transactions_1 = require("../../database/repositories/transactions");
const provider_manager_1 = require("../../integrations/vtu-bills/provider-manager");
const vtu_bills_types_1 = require("../../integrations/vtu-bills/vtu-bills.types");
class VtuBillsService {
    static instance;
    constructor() { }
    static getInstance() {
        if (!VtuBillsService.instance) {
            VtuBillsService.instance = new VtuBillsService();
        }
        return VtuBillsService.instance;
    }
    async buyVtu(params) {
        try {
            const existingTx = await transactions_1.transactionRepository.findByReference(params.reference);
            if (existingTx && existingTx.status === 'successful') {
                const existingOrder = await vtu_1.vtuRepository.findOne({ transactionId: existingTx.id });
                if (existingOrder && existingOrder.status === 'successful') {
                    console.log(`[VTUBillsService] Found existing successful transaction: ${params.reference}`);
                    return {
                        order: existingOrder,
                        response: {
                            success: true,
                            providerReference: existingOrder.providerReference || '',
                            transactionId: existingOrder.transactionId?.toString() || '',
                            status: 'successful',
                            message: 'Transaction already completed successfully',
                        }
                    };
                }
            }
            this.validateVtuRequest(params);
            console.log(`[VTUBillsService] Processing VTU request:`, {
                userId: params.userId,
                type: params.type,
                network: params.network,
                phone: params.phone,
                amount: params.amount,
                currency: params.currency,
                reference: params.reference,
                provider: params.provider
            });
            await this.checkWalletBalance(params.userId, params.currency, params.amount);
            const order = await vtu_1.vtuRepository.create({
                userId: params.userId,
                type: params.type,
                network: params.network,
                phone: params.phone,
                amount: params.amount,
                planCode: params.planCode,
                status: 'pending',
                provider: params.provider || 'asbdata',
            });
            const { transaction } = await wallet_service_1.walletService.debit({
                userId: params.userId,
                currency: params.currency,
                amount: params.amount,
                reference: params.reference,
                category: 'vtu',
                description: `${params.type} purchase on ${params.network} for ${params.phone}`,
                metadata: { orderId: order.id, provider: params.provider, vtuType: params.type }
            });
            const updatedOrder = await vtu_1.vtuRepository.updateById(order.id, {
                transactionId: transaction.id,
                status: 'processing'
            });
            if (!updatedOrder)
                throw new Error('Failed to update VTU order');
            let selectionMeta = {};
            try {
                const { productCatalogService } = await Promise.resolve().then(() => __importStar(require('../../services/product-catalog.service')));
                const canonicalCode = params.type === 'data' && params.planCode && params.network
                    ? `${String(params.network).toUpperCase()}_${String(params.planCode).toUpperCase()}`
                    : undefined;
                const catalog = await productCatalogService.getProviderCostsAndGuardrails({
                    canonicalCode,
                    planCode: params.planCode,
                    network: params.network,
                    type: params.type,
                });
                if (catalog?.maxAllowedCost)
                    selectionMeta.maxAllowedCost = catalog.maxAllowedCost;
                if (catalog?.providerCosts)
                    selectionMeta.providerCosts = catalog.providerCosts;
            }
            catch { }
            const providerRequest = {
                amount: params.amount,
                type: params.type,
                network: params.network,
                phone: params.phone,
                planCode: params.planCode || '',
                reference: params.reference,
                metadata: {
                    userId: params.userId,
                    orderId: order.id,
                    transactionId: transaction.id,
                    ...(params.provider ? { providerPriority: [params.provider] } : {}),
                    ...((params?.maxAllowedCost) ? { maxAllowedCost: params.maxAllowedCost } : {}),
                    ...((params?.providerCosts) ? { providerCosts: params.providerCosts } : {}),
                    ...selectionMeta,
                },
            };
            let providerResponse;
            if (params.type === 'airtime') {
                providerResponse = await provider_manager_1.vtuBillsProviderManager.buyAirtime(providerRequest);
            }
            else {
                providerResponse = await provider_manager_1.vtuBillsProviderManager.buyData(providerRequest);
            }
            if (providerResponse.success) {
                console.log(`[VtuBillsService] Transaction successful: ${params.reference}`);
                await vtu_1.vtuRepository.updateById(updatedOrder.id, {
                    status: 'successful',
                    providerReference: providerResponse.providerReference,
                    metadata: providerResponse.meta,
                });
            }
            else {
                console.log(`[VtuBillsService] Transaction failed: ${params.reference}, Status: ${providerResponse.status}`);
                if (providerResponse.status === 'failed') {
                    await vtu_1.vtuRepository.updateById(updatedOrder.id, {
                        status: 'failed',
                        metadata: providerResponse.meta,
                    });
                    await this.processReversal(params, transaction.id, updatedOrder.id);
                }
                else {
                    await vtu_1.vtuRepository.updateById(updatedOrder.id, {
                        status: 'pending',
                        metadata: providerResponse.meta,
                    });
                }
            }
            const finalOrder = await vtu_1.vtuRepository.findById(updatedOrder.id);
            if (!finalOrder)
                throw new Error('VTU order not found after update');
            return { order: finalOrder, response: providerResponse };
        }
        catch (error) {
            console.error('[VtuBillsService] buyVtu failed:', error);
            throw error;
        }
    }
    async getDataPlans(network, planType) {
        try {
            const { ProductRepository } = await Promise.resolve().then(() => __importStar(require('../../database/repositories/product.repo')));
            const repo = new ProductRepository();
            const products = await repo.findActiveByNetworkAndType(network, 'data', planType);
            const filtered = products;
            return filtered.map(p => ({
                id: p.id || p._id?.toString?.() || p.canonicalCode,
                name: p.name,
                network: p.network,
                size: p.size || '',
                validity: p.validity || '',
                price: p.sellingPrice ?? 0,
                planCode: (p.providers?.[0]?.vendorCode) || p.canonicalCode,
                description: p.description || undefined,
            }));
        }
        catch (error) {
            console.error('[VtuBillsService] getDataPlans failed:', error);
            return await provider_manager_1.vtuBillsProviderManager.getDataPlans(network);
        }
    }
    async getNetworks() {
        try {
            return await provider_manager_1.vtuBillsProviderManager.getNetworks();
        }
        catch (error) {
            console.error('[VtuBillsService] getNetworks failed:', error);
            throw error;
        }
    }
    async getDataPlanTypes() {
        try {
            const { ProductRepository } = await Promise.resolve().then(() => __importStar(require('../../database/repositories/product.repo')));
            const repo = new ProductRepository();
            const types = await repo.findDistinctDataPlanTypes();
            return types.map(type => ({
                value: type,
                label: type.charAt(0).toUpperCase() + type.slice(1),
                description: this.getPlanTypeDescription(type),
            }));
        }
        catch (error) {
            console.error('[VtuBillsService] getDataPlanTypes failed:', error);
            return [
                { value: 'sme', label: 'SME', description: 'Small and Medium Enterprise plans' },
                { value: 'sme2', label: 'SME2', description: 'Enhanced SME plans' },
                { value: 'corporate', label: 'Corporate', description: 'Corporate data plans' },
                { value: 'awoof', label: 'Awoof', description: 'Special promotional plans' },
                { value: 'gifting', label: 'Gifting', description: 'Data gifting plans' },
            ];
        }
    }
    getPlanTypeDescription(type) {
        const descriptions = {
            'sme': 'Small and Medium Enterprise plans',
            'sme2': 'Enhanced SME plans with better rates',
            'corporate': 'Corporate data plans for businesses',
            'awoof': 'Special promotional plans with discounts',
            'gifting': 'Data gifting plans for sharing with others',
        };
        return descriptions[type] || 'Data plan';
    }
    async payElectricityBill(params) {
        try {
            const existingTx = await transactions_1.transactionRepository.findByReference(params.reference);
            if (existingTx) {
                return {
                    transaction: existingTx,
                    response: {
                        success: existingTx.status === 'successful',
                        providerReference: existingTx.externalReference || '',
                        transactionId: existingTx.id,
                        status: existingTx.status,
                        message: 'Transaction already processed',
                    }
                };
            }
            this.validateBillRequest(params);
            await this.checkWalletBalance(params.userId, params.currency, params.amount);
            const { transaction } = await wallet_service_1.walletService.debit({
                userId: params.userId,
                currency: params.currency,
                amount: params.amount,
                reference: params.reference,
                category: 'bill_payment',
                description: `Electricity bill payment for ${params.electricityProvider}`,
                metadata: {
                    billType: 'electricity',
                    provider: params.provider,
                    meterNumber: params.meterNumber,
                }
            });
            const providerRequest = {
                amount: params.amount,
                type: 'electricity',
                provider: params.electricityProvider,
                customerId: params.customerId,
                reference: params.reference,
                meterNumber: params.meterNumber,
                meterType: params.meterType,
                customerName: params.customerName || '',
                customerAddress: params.customerAddress || '',
                units: params.units || 0,
                metadata: {
                    userId: params.userId,
                    transactionId: transaction.id,
                },
            };
            const providerResponse = await provider_manager_1.vtuBillsProviderManager.payElectricityBill(providerRequest);
            if (providerResponse.success) {
                await transactions_1.transactionRepository.updateById(transaction.id, {
                    status: 'successful',
                    externalReference: providerResponse.providerReference,
                    metadata: { ...transaction.metadata, providerResponse: providerResponse.meta },
                });
            }
            else {
                await transactions_1.transactionRepository.updateById(transaction.id, {
                    status: 'failed',
                    metadata: { ...transaction.metadata, providerResponse: providerResponse.meta },
                });
                await this.processReversal(params, transaction.id);
            }
            const updatedTransaction = await transactions_1.transactionRepository.findById(transaction.id);
            if (!updatedTransaction)
                throw new Error('Transaction not found after update');
            return { transaction: updatedTransaction, response: providerResponse };
        }
        catch (error) {
            console.error('[VtuBillsService] payElectricityBill failed:', error);
            throw error;
        }
    }
    async payCableBill(params) {
        try {
            const existingTx = await transactions_1.transactionRepository.findByReference(params.reference);
            if (existingTx) {
                return {
                    transaction: existingTx,
                    response: {
                        success: existingTx.status === 'successful',
                        providerReference: existingTx.externalReference || '',
                        transactionId: existingTx.id,
                        status: existingTx.status,
                        message: 'Transaction already processed',
                    }
                };
            }
            this.validateBillRequest(params);
            await this.checkWalletBalance(params.userId, params.currency, params.amount);
            const { transaction } = await wallet_service_1.walletService.debit({
                userId: params.userId,
                currency: params.currency,
                amount: params.amount,
                reference: params.reference,
                category: 'bill_payment',
                description: `Cable bill payment for ${params.cableProvider}`,
                metadata: {
                    billType: 'cable',
                    provider: params.provider,
                    smartCardNumber: params.smartCardNumber,
                }
            });
            const providerRequest = {
                amount: params.amount,
                type: 'cable',
                provider: params.cableProvider,
                customerId: params.customerId,
                reference: params.reference,
                smartCardNumber: params.smartCardNumber,
                customerName: params.customerName || '',
                package: params.package || '',
                months: params.months || 1,
                metadata: {
                    userId: params.userId,
                    transactionId: transaction.id,
                },
            };
            const providerResponse = await provider_manager_1.vtuBillsProviderManager.payCableBill(providerRequest);
            if (providerResponse.success) {
                await transactions_1.transactionRepository.updateById(transaction.id, {
                    status: 'successful',
                    externalReference: providerResponse.providerReference,
                    metadata: { ...transaction.metadata, providerResponse: providerResponse.meta },
                });
            }
            else {
                await transactions_1.transactionRepository.updateById(transaction.id, {
                    status: 'failed',
                    metadata: { ...transaction.metadata, providerResponse: providerResponse.meta },
                });
                await this.processReversal(params, transaction.id);
            }
            const updatedTransaction = await transactions_1.transactionRepository.findById(transaction.id);
            if (!updatedTransaction)
                throw new Error('Transaction not found after update');
            return { transaction: updatedTransaction, response: providerResponse };
        }
        catch (error) {
            console.error('[VtuBillsService] payCableBill failed:', error);
            throw error;
        }
    }
    async validateElectricityMeter(meterNumber, provider, meterType) {
        try {
            return await provider_manager_1.vtuBillsProviderManager.validateElectricityMeter(meterNumber, provider, meterType);
        }
        catch (error) {
            console.error('[VtuBillsService] validateElectricityMeter failed:', error);
            throw error;
        }
    }
    async validateCableIUC(smartCardNumber, provider) {
        try {
            return await provider_manager_1.vtuBillsProviderManager.validateCableIUC(smartCardNumber, provider);
        }
        catch (error) {
            console.error('[VtuBillsService] validateCableIUC failed:', error);
            throw error;
        }
    }
    async buyExamPin(params) {
        try {
            const existingTx = await transactions_1.transactionRepository.findByReference(params.reference);
            if (existingTx) {
                return {
                    transaction: existingTx,
                    response: {
                        success: existingTx.status === 'successful',
                        providerReference: existingTx.externalReference || '',
                        pins: [],
                        quantity: params.quantity,
                        examType: params.examType,
                        message: 'Transaction already processed',
                    }
                };
            }
            this.validateExamPinRequest(params);
            await this.checkWalletBalance(params.userId, params.currency, params.amount);
            const { transaction } = await wallet_service_1.walletService.debit({
                userId: params.userId,
                currency: params.currency,
                amount: params.amount,
                reference: params.reference,
                category: 'bill_payment',
                description: `Exam pin purchase for ${params.examType}`,
                metadata: {
                    examType: params.examType,
                    billType: 'education',
                    quantity: params.quantity,
                    provider: params.provider,
                }
            });
            const providerRequest = {
                examType: params.examType,
                quantity: params.quantity,
                reference: params.reference,
                metadata: {
                    userId: params.userId,
                    transactionId: transaction.id,
                },
            };
            const providerResponse = await provider_manager_1.vtuBillsProviderManager.buyExamPin(providerRequest);
            if (providerResponse.success) {
                await transactions_1.transactionRepository.updateById(transaction.id, {
                    status: 'successful',
                    externalReference: providerResponse.providerReference,
                    metadata: {
                        ...transaction.metadata,
                        providerResponse: providerResponse.meta,
                        pins: providerResponse.pins,
                    },
                });
            }
            else {
                await transactions_1.transactionRepository.updateById(transaction.id, {
                    status: 'failed',
                    metadata: { ...transaction.metadata, providerResponse: providerResponse.meta },
                });
                await this.processReversal(params, transaction.id);
            }
            const updatedTransaction = await transactions_1.transactionRepository.findById(transaction.id);
            if (!updatedTransaction)
                throw new Error('Transaction not found after update');
            return { transaction: updatedTransaction, response: providerResponse };
        }
        catch (error) {
            console.error('[VtuBillsService] buyExamPin failed:', error);
            throw error;
        }
    }
    async generateRechargePins(userId, currency, amount, quantity, reference) {
        try {
            const existingTx = await transactions_1.transactionRepository.findByReference(reference);
            if (existingTx) {
                return {
                    transaction: existingTx,
                    response: {
                        success: existingTx.status === 'successful',
                        providerReference: existingTx.externalReference || '',
                        pins: [],
                        quantity,
                        examType: 'recharge_card',
                        message: 'Transaction already processed',
                    }
                };
            }
            await this.checkWalletBalance(userId, currency, amount);
            const { transaction } = await wallet_service_1.walletService.debit({
                userId,
                currency,
                amount,
                reference,
                category: 'bill_payment',
                description: `Recharge pin generation`,
                metadata: {
                    billType: 'recharge',
                    quantity,
                    amount,
                }
            });
            const providerResponse = await provider_manager_1.vtuBillsProviderManager.generateRechargePins(amount, quantity, reference);
            if (providerResponse.success) {
                await transactions_1.transactionRepository.updateById(transaction.id, {
                    status: 'successful',
                    externalReference: providerResponse.providerReference,
                    metadata: {
                        ...transaction.metadata,
                        providerResponse: providerResponse.meta,
                        pins: providerResponse.pins,
                    },
                });
            }
            else {
                await transactions_1.transactionRepository.updateById(transaction.id, {
                    status: 'failed',
                    metadata: { ...transaction.metadata, providerResponse: providerResponse.meta },
                });
                await this.processReversal({ userId, currency, amount, reference }, transaction.id);
            }
            const updatedTransaction = await transactions_1.transactionRepository.findById(transaction.id);
            if (!updatedTransaction)
                throw new Error('Transaction not found after update');
            return { transaction: updatedTransaction, response: providerResponse };
        }
        catch (error) {
            console.error('[VtuBillsService] generateRechargePins failed:', error);
            throw error;
        }
    }
    async queryTransaction(query) {
        try {
            return await provider_manager_1.vtuBillsProviderManager.queryTransaction(query);
        }
        catch (error) {
            console.error('[VtuBillsService] queryTransaction failed:', error);
            throw error;
        }
    }
    async getTransactionHistory(filters) {
        try {
            return await provider_manager_1.vtuBillsProviderManager.getTransactionHistory(filters);
        }
        catch (error) {
            console.error('[VtuBillsService] getTransactionHistory failed:', error);
            throw error;
        }
    }
    getProviderStatus() {
        return provider_manager_1.vtuBillsProviderManager.getProviderStatus();
    }
    getAvailableProviders() {
        return provider_manager_1.vtuBillsProviderManager.getAvailableProviders();
    }
    setProvider(providerName) {
        provider_manager_1.vtuBillsProviderManager.setProvider(providerName);
    }
    enableSandboxMode() {
        provider_manager_1.vtuBillsProviderManager.enableSandboxMode();
    }
    disableSandboxMode() {
        provider_manager_1.vtuBillsProviderManager.disableSandboxMode();
    }
    isSandboxMode() {
        return provider_manager_1.vtuBillsProviderManager.isSandboxMode();
    }
    validateVtuRequest(params) {
        if (!params.amount || params.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!params.phone || !/^\+?234\d{10}$/.test(params.phone)) {
            throw new vtu_bills_types_1.ValidationError('Invalid phone number format');
        }
        if (!params.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
        if (params.type === 'data' && !params.planCode) {
            throw new vtu_bills_types_1.ValidationError('Plan code is required for data purchase');
        }
    }
    validateBillRequest(params) {
        if (!params.amount || params.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!params.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
        if (!params.customerId) {
            throw new vtu_bills_types_1.ValidationError('Customer ID is required');
        }
    }
    validateExamPinRequest(params) {
        if (!params.amount || params.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!params.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
        if (!params.examType) {
            throw new vtu_bills_types_1.ValidationError('Exam type is required');
        }
        if (!params.quantity || params.quantity <= 0) {
            throw new vtu_bills_types_1.ValidationError('Quantity must be greater than 0');
        }
    }
    async checkWalletBalance(userId, currency, amount) {
        const wallet = await wallet_service_1.walletService.getWallet(userId, currency);
        if (wallet.availableBalance < amount) {
            throw new vtu_bills_types_1.InsufficientBalanceError(amount, wallet.availableBalance);
        }
    }
    async processReversal(params, transactionId, orderId) {
        const reversalRef = `${params.reference}-reversal-${Date.now()}`;
        await wallet_service_1.walletService.credit({
            userId: params.userId,
            currency: params.currency,
            amount: params.amount,
            reference: reversalRef,
            category: 'reversal',
            description: `Reversal for failed transaction ${params.reference}`,
            metadata: {
                originalReference: params.reference,
                originalTransactionId: transactionId,
                orderId,
            }
        });
    }
}
exports.VtuBillsService = VtuBillsService;
exports.vtuBillsService = VtuBillsService.getInstance();
