"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VtuBillsController = void 0;
const errors_1 = require("../../core/errors");
const vtu_bills_service_1 = require("./vtu-bills.service");
const vtu_bills_types_1 = require("../../integrations/vtu-bills/vtu-bills.types");
const service_configuration_repo_1 = require("../../database/repositories/service-configuration.repo");
class VtuBillsController {
    static buyAirtime = (0, errors_1.asyncHandler)(async (req, res) => {
        const { amount, currency, reference, network, phone, provider, } = req.body;
        if (!amount || !currency || !reference || !network || !phone) {
            throw new vtu_bills_types_1.ValidationError('Amount, currency, reference, network, and phone are required');
        }
        const result = await vtu_bills_service_1.vtuBillsService.buyVtu({
            userId: req.user.userId,
            amount: Number(amount),
            currency: currency,
            type: 'airtime',
            network: network,
            phone,
            reference,
            provider,
        });
        res.status(200).json({
            success: true,
            message: 'Airtime purchase initiated successfully',
            data: {
                order: result.order,
                response: result.response,
            },
        });
    });
    static buyData = (0, errors_1.asyncHandler)(async (req, res) => {
        const { amount, currency, reference, network, phone, planCode, provider, } = req.body;
        if (!amount || !currency || !reference || !network || !phone || !planCode) {
            throw new vtu_bills_types_1.ValidationError('Amount, currency, reference, network, phone, and planCode are required');
        }
        const result = await vtu_bills_service_1.vtuBillsService.buyVtu({
            userId: req.user.userId,
            amount: Number(amount),
            currency: currency,
            type: 'data',
            network: network,
            phone,
            planCode,
            reference,
            provider,
        });
        res.status(200).json({
            success: true,
            message: 'Data purchase initiated successfully',
            data: {
                order: result.order,
                response: result.response,
            },
        });
    });
    static getDataPlans = (0, errors_1.asyncHandler)(async (req, res) => {
        const { network } = req.params;
        const { dataPlanType } = req.query;
        console.log(`[VtuBillsController] getDataPlans called with network: ${network}, dataPlanType: ${dataPlanType}`);
        if (!network) {
            throw new vtu_bills_types_1.ValidationError('Network parameter is required');
        }
        let networkName = network;
        if (network === '1') {
            networkName = 'mtn';
        }
        else if (network === '2') {
            networkName = 'glo';
        }
        else if (network === '3') {
            networkName = '9mobile';
        }
        else if (network === '4') {
            networkName = 'airtel';
        }
        console.log(`[VtuBillsController] Converted network ${network} to ${networkName}`);
        const plans = await vtu_bills_service_1.vtuBillsService.getDataPlans(networkName, dataPlanType);
        console.log(`[VtuBillsController] Found ${plans.length} plans for ${networkName}`);
        res.status(200).json({
            success: true,
            message: 'Data plans retrieved successfully',
            data: plans,
        });
    });
    static getNetworks = (0, errors_1.asyncHandler)(async (_req, res) => {
        const networks = await vtu_bills_service_1.vtuBillsService.getNetworks();
        res.status(200).json({
            success: true,
            message: 'Networks retrieved successfully',
            data: networks,
        });
    });
    static getDataPlanTypes = (0, errors_1.asyncHandler)(async (_req, res) => {
        const planTypes = await vtu_bills_service_1.vtuBillsService.getDataPlanTypes();
        res.status(200).json({
            success: true,
            message: 'Data plan types retrieved successfully',
            data: planTypes,
        });
    });
    static payElectricityBill = (0, errors_1.asyncHandler)(async (req, res) => {
        const { amount, currency, reference, customerId, meterNumber, meterType, electricityProvider, customerName, customerAddress, units, provider, } = req.body;
        if (!amount || !currency || !reference || !customerId || !meterNumber || !meterType || !electricityProvider) {
            throw new vtu_bills_types_1.ValidationError('Amount, currency, reference, customerId, meterNumber, meterType, and electricityProvider are required');
        }
        try {
            const cfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey('bills_electricity_enabled', 'service', { serviceType: 'bills' });
            if (cfg && typeof cfg.value === 'boolean' && cfg.value === false) {
                res.status(403).json({ success: false, message: 'Electricity payments are disabled' });
                return;
            }
        }
        catch { }
        const result = await vtu_bills_service_1.vtuBillsService.payElectricityBill({
            userId: req.user.userId,
            amount: Number(amount),
            currency: currency,
            type: 'electricity',
            provider: provider || 'asbdata',
            customerId,
            reference,
            meterNumber,
            meterType: meterType,
            electricityProvider: electricityProvider,
            customerName,
            customerAddress,
            units: units ? Number(units) : 0,
        });
        res.status(200).json({
            success: true,
            message: 'Electricity bill payment initiated successfully',
            data: {
                transaction: result.transaction,
                response: result.response,
            },
        });
    });
    static payCableBill = (0, errors_1.asyncHandler)(async (req, res) => {
        const { amount, currency, reference, customerId, smartCardNumber, cableProvider, customerName, package: packageName, months, provider, } = req.body;
        if (!amount || !currency || !reference || !customerId || !smartCardNumber || !cableProvider) {
            throw new vtu_bills_types_1.ValidationError('Amount, currency, reference, customerId, smartCardNumber, and cableProvider are required');
        }
        const result = await vtu_bills_service_1.vtuBillsService.payCableBill({
            userId: req.user.userId,
            amount: Number(amount),
            currency: currency,
            type: 'cable',
            provider: provider || 'asbdata',
            customerId,
            reference,
            smartCardNumber,
            cableProvider: cableProvider,
            customerName,
            package: packageName,
            months: months ? Number(months) : 1,
        });
        res.status(200).json({
            success: true,
            message: 'Cable bill payment initiated successfully',
            data: {
                transaction: result.transaction,
                response: result.response,
            },
        });
    });
    static validateElectricityMeter = (0, errors_1.asyncHandler)(async (req, res) => {
        const { meterNumber, provider, meterType } = req.body;
        if (!meterNumber || !provider || !meterType) {
            throw new vtu_bills_types_1.ValidationError('Meter number, provider, and meter type are required');
        }
        const result = await vtu_bills_service_1.vtuBillsService.validateElectricityMeter(meterNumber, provider, meterType);
        res.status(200).json({
            success: true,
            message: 'Electricity meter validation completed',
            data: result,
        });
    });
    static validateCableIUC = (0, errors_1.asyncHandler)(async (req, res) => {
        const { smartCardNumber, provider } = req.body;
        if (!smartCardNumber || !provider) {
            throw new vtu_bills_types_1.ValidationError('Smart card number and provider are required');
        }
        const result = await vtu_bills_service_1.vtuBillsService.validateCableIUC(smartCardNumber, provider);
        res.status(200).json({
            success: true,
            message: 'Cable IUC validation completed',
            data: result,
        });
    });
    static buyExamPin = (0, errors_1.asyncHandler)(async (req, res) => {
        const { amount, currency, reference, examType, quantity, provider, } = req.body;
        if (!amount || !currency || !reference || !examType || !quantity) {
            throw new vtu_bills_types_1.ValidationError('Amount, currency, reference, examType, and quantity are required');
        }
        try {
            const cfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey('bills_exampins_enabled', 'service', { serviceType: 'bills' });
            if (cfg && typeof cfg.value === 'boolean' && cfg.value === false) {
                res.status(403).json({ success: false, message: 'Exam pin purchase is disabled' });
                return;
            }
        }
        catch { }
        const result = await vtu_bills_service_1.vtuBillsService.buyExamPin({
            userId: req.user.userId,
            amount: Number(amount),
            currency: currency,
            examType,
            quantity: Number(quantity),
            reference,
            provider,
        });
        res.status(200).json({
            success: true,
            message: 'Exam pin purchase initiated successfully',
            data: {
                transaction: result.transaction,
                response: result.response,
            },
        });
    });
    static generateRechargePins = (0, errors_1.asyncHandler)(async (req, res) => {
        const { amount, currency, reference, quantity, } = req.body;
        if (!amount || !currency || !reference || !quantity) {
            throw new vtu_bills_types_1.ValidationError('Amount, currency, reference, and quantity are required');
        }
        const result = await vtu_bills_service_1.vtuBillsService.generateRechargePins(req.user.userId, currency, Number(amount), Number(quantity), reference);
        res.status(200).json({
            success: true,
            message: 'Recharge pins generated successfully',
            data: {
                transaction: result.transaction,
                response: result.response,
            },
        });
    });
    static queryTransaction = (0, errors_1.asyncHandler)(async (req, res) => {
        const { reference, providerReference, startDate, endDate, status, type } = req.query;
        if (!reference && !providerReference) {
            throw new vtu_bills_types_1.ValidationError('Either reference or providerReference is required');
        }
        const result = await vtu_bills_service_1.vtuBillsService.queryTransaction({
            reference: reference,
            providerReference: providerReference,
            startDate: startDate,
            endDate: endDate,
            status: status,
            type: type,
        });
        res.status(200).json({
            success: true,
            message: 'Transaction query completed',
            data: result,
        });
    });
    static getTransactionHistory = (0, errors_1.asyncHandler)(async (req, res) => {
        const { type, status, startDate, endDate } = req.query;
        const filters = {};
        if (type)
            filters['type'] = type;
        if (status)
            filters['status'] = status;
        if (startDate)
            filters['startDate'] = startDate;
        if (endDate)
            filters['endDate'] = endDate;
        const result = await vtu_bills_service_1.vtuBillsService.getTransactionHistory(filters);
        res.status(200).json({
            success: true,
            message: 'Transaction history retrieved successfully',
            data: result,
        });
    });
    static getProviderStatus = (0, errors_1.asyncHandler)(async (_req, res) => {
        const status = vtu_bills_service_1.vtuBillsService.getProviderStatus();
        res.status(200).json({
            success: true,
            message: 'Provider status retrieved successfully',
            data: status,
        });
    });
    static getAvailableProviders = (0, errors_1.asyncHandler)(async (_req, res) => {
        const providers = vtu_bills_service_1.vtuBillsService.getAvailableProviders();
        res.status(200).json({
            success: true,
            message: 'Available providers retrieved successfully',
            data: providers,
        });
    });
    static setProvider = (0, errors_1.asyncHandler)(async (req, res) => {
        const { providerName } = req.body;
        if (!providerName) {
            throw new vtu_bills_types_1.ValidationError('Provider name is required');
        }
        vtu_bills_service_1.vtuBillsService.setProvider(providerName);
        res.status(200).json({
            success: true,
            message: `Provider switched to ${providerName} successfully`,
            data: {
                currentProvider: providerName,
            },
        });
    });
    static enableSandboxMode = (0, errors_1.asyncHandler)(async (_req, res) => {
        vtu_bills_service_1.vtuBillsService.enableSandboxMode();
        res.status(200).json({
            success: true,
            message: 'Sandbox mode enabled successfully',
            data: {
                sandboxMode: true,
                currentProvider: 'mock',
            },
        });
    });
    static disableSandboxMode = (0, errors_1.asyncHandler)(async (_req, res) => {
        vtu_bills_service_1.vtuBillsService.disableSandboxMode();
        res.status(200).json({
            success: true,
            message: 'Sandbox mode disabled successfully',
            data: {
                sandboxMode: false,
                currentProvider: 'asbdata',
            },
        });
    });
    static getSandboxStatus = (0, errors_1.asyncHandler)(async (_req, res) => {
        const sandboxMode = vtu_bills_service_1.vtuBillsService.isSandboxMode();
        res.status(200).json({
            success: true,
            message: 'Sandbox status retrieved successfully',
            data: {
                sandboxMode,
                currentProvider: vtu_bills_service_1.vtuBillsService.getProviderStatus().currentProvider,
            },
        });
    });
    static healthCheck = (0, errors_1.asyncHandler)(async (_req, res) => {
        const providerStatus = vtu_bills_service_1.vtuBillsService.getProviderStatus();
        const availableProviders = vtu_bills_service_1.vtuBillsService.getAvailableProviders();
        res.status(200).json({
            success: true,
            message: 'VTU/Bills service is healthy',
            data: {
                providerStatus,
                availableProviders,
                timestamp: new Date().toISOString(),
            },
        });
    });
}
exports.VtuBillsController = VtuBillsController;
