"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verificationService = exports.VerificationService = void 0;
const uuid_1 = require("uuid");
const redis_1 = require("../../config/redis");
const provider_manager_1 = require("../../integrations/verification/provider-manager");
const service_configuration_repo_1 = require("../../database/repositories/service-configuration.repo");
const verification_types_1 = require("../../integrations/verification/verification.types");
class VerificationService {
    static instance;
    constructor() { }
    static getInstance() {
        if (!VerificationService.instance) {
            VerificationService.instance = new VerificationService();
        }
        return VerificationService.instance;
    }
    async verifyNin(request, idempotencyKey) {
        const key = idempotencyKey || this.generateIdempotencyKey();
        try {
            const apiKeyConfig = await service_configuration_repo_1.serviceConfigurationRepository.getByKey('verification_provider_unify_api_key', 'provider', { serviceType: 'verification', providerId: 'unify' });
            if (apiKeyConfig?.value) {
                provider_manager_1.verificationProviderManager.updateUnifyProviderConfig(apiKeyConfig.value);
            }
            const existingResult = await this.checkIdempotency(key);
            if (existingResult) {
                return existingResult;
            }
            this.validateNinRequest(request);
            const transaction = await this.createTransaction({
                idempotencyKey: key,
                type: 'nin',
                request,
                status: 'pending',
                provider: provider_manager_1.verificationProviderManager.getCurrentProvider(),
                userId: request.userId,
            });
            const subService = this.resolveNinSubService(request);
            await this.ensureServiceEnabled('nin', subService);
            const providerInstance = await this.resolveProviderFor('nin', subService);
            if (apiKeyConfig?.value && this.isValidFullApiKey(apiKeyConfig.value)) {
                try {
                    providerInstance.updateConfig?.({ apiKey: apiKeyConfig.value });
                }
                catch { }
            }
            const response = await providerInstance.verifyNin(request);
            await this.updateTransaction(transaction.id, {
                status: 'completed',
                response,
                trackingId: response.trackingId,
            });
            const result = {
                success: true,
                data: response,
                trackingId: response.trackingId,
                timestamp: Date.now(),
                idempotencyKey: key,
            };
            await this.storeIdempotencyResult(key, result);
            return result;
        }
        catch (error) {
            if (error instanceof verification_types_1.IdempotencyError) {
                throw error;
            }
            await this.updateTransaction(key, {
                status: 'failed',
                response: { error: error instanceof Error ? error.message : 'Unknown error' },
            });
            throw error;
        }
    }
    async verifyBvn(request, idempotencyKey) {
        const key = idempotencyKey || this.generateIdempotencyKey();
        try {
            const [apiKeyConfig, baseUrlConfig] = await Promise.all([
                service_configuration_repo_1.serviceConfigurationRepository.getByKey('verification_provider_unify_api_key', 'provider', { serviceType: 'verification', providerId: 'unify' }),
                service_configuration_repo_1.serviceConfigurationRepository.getByKey('verification_provider_unify_base_url', 'provider', { serviceType: 'verification', providerId: 'unify' })
            ]);
            console.log('[VerificationService] Loading API key from admin config:', {
                found: !!apiKeyConfig,
                value: apiKeyConfig?.value ? `${apiKeyConfig.value.substring(0, 8)}...` : 'MISSING',
                length: apiKeyConfig?.value ? apiKeyConfig.value.length : 0
            });
            console.log('[VerificationService] Loading baseUrl from admin config:', {
                found: !!baseUrlConfig,
                value: baseUrlConfig?.value || 'NOT_PROVIDED'
            });
            if (apiKeyConfig?.value) {
                provider_manager_1.verificationProviderManager.updateUnifyProviderConfig(apiKeyConfig.value, baseUrlConfig?.value);
            }
            const existingResult = await this.checkIdempotency(key);
            if (existingResult) {
                return existingResult;
            }
            this.validateBvnRequest(request);
            const transaction = await this.createTransaction({
                idempotencyKey: key,
                type: 'bvn',
                request,
                status: 'pending',
                provider: provider_manager_1.verificationProviderManager.getCurrentProvider(),
                userId: request.userId,
            });
            await this.ensureServiceEnabled('bvn', 'by_bvn');
            const providerInstance = await this.resolveProviderFor('bvn', 'by_bvn');
            if (apiKeyConfig?.value && this.isValidFullApiKey(apiKeyConfig.value)) {
                try {
                    providerInstance.updateConfig?.({ apiKey: apiKeyConfig.value, baseUrl: baseUrlConfig?.value });
                }
                catch { }
            }
            const response = await providerInstance.verifyBvn(request);
            await this.updateTransaction(transaction.id, {
                status: 'completed',
                response,
            });
            const result = {
                success: true,
                data: response,
                timestamp: Date.now(),
                idempotencyKey: key,
            };
            await this.storeIdempotencyResult(key, result);
            return result;
        }
        catch (error) {
            if (error instanceof verification_types_1.IdempotencyError) {
                throw error;
            }
            await this.updateTransaction(key, {
                status: 'failed',
                response: { error: error instanceof Error ? error.message : 'Unknown error' },
            });
            throw error;
        }
    }
    async verifyBvnByPhone(request, idempotencyKey) {
        const key = idempotencyKey || this.generateIdempotencyKey();
        try {
            const existingResult = await this.checkIdempotency(key);
            if (existingResult) {
                return existingResult;
            }
            this.validateBvnByPhoneRequest(request);
            const transaction = await this.createTransaction({
                idempotencyKey: key,
                type: 'bvn_phone',
                request,
                status: 'pending',
                provider: provider_manager_1.verificationProviderManager.getCurrentProvider(),
            });
            throw new verification_types_1.ProviderError('BVN verification by phone number is not supported by the current provider. Please use BVN verification with the actual BVN number instead.', 400);
        }
        catch (error) {
            if (error instanceof verification_types_1.IdempotencyError) {
                throw error;
            }
            await this.updateTransaction(key, {
                status: 'failed',
                response: { error: error instanceof Error ? error.message : 'Unknown error' },
            });
            throw error;
        }
    }
    async createIpeClearance(request, idempotencyKey) {
        const key = idempotencyKey || this.generateIdempotencyKey();
        try {
            const existingResult = await this.checkIdempotency(key);
            if (existingResult) {
                return existingResult;
            }
            this.validateIpeClearanceRequest(request);
            const transaction = await this.createTransaction({
                idempotencyKey: key,
                type: 'ipe',
                request,
                status: 'pending',
                provider: provider_manager_1.verificationProviderManager.getCurrentProvider(),
            });
            const response = await provider_manager_1.verificationProviderManager.createIpeClearance(request);
            await this.updateTransaction(transaction.id, {
                status: 'completed',
                response,
                transactionId: String(response.trans_id),
            });
            const result = {
                success: true,
                data: response,
                transactionId: String(response.trans_id),
                timestamp: Date.now(),
                idempotencyKey: key,
            };
            await this.storeIdempotencyResult(key, result);
            return result;
        }
        catch (error) {
            if (error instanceof verification_types_1.IdempotencyError) {
                throw error;
            }
            await this.updateTransaction(key, {
                status: 'failed',
                response: { error: error instanceof Error ? error.message : 'Unknown error' },
            });
            throw error;
        }
    }
    async getIpeClearanceStatus(transactionId) {
        try {
            if (!transactionId || transactionId <= 0) {
                throw new verification_types_1.ValidationError('Invalid transaction ID');
            }
            return await provider_manager_1.verificationProviderManager.getIpeClearanceStatus(transactionId);
        }
        catch (error) {
            console.error('[VerificationService] IPE clearance status check failed:', error);
            throw error;
        }
    }
    async createPersonalization(request, idempotencyKey) {
        const key = idempotencyKey || this.generateIdempotencyKey();
        try {
            const existingResult = await this.checkIdempotency(key);
            if (existingResult) {
                return existingResult;
            }
            this.validatePersonalizationRequest(request);
            const transaction = await this.createTransaction({
                idempotencyKey: key,
                type: 'personalization',
                request,
                status: 'pending',
                provider: provider_manager_1.verificationProviderManager.getCurrentProvider(),
            });
            const response = await provider_manager_1.verificationProviderManager.createPersonalization(request);
            await this.updateTransaction(transaction.id, {
                status: 'completed',
                response,
                trackingId: response.trackingid,
            });
            const result = {
                success: true,
                data: response,
                trackingId: response.trackingid,
                timestamp: Date.now(),
                idempotencyKey: key,
            };
            await this.storeIdempotencyResult(key, result);
            return result;
        }
        catch (error) {
            if (error instanceof verification_types_1.IdempotencyError) {
                throw error;
            }
            await this.updateTransaction(key, {
                status: 'failed',
                response: { error: error instanceof Error ? error.message : 'Unknown error' },
            });
            throw error;
        }
    }
    async getProviderStatus() {
        return await provider_manager_1.verificationProviderManager.getProviderStatus();
    }
    async healthCheck() {
        return await provider_manager_1.verificationProviderManager.healthCheck();
    }
    enableSandboxMode() {
        provider_manager_1.verificationProviderManager.enableSandboxMode();
    }
    disableSandboxMode() {
        provider_manager_1.verificationProviderManager.disableSandboxMode();
    }
    isSandboxMode() {
        return provider_manager_1.verificationProviderManager.isSandboxMode();
    }
    getSandboxConfig() {
        return provider_manager_1.verificationProviderManager.getSandboxConfig();
    }
    updateSandboxConfig(config) {
        provider_manager_1.verificationProviderManager.updateSandboxConfig(config);
    }
    async getTransaction(idempotencyKey) {
        try {
            const transactionData = await redis_1.redisManager.get(`verification:transaction:${idempotencyKey}`);
            return transactionData ? JSON.parse(transactionData) : null;
        }
        catch (error) {
            console.error('[VerificationService] Failed to get transaction:', error);
            return null;
        }
    }
    async getTransactionsByType(type, limit = 50) {
        try {
            const pattern = `verification:transaction:*`;
            const keys = await redis_1.redisManager.keys(pattern);
            const transactions = [];
            for (const key of keys.slice(0, limit)) {
                const data = await redis_1.redisManager.get(key);
                if (data) {
                    const transaction = JSON.parse(data);
                    if (transaction.type === type) {
                        transactions.push(transaction);
                    }
                }
            }
            return transactions.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
        }
        catch (error) {
            console.error('[VerificationService] Failed to get transactions:', error);
            return [];
        }
    }
    async getUserTransactions(userId, limit = 50, offset = 0) {
        try {
            const pattern = `verification:transaction:*`;
            const keys = await redis_1.redisManager.keys(pattern);
            const transactions = [];
            for (const key of keys) {
                const data = await redis_1.redisManager.get(key);
                if (data) {
                    const transaction = JSON.parse(data);
                    if (transaction.metadata?.userId === userId || transaction.userId === userId) {
                        transactions.push(transaction);
                    }
                }
            }
            const sortedTransactions = transactions.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
            return sortedTransactions.slice(offset, offset + limit);
        }
        catch (error) {
            console.error('[VerificationService] Failed to get user transactions:', error);
            return [];
        }
    }
    generateIdempotencyKey() {
        return `verification_${(0, uuid_1.v4)()}`;
    }
    resolveNinSubService(request) {
        if (request.nin)
            return 'by_nin';
        if (request.phone)
            return 'by_phone';
        return 'by_details';
    }
    async ensureServiceEnabled(service, subService) {
        const keys = [
            `verification_${service}_${subService}_enabled`,
            `verification_${service}_enabled`,
            `verification_enabled`,
        ];
        for (const key of keys) {
            const cfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(key, 'service', { serviceType: 'verification' });
            if (cfg && typeof cfg.value === 'boolean' && cfg.value === false) {
                throw new verification_types_1.ValidationError(`${service.toUpperCase()} ${subService} is disabled`);
            }
        }
    }
    async resolveProviderFor(service, subService) {
        const nameKey = `verification_${service}_${subService}_provider`;
        const baseUrlKey = `verification_${service}_${subService}_base_url`;
        const apiKeyKey = `verification_${service}_${subService}_api_key`;
        const timeoutKey = `verification_${service}_${subService}_timeout`;
        const [nameCfg, baseUrlCfg, apiKeyCfg, timeoutCfg] = await Promise.all([
            service_configuration_repo_1.serviceConfigurationRepository.getByKey(nameKey, 'service', { serviceType: 'verification' }),
            service_configuration_repo_1.serviceConfigurationRepository.getByKey(baseUrlKey, 'service', { serviceType: 'verification' }),
            service_configuration_repo_1.serviceConfigurationRepository.getByKey(apiKeyKey, 'service', { serviceType: 'verification' }),
            service_configuration_repo_1.serviceConfigurationRepository.getByKey(timeoutKey, 'service', { serviceType: 'verification' }),
        ]);
        const providerName = nameCfg?.value || (provider_manager_1.verificationProviderManager.isSandboxMode() ? 'mock' : 'unify');
        const overrides = {};
        if (baseUrlCfg?.value)
            overrides.baseUrl = baseUrlCfg.value;
        if (apiKeyCfg?.value && this.isValidFullApiKey(apiKeyCfg.value)) {
            overrides.apiKey = apiKeyCfg.value;
        }
        if (typeof timeoutCfg?.value === 'number')
            overrides.timeout = timeoutCfg.value;
        return provider_manager_1.verificationProviderManager.createProviderInstance(providerName, overrides);
    }
    isValidFullApiKey(value) {
        if (!value || typeof value !== 'string')
            return false;
        if (value.includes('...'))
            return false;
        return value.length >= 16;
    }
    async checkIdempotency(key) {
        try {
            const result = await redis_1.redisManager.get(`verification:idempotency:${key}`);
            if (result) {
                const parsed = JSON.parse(result);
                if (Date.now() - parsed.timestamp < 24 * 60 * 60 * 1000) {
                    return parsed;
                }
            }
            return null;
        }
        catch (error) {
            console.error('[VerificationService] Idempotency check failed:', error);
            return null;
        }
    }
    async storeIdempotencyResult(key, result) {
        try {
            await redis_1.redisManager.setex(`verification:idempotency:${key}`, 24 * 60 * 60, JSON.stringify(result));
        }
        catch (error) {
            console.error('[VerificationService] Failed to store idempotency result:', error);
        }
    }
    async createTransaction(data) {
        const transaction = {
            id: (0, uuid_1.v4)(),
            idempotencyKey: data.idempotencyKey,
            type: data.type,
            request: data.request,
            status: data.status,
            provider: data.provider,
            createdAt: new Date(),
            updatedAt: new Date(),
            metadata: data.metadata || {},
            userId: data.userId,
        };
        try {
            await redis_1.redisManager.setex(`verification:transaction:${transaction.idempotencyKey}`, 7 * 24 * 60 * 60, JSON.stringify(transaction));
        }
        catch (error) {
            console.error('[VerificationService] Failed to create transaction:', error);
        }
        return transaction;
    }
    async updateTransaction(idempotencyKey, updates) {
        try {
            const existing = await this.getTransaction(idempotencyKey);
            if (existing) {
                const updated = { ...existing, ...updates, updatedAt: new Date() };
                await redis_1.redisManager.setex(`verification:transaction:${idempotencyKey}`, 7 * 24 * 60 * 60, JSON.stringify(updated));
            }
        }
        catch (error) {
            console.error('[VerificationService] Failed to update transaction:', error);
        }
    }
    validateNinRequest(request) {
        const hasNin = request.nin && request.nin.length === 11;
        const hasNameAndDetails = request.first_name && request.last_name && request.gender && request.dob;
        const hasPhone = request.phone && request.phone.length >= 10;
        if (!hasNin && !hasNameAndDetails && !hasPhone) {
            throw new verification_types_1.ValidationError('NIN verification requires either: NIN (11 digits), name/gender/dob combination, or phone number');
        }
        if (request.nin && !/^\d{11}$/.test(request.nin)) {
            throw new verification_types_1.ValidationError('NIN must be exactly 11 digits');
        }
        if (request.phone && !/^\d{10,11}$/.test(request.phone)) {
            throw new verification_types_1.ValidationError('Phone number must be 10-11 digits');
        }
        if (request.gender && !['male', 'female'].includes(request.gender)) {
            throw new verification_types_1.ValidationError('Gender must be either "male" or "female"');
        }
        if (request.dob && !/^\d{2}-\d{2}-\d{4}$/.test(request.dob)) {
            throw new verification_types_1.ValidationError('Date of birth must be in format: DD-MM-YYYY');
        }
    }
    validateBvnRequest(request) {
        if (!request.BVN || request.BVN.length !== 11) {
            throw new verification_types_1.ValidationError('BVN must be exactly 11 digits');
        }
        if (!/^\d{11}$/.test(request.BVN)) {
            throw new verification_types_1.ValidationError('BVN must contain only digits');
        }
    }
    validateBvnByPhoneRequest(request) {
        if (!request.phone) {
            throw new verification_types_1.ValidationError('Phone number is required');
        }
        const phoneRegex = /^(\+234|234|0)?[789][01]\d{8}$/;
        if (!phoneRegex.test(request.phone)) {
            throw new verification_types_1.ValidationError('Invalid Nigerian phone number format');
        }
        if (!request.userId) {
            throw new verification_types_1.ValidationError('User ID is required');
        }
    }
    validateIpeClearanceRequest(request) {
        if (!request.type || !request.tracking_ids || !Array.isArray(request.tracking_ids) || request.tracking_ids.length === 0) {
            throw new verification_types_1.ValidationError('Invalid IPE clearance request. Type and tracking_ids array are required.');
        }
        if (request.tracking_ids.length > 100) {
            throw new verification_types_1.ValidationError('Maximum 100 tracking IDs allowed per request');
        }
    }
    validatePersonalizationRequest(request) {
        if (!request.type || !request.tracking_id) {
            throw new verification_types_1.ValidationError('Invalid personalization request. Type and tracking_id are required.');
        }
        if (request.tracking_id.length < 10) {
            throw new verification_types_1.ValidationError('Invalid tracking ID format');
        }
    }
    async createPendingBvnByPhoneRequest(request, idempotencyKey, price = 0) {
        const key = idempotencyKey || this.generateIdempotencyKey();
        try {
            const existingResult = await this.checkIdempotency(key);
            if (existingResult) {
                return existingResult;
            }
            this.validateBvnByPhoneRequest(request);
            const transaction = await this.createTransaction({
                idempotencyKey: key,
                type: 'bvn_phone',
                request,
                status: 'pending',
                provider: 'admin_review',
                userId: request.userId,
                metadata: { price: price }
            });
            console.log('[VerificationService] Created pending BVN by phone request:', {
                idempotencyKey: key,
                transactionId: transaction.id,
                phoneNumber: request.phone,
                price: price
            });
            return {
                idempotencyKey: key,
                transactionId: transaction.id,
                status: 'pending',
                phoneNumber: request.phone,
                price: price,
                submittedAt: new Date().toISOString(),
                estimatedProcessingTime: '24-48 hours'
            };
        }
        catch (error) {
            console.error('[VerificationService] Error creating pending BVN by phone request:', error);
            throw error;
        }
    }
}
exports.VerificationService = VerificationService;
exports.verificationService = VerificationService.getInstance();
