"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.VerificationController = void 0;
const errors_1 = require("../../core/errors");
const verification_service_1 = require("./verification.service");
const verification_types_1 = require("../../integrations/verification/verification.types");
const crypto_1 = __importDefault(require("crypto"));
const service_configuration_repo_1 = require("../../database/repositories/service-configuration.repo");
const nin_slip_1 = require("../../database/models/nin-slip");
const provider_manager_1 = require("../../integrations/verification/provider-manager");
const verification_monitor_1 = require("./verification-monitor");
class VerificationController {
    static getPricing = (0, errors_1.asyncHandler)(async (_req, res) => {
        const configs = await service_configuration_repo_1.serviceConfigurationRepository.listByCategory('verification_pricing');
        const pricing = {};
        const subServices = {};
        for (const c of configs) {
            if (typeof c.value === 'number' && c.key.startsWith('verification_price_')) {
                const name = c.key.replace('verification_price_', '');
                if (name.includes('_')) {
                    const [mainService, subService] = name.split('_', 2);
                    if (!subServices[mainService]) {
                        subServices[mainService] = {};
                    }
                    subServices[mainService][subService] = c.value;
                }
                else {
                    pricing[name] = c.value;
                }
            }
        }
        res.json({
            success: true,
            data: pricing,
            subServices: subServices
        });
    });
    static healthCheck = (0, errors_1.asyncHandler)(async (_req, res) => {
        const isHealthy = await verification_service_1.verificationService.healthCheck();
        const providerStatus = await verification_service_1.verificationService.getProviderStatus();
        res.json({
            success: true,
            healthy: isHealthy,
            timestamp: new Date().toISOString(),
            providerStatus,
            sandboxMode: verification_service_1.verificationService.isSandboxMode(),
        });
    });
    static detailedHealthCheck = (0, errors_1.asyncHandler)(async (_req, res) => {
        const monitor = verification_monitor_1.VerificationMonitor.getInstance();
        const healthReport = await monitor.getHealthReport();
        res.json({
            success: true,
            ...healthReport
        });
    });
    static getMetrics = (0, errors_1.asyncHandler)(async (_req, res) => {
        const monitor = verification_monitor_1.VerificationMonitor.getInstance();
        const metrics = await monitor.getMetrics();
        res.json({
            success: true,
            metrics
        });
    });
    static verifyNin = (0, errors_1.asyncHandler)(async (req, res) => {
        const idempotencyKey = req.headers['x-idempotency-key'];
        const request = req.body;
        const ninPriceCfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey('verification_price_nin', 'service', { serviceType: 'verification' });
        const price = ninPriceCfg?.value ?? 0;
        if (price > 0) {
            const { walletService } = await Promise.resolve().then(() => __importStar(require('../wallet/wallet.service')));
            const balances = await walletService.getUserBalanceByCurrency(req.user?.userId, 'NGN');
            if (!balances || balances.availableBalance < price) {
                res.status(402).json({
                    success: false,
                    message: `Insufficient balance. Required ₦${price}, Available ₦${balances?.availableBalance ?? 0}`,
                    requiredAmount: price,
                    availableBalance: balances?.availableBalance ?? 0,
                });
                return;
            }
        }
        try {
            const result = await verification_service_1.verificationService.verifyNin(request, idempotencyKey);
            try {
                if (price > 0) {
                    const { walletService } = await Promise.resolve().then(() => __importStar(require('../wallet/wallet.service')));
                    const { transactionRepository } = await Promise.resolve().then(() => __importStar(require('../../database/repositories/transactions')));
                    const chargeRef = `VERIF_NIN_CHARGE_${result?.idempotencyKey || idempotencyKey}`;
                    const exists = await transactionRepository.findByReference(chargeRef);
                    if (!exists) {
                        await walletService.debit({
                            userId: req.user?.userId,
                            currency: 'NGN',
                            amount: price,
                            reference: chargeRef,
                            category: 'bill_payment',
                            description: 'NIN verification charge',
                            metadata: { type: 'verification', service: 'nin', idempotencyKey: result?.idempotencyKey || idempotencyKey }
                        });
                    }
                }
            }
            catch (e) {
                console.warn('[verifyNin] Wallet debit skipped/failed:', e?.message);
            }
            try {
                const userId = req.user?.userId;
                const trackingId = result?.trackingId || result?.data?.trackingId;
                const payload = result?.data || {};
                if (userId && trackingId) {
                    await nin_slip_1.NinSlip.findOneAndUpdate({ userId, trackingId }, {
                        userId,
                        trackingId,
                        nin: payload.nin || '',
                        surname: payload.surname || payload.lastname || payload.lastName || '',
                        firstname: payload.firstname || payload.firstName || '',
                        middlename: payload.middlename || payload.middleName || '',
                        gender: payload.gender || '',
                        birthdate: payload.birthdate || payload.dob || '',
                        photo: payload.photo || '',
                    }, { upsert: true, new: true });
                }
            }
            catch (e) {
                console.warn('[verifyNin] Failed to upsert NinSlip record:', e?.message);
            }
            res.json({
                success: true,
                data: result,
                price,
                message: 'NIN verification completed successfully',
            });
        }
        catch (err) {
            const msg = err?.message || '';
            const status = err?.statusCode || err?.status || 500;
            if (status === 504 || /timeout/i.test(String(msg))) {
                res.status(504).json({ success: false, message: 'Verification provider timed out. Please try again later.' });
                return;
            }
            if (status === 503 || /temporarily unavailable|try again later/i.test(String(msg))) {
                res.setHeader('Retry-After', '30');
                res.status(503).json({ success: false, message: 'Verification service temporarily unavailable. Please try again later.' });
                return;
            }
            if (typeof msg === 'string' && msg.toLowerCase().includes('no record found')) {
                res.status(404).json({ success: false, message: 'No record found. Please check your details and try again.' });
                return;
            }
            throw err;
        }
    });
    static verifyBvn = (0, errors_1.asyncHandler)(async (req, res) => {
        const idempotencyKey = req.headers['x-idempotency-key'];
        const request = req.body;
        try {
            const subService = 'by_bvn';
            const cfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(`verification_bvn_${subService}_enabled`, 'service', { serviceType: 'verification' });
            if (cfg && typeof cfg.value === 'boolean' && cfg.value === false) {
                res.status(403).json({ success: false, message: 'BVN verification is disabled' });
                return;
            }
        }
        catch { }
        const bvnPriceCfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey('verification_price_bvn_verification', 'service', { serviceType: 'verification' });
        const price = bvnPriceCfg?.value ?? 0;
        console.log('[verifyBvn] Price configuration:', {
            found: !!bvnPriceCfg,
            price: price,
            subService: 'bvn_verification',
            userId: req.user?.userId
        });
        if (price > 0) {
            const { walletService } = await Promise.resolve().then(() => __importStar(require('../wallet/wallet.service')));
            const balances = await walletService.getUserBalanceByCurrency(req.user?.userId, 'NGN');
            if (!balances || balances.availableBalance < price) {
                res.status(402).json({
                    success: false,
                    message: `Insufficient balance. Required ₦${price}, Available ₦${balances?.availableBalance ?? 0}`,
                    requiredAmount: price,
                    availableBalance: balances?.availableBalance ?? 0,
                });
                return;
            }
        }
        try {
            const result = await verification_service_1.verificationService.verifyBvn(request, idempotencyKey);
            try {
                console.log('[verifyBvn] Attempting wallet debit:', {
                    price: price,
                    userId: req.user?.userId,
                    resultIdempotencyKey: result?.idempotencyKey,
                    requestIdempotencyKey: idempotencyKey
                });
                if (price > 0) {
                    const { walletService } = await Promise.resolve().then(() => __importStar(require('../wallet/wallet.service')));
                    const { transactionRepository } = await Promise.resolve().then(() => __importStar(require('../../database/repositories/transactions')));
                    const chargeRef = `VERIF_BVN_CHARGE_${result?.idempotencyKey || idempotencyKey}`;
                    const exists = await transactionRepository.findByReference(chargeRef);
                    console.log('[verifyBvn] Transaction check:', {
                        chargeRef: chargeRef,
                        exists: !!exists
                    });
                    if (!exists) {
                        console.log('[verifyBvn] Proceeding with wallet debit...');
                        await walletService.debit({
                            userId: req.user?.userId,
                            currency: 'NGN',
                            amount: price,
                            reference: chargeRef,
                            category: 'bill_payment',
                            description: 'BVN verification charge',
                            metadata: { type: 'verification', service: 'bvn', idempotencyKey: result?.idempotencyKey || idempotencyKey }
                        });
                        console.log('[verifyBvn] Wallet debit completed successfully');
                    }
                    else {
                        console.log('[verifyBvn] Transaction already exists, skipping debit');
                    }
                }
                else {
                    console.log('[verifyBvn] Price is 0, skipping wallet debit');
                }
            }
            catch (e) {
                console.warn('[verifyBvn] Wallet debit skipped/failed:', e?.message);
            }
            res.json({
                success: true,
                data: result,
                price,
                message: 'BVN verification completed successfully',
            });
        }
        catch (err) {
            const msg = err?.message || '';
            const status = err?.statusCode || err?.status || 500;
            if (status === 504 || /timeout/i.test(String(msg))) {
                res.status(504).json({ success: false, message: 'Verification provider timed out. Please try again later.' });
                return;
            }
            if (status === 503 || /temporarily unavailable|try again later/i.test(String(msg))) {
                res.setHeader('Retry-After', '30');
                res.status(503).json({ success: false, message: 'Verification service temporarily unavailable. Please try again later.' });
                return;
            }
            if (typeof msg === 'string' && msg.toLowerCase().includes('no record found')) {
                res.status(404).json({ success: false, message: 'No record found. Please check your details and try again.' });
                return;
            }
            throw err;
        }
    });
    static verifyBvnByPhone = (0, errors_1.asyncHandler)(async (req, res) => {
        const idempotencyKey = req.headers['x-idempotency-key'];
        const request = req.body;
        try {
            const subService = 'by_phone';
            const cfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(`verification_bvn_${subService}_enabled`, 'service', { serviceType: 'verification' });
            if (cfg && typeof cfg.value === 'boolean' && cfg.value === false) {
                res.status(403).json({ success: false, message: 'BVN verification by phone is disabled' });
                return;
            }
        }
        catch { }
        const bvnPhonePriceCfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey('verification_price_bvn_by_phone', 'service', { serviceType: 'verification' });
        const price = bvnPhonePriceCfg?.value ?? 0;
        console.log('[verifyBvnByPhone] Price configuration:', {
            found: !!bvnPhonePriceCfg,
            price: price,
            subService: 'bvn_by_phone',
            userId: req.user?.userId
        });
        if (price > 0) {
            const { walletService } = await Promise.resolve().then(() => __importStar(require('../wallet/wallet.service')));
            const balances = await walletService.getUserBalanceByCurrency(req.user?.userId, 'NGN');
            if (!balances || balances.availableBalance < price) {
                res.status(402).json({
                    success: false,
                    message: `Insufficient balance. Required ₦${price}, Available ₦${balances?.availableBalance ?? 0}`,
                    requiredAmount: price,
                    availableBalance: balances?.availableBalance ?? 0,
                });
                return;
            }
        }
        try {
            const result = await verification_service_1.verificationService.createPendingBvnByPhoneRequest(request, idempotencyKey, price);
            res.json({
                success: true,
                data: {
                    ...result,
                    status: 'pending_review',
                    message: 'Your BVN by phone verification request has been submitted for admin review. You will be notified once processed.',
                    estimatedProcessingTime: '24-48 hours'
                },
                price,
            });
        }
        catch (err) {
            if (err.statusCode) {
                res.status(err.statusCode).json({
                    success: false,
                    message: err.message,
                    error: err.error,
                });
                return;
            }
            throw err;
        }
    });
    static createIpeClearance = (0, errors_1.asyncHandler)(async (req, res) => {
        const idempotencyKey = req.headers['x-idempotency-key'];
        const request = req.body;
        const result = await verification_service_1.verificationService.createIpeClearance(request, idempotencyKey);
        res.json({
            success: true,
            data: result,
            message: 'IPE clearance created successfully',
        });
    });
    static getIpeClearanceStatus = (0, errors_1.asyncHandler)(async (req, res) => {
        const { transactionId } = req.params;
        const parsedTransactionId = parseInt(transactionId, 10);
        if (isNaN(parsedTransactionId)) {
            throw new verification_types_1.ValidationError('Invalid transaction ID');
        }
        const result = await verification_service_1.verificationService.getIpeClearanceStatus(parsedTransactionId);
        res.json({
            success: true,
            data: result,
            message: 'IPE clearance status retrieved successfully',
        });
    });
    static createPersonalization = (0, errors_1.asyncHandler)(async (req, res) => {
        const idempotencyKey = req.headers['x-idempotency-key'];
        const request = req.body;
        const result = await verification_service_1.verificationService.createPersonalization(request, idempotencyKey);
        res.json({
            success: true,
            data: result,
            message: 'Personalization created successfully',
        });
    });
    static getProviderStatus = (0, errors_1.asyncHandler)(async (_req, res) => {
        const status = await verification_service_1.verificationService.getProviderStatus();
        res.json({
            success: true,
            data: status,
            currentProvider: provider_manager_1.verificationProviderManager.getCurrentProviderName(),
            sandboxMode: verification_service_1.verificationService.isSandboxMode(),
        });
    });
    static enableSandboxMode = (0, errors_1.asyncHandler)(async (_req, res) => {
        verification_service_1.verificationService.enableSandboxMode();
        res.json({
            success: true,
            message: 'Sandbox mode enabled',
            sandboxMode: true,
            currentProvider: provider_manager_1.verificationProviderManager.getCurrentProviderName(),
        });
    });
    static disableSandboxMode = (0, errors_1.asyncHandler)(async (_req, res) => {
        verification_service_1.verificationService.disableSandboxMode();
        res.json({
            success: true,
            message: 'Sandbox mode disabled',
            sandboxMode: false,
            currentProvider: provider_manager_1.verificationProviderManager.getCurrentProviderName(),
        });
    });
    static getSandboxConfig = (0, errors_1.asyncHandler)(async (_req, res) => {
        const config = verification_service_1.verificationService.getSandboxConfig();
        res.json({
            success: true,
            data: config,
            currentProvider: provider_manager_1.verificationProviderManager.getCurrentProviderName(),
        });
    });
    static updateSandboxConfig = (0, errors_1.asyncHandler)(async (req, res) => {
        const config = req.body;
        verification_service_1.verificationService.updateSandboxConfig(config);
        res.json({
            success: true,
            message: 'Sandbox configuration updated',
            data: verification_service_1.verificationService.getSandboxConfig(),
            currentProvider: provider_manager_1.verificationProviderManager.getCurrentProviderName(),
        });
    });
    static updateProviderConfig = (0, errors_1.asyncHandler)(async (req, res) => {
        if (!req.user || req.user.role !== 'admin') {
            res.status(403).json({ success: false, message: 'Admin access required' });
            return;
        }
        const { baseUrl, apiKey, timeout, retries } = req.body || {};
        provider_manager_1.verificationProviderManager.updateCurrentProviderConfig({ baseUrl, apiKey, timeout, retries });
        const currentProvider = provider_manager_1.verificationProviderManager.getCurrentProviderName();
        if (baseUrl) {
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: `verification_provider_${currentProvider}_base_url`,
                scope: 'provider',
                serviceType: 'verification',
                providerId: currentProvider,
                category: 'verification_provider',
                value: baseUrl,
            });
        }
        if (apiKey) {
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: `verification_provider_${currentProvider}_api_key`,
                scope: 'provider',
                serviceType: 'verification',
                providerId: currentProvider,
                category: 'verification_provider',
                value: apiKey,
            });
        }
        if (typeof timeout === 'number') {
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: `verification_provider_${currentProvider}_timeout`,
                scope: 'provider',
                serviceType: 'verification',
                providerId: currentProvider,
                category: 'verification_provider',
                value: timeout,
            });
        }
        if (typeof retries === 'number') {
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: `verification_provider_${currentProvider}_retries`,
                scope: 'provider',
                serviceType: 'verification',
                providerId: currentProvider,
                category: 'verification_provider',
                value: retries,
            });
        }
        res.json({
            success: true,
            message: 'Provider configuration updated',
            currentProvider,
        });
    });
    static getTransaction = (0, errors_1.asyncHandler)(async (req, res) => {
        const { idempotencyKey } = req.params;
        const transaction = await verification_service_1.verificationService.getTransaction(idempotencyKey);
        if (!transaction) {
            res.status(404).json({
                success: false,
                message: 'Transaction not found',
            });
            return;
        }
        res.json({
            success: true,
            data: transaction,
        });
    });
    static getTransactionsByType = (0, errors_1.asyncHandler)(async (req, res) => {
        const { type } = req.params;
        const limit = parseInt(req.query['limit'], 10) || 50;
        const transactions = await verification_service_1.verificationService.getTransactionsByType(type, limit);
        res.json({
            success: true,
            data: transactions,
            count: transactions.length,
        });
    });
    static getIndividualTransactions = (0, errors_1.asyncHandler)(async (req, res) => {
        const userId = req.user?.userId;
        if (!userId) {
            throw new verification_types_1.ValidationError('User ID is required');
        }
        const limit = parseInt(req.query['limit'], 10) || 50;
        const offset = parseInt(req.query['offset'], 10) || 0;
        const transactions = await verification_service_1.verificationService.getUserTransactions(userId, limit, offset);
        res.json({
            success: true,
            data: transactions,
            count: transactions.length,
            limit,
            offset,
        });
    });
    static handleWebhook = (0, errors_1.asyncHandler)(async (req, res) => {
        const rawBody = req.body;
        const signature = req.headers['x-webhook-signature'];
        const timestamp = req.headers['x-webhook-timestamp'];
        const idempotencyKey = req.headers['x-idempotency-key'];
        if (!signature || !timestamp || !idempotencyKey) {
            res.status(400).json({
                success: false,
                message: 'Missing required webhook headers: x-webhook-signature, x-webhook-timestamp, x-idempotency-key',
            });
            return;
        }
        const isValidSignature = this.verifyWebhookSignature(rawBody, signature, timestamp);
        if (!isValidSignature) {
            res.status(401).json({
                success: false,
                message: 'Invalid webhook signature',
            });
            return;
        }
        const existingResult = await verification_service_1.verificationService.getTransaction(idempotencyKey);
        if (existingResult) {
            res.json({
                success: true,
                message: 'Webhook already processed',
                data: existingResult,
            });
            return;
        }
        const eventType = req.headers['x-webhook-event'];
        let result;
        try {
            switch (eventType) {
                case 'nin.verification.completed':
                    result = await this.processNinVerificationWebhook(rawBody, idempotencyKey);
                    break;
                case 'bvn.verification.completed':
                    result = await this.processBvnVerificationWebhook(rawBody, idempotencyKey);
                    break;
                case 'ipe.clearance.completed':
                    result = await this.processIpeClearanceWebhook(rawBody, idempotencyKey);
                    break;
                case 'personalization.completed':
                    result = await this.processPersonalizationWebhook(rawBody, idempotencyKey);
                    break;
                default:
                    res.status(400).json({
                        success: false,
                        message: `Unsupported webhook event type: ${eventType}`,
                    });
                    return;
            }
            res.json({
                success: true,
                message: 'Webhook processed successfully',
                data: result,
            });
        }
        catch (error) {
            console.error('[VerificationController] Webhook processing failed:', error);
            res.status(500).json({
                success: false,
                message: 'Webhook processing failed',
                error: error instanceof Error ? error.message : String(error),
            });
        }
    });
    static verifyWebhookSignature(rawBody, signature, timestamp) {
        try {
            const webhookSecret = process.env['UNIFY_WEBHOOK_SECRET'];
            if (!webhookSecret) {
                console.warn('[VerificationController] Webhook secret not configured');
                return true;
            }
            const payload = typeof rawBody === 'string' ? rawBody : JSON.stringify(rawBody);
            const expectedSignature = crypto_1.default
                .createHmac('sha256', webhookSecret)
                .update(`${timestamp}.${payload}`)
                .digest('hex');
            return crypto_1.default.timingSafeEqual(Buffer.from(signature, 'hex'), Buffer.from(expectedSignature, 'hex'));
        }
        catch (error) {
            console.error('[VerificationController] Signature verification failed:', error);
            return false;
        }
    }
    static async processNinVerificationWebhook(rawBody, idempotencyKey) {
        const ninData = rawBody.data;
        await verification_service_1.verificationService.createTransaction({
            idempotencyKey,
            type: 'nin',
            request: { webhook: true },
            status: 'completed',
            provider: verification_service_1.verificationService.isSandboxMode() ? 'mock' : 'unify',
            response: ninData,
            trackingId: ninData.trackingId,
        });
        return {
            type: 'nin.verification.completed',
            data: ninData,
            timestamp: Date.now(),
            idempotencyKey,
        };
    }
    static async processBvnVerificationWebhook(rawBody, idempotencyKey) {
        const bvnData = rawBody.data;
        await verification_service_1.verificationService.createTransaction({
            idempotencyKey,
            type: 'bvn',
            request: { webhook: true },
            status: 'completed',
            provider: verification_service_1.verificationService.isSandboxMode() ? 'mock' : 'unify',
            response: bvnData,
        });
        return {
            type: 'bvn.verification.completed',
            data: bvnData,
            timestamp: Date.now(),
            idempotencyKey,
        };
    }
    static async processIpeClearanceWebhook(rawBody, idempotencyKey) {
        const ipeData = rawBody.data;
        await verification_service_1.verificationService.createTransaction({
            idempotencyKey,
            type: 'ipe',
            request: { webhook: true },
            status: 'completed',
            provider: verification_service_1.verificationService.isSandboxMode() ? 'mock' : 'unify',
            response: ipeData,
            transactionId: ipeData.transactionId,
        });
        return {
            type: 'ipe.clearance.completed',
            data: ipeData,
            timestamp: Date.now(),
            idempotencyKey,
        };
    }
    static async processPersonalizationWebhook(rawBody, idempotencyKey) {
        const personalizationData = rawBody.data;
        await verification_service_1.verificationService.createTransaction({
            idempotencyKey,
            type: 'personalization',
            request: { webhook: true },
            status: 'completed',
            provider: verification_service_1.verificationService.isSandboxMode() ? 'mock' : 'unify',
            response: personalizationData,
            trackingId: personalizationData.trackingid,
        });
        return {
            type: 'personalization.completed',
            data: personalizationData,
            timestamp: Date.now(),
            idempotencyKey,
        };
    }
    static handleValidationError = (error, _req, res) => {
        res.status(400).json({
            success: false,
            error: 'Validation Error',
            message: error.message,
            details: error.details,
        });
    };
    static handleIdempotencyError = (error, _req, res) => {
        res.status(409).json({
            success: false,
            error: 'Idempotency Error',
            message: error.message,
        });
    };
    static handleProviderError = (error, _req, res) => {
        res.status(error.statusCode || 500).json({
            success: false,
            error: 'Provider Error',
            message: error.message,
            details: error.details,
        });
    };
}
exports.VerificationController = VerificationController;
