"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NinSlipController = void 0;
const pdfkit_1 = __importDefault(require("pdfkit"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const nin_slip_1 = require("../../database/models/nin-slip");
function writePremiumSlip(doc, details) {
    const bgPath = path_1.default.join(process.cwd(), 'public', 'img', 'premium7.png');
    if (fs_1.default.existsSync(bgPath)) {
        doc.image(bgPath, 0, 0, { width: 243, height: 153 });
    }
    const { nin = '', surname = '', firstname = '', middlename = '', gender = '', birthdate = '', photo = '' } = details || {};
    try {
        if (photo) {
            if (photo.startsWith('data:image')) {
                const base64 = photo.split(',')[1] || '';
                const buf = Buffer.from(base64, 'base64');
                doc.image(buf, 9, 50, { width: 80, height: 95 });
            }
            else if (fs_1.default.existsSync(photo)) {
                doc.image(photo, 9, 50, { width: 80, height: 95 });
            }
        }
    }
    catch { }
    const contentLeft = 110;
    const contentTop = 25;
    doc.fillColor('#000');
    doc.fontSize(10).text(String(surname || '').toUpperCase(), contentLeft - 9, contentTop + 42, { width: 120, ellipsis: true });
    const firstMiddle = middlename ? `${String(firstname || '').toUpperCase()}, ${String(middlename || '').toUpperCase()}` : String(firstname || '').toUpperCase();
    doc.fontSize(10).text(firstMiddle, contentLeft - 9, contentTop + 70, { width: 120, ellipsis: true });
    doc.fontSize(10).text(String(birthdate || ''), contentLeft - 9, contentTop + 82, { width: 120, ellipsis: true });
    doc.fontSize(10).text(String(gender || ''), 85, 100, { width: 60, ellipsis: true });
    const ninRaw = String(nin || '');
    if (ninRaw) {
        doc.save().fillColor('#e3dcdc').fontSize(7).rotate(330, { origin: [0, 0] }).text(ninRaw, -95, 107).restore();
        doc.save().fillColor('#a6a1a150').fontSize(7).rotate(180, { origin: [0, 0] }).text(ninRaw, -133, -75).restore();
    }
    const tight = ninRaw.replace(/\s+/g, '');
    const pretty = tight ? `${tight.slice(0, 4)}  ${tight.slice(4, 7)}  ${tight.slice(7, 11)}` : '';
    if (pretty) {
        doc.fontSize(20).fillColor('#131111');
        const textWidth = doc.widthOfString(pretty);
        const centerX = 243 / 2 - textWidth / 2;
        doc.text(pretty, centerX, 153 - 24);
    }
    const now = new Date().toLocaleDateString('en-GB', { day: '2-digit', month: 'short', year: 'numeric' }).replaceAll('/', ' ');
    doc.fontSize(9).fillColor('#000');
    const dateWidth = doc.widthOfString(now);
    doc.text(now, 243 - 18 - dateWidth, 153 - 60);
}
class NinSlipController {
    static async getPremiumSlip(req, res) {
        try {
            const userId = req.user?.userId;
            const trackingId = req.params['trackingId'];
            if (!userId || !trackingId) {
                res.status(400).json({ success: false, message: 'Missing user or trackingId' });
                return;
            }
            const existing = await nin_slip_1.NinSlip.findOne({ userId, trackingId }).lean();
            if (!existing) {
                res.status(404).json({ success: false, message: 'Slip not available. Please verify again to generate.' });
                return;
            }
            res.setHeader('Content-Type', 'application/pdf');
            res.setHeader('Content-Disposition', `attachment; filename="nin-slip-${trackingId}.pdf"`);
            res.setHeader('Cache-Control', 'no-store');
            const doc = new pdfkit_1.default({ size: [243, 153], margin: 0 });
            doc.pipe(res);
            writePremiumSlip(doc, existing);
            const backPath = path_1.default.join(process.cwd(), 'public', 'img', 'back.png');
            if (fs_1.default.existsSync(backPath)) {
                doc.addPage({ size: [243, 153], margin: 0 });
                doc.image(backPath, 0, 0, { width: 243, height: 153 });
            }
            doc.end();
        }
        catch (err) {
            console.error('[NinSlipController] Failed to generate slip:', err);
            res.status(500).json({ success: false, message: 'Failed to generate slip' });
        }
    }
    static async getStandardSlip(req, res) {
        try {
            const userId = req.user?.userId;
            const trackingId = req.params['trackingId'];
            if (!userId || !trackingId) {
                res.status(400).json({ success: false, message: 'Missing user or trackingId' });
                return;
            }
            const existing = await nin_slip_1.NinSlip.findOne({ userId, trackingId }).lean();
            if (!existing) {
                res.status(404).json({ success: false, message: 'Slip data not found. Verify first.' });
                return;
            }
            res.setHeader('Content-Type', 'application/pdf');
            res.setHeader('Content-Disposition', `attachment; filename="nin-standard-slip-${trackingId}.pdf"`);
            res.setHeader('Cache-Control', 'no-store');
            const doc = new pdfkit_1.default({ size: [243, 135], margin: 0 });
            doc.pipe(res);
            const bgPath = path_1.default.join(process.cwd(), 'public', 'img', 'standard8.png');
            if (fs_1.default.existsSync(bgPath)) {
                doc.image(bgPath, 0, 0, { width: 243, height: 135 });
            }
            try {
                const photo = existing.photo;
                if (photo) {
                    if (photo.startsWith('data:image')) {
                        const base64 = photo.split(',')[1] || '';
                        const buf = Buffer.from(base64, 'base64');
                        doc.image(buf, 10, 35, { width: 56, height: 70 });
                    }
                    else if (fs_1.default.existsSync(photo)) {
                        doc.image(photo, 10, 35, { width: 56, height: 70 });
                    }
                }
            }
            catch { }
            doc.fontSize(10).fillColor('#000');
            const last = existing.surname || '';
            const first = existing.firstname || '';
            const mid = existing.middlename || '';
            doc.text(String(last).toUpperCase(), 79.2 - 11.52, 43.2 - 0.72);
            doc.text(mid ? `${String(first).toUpperCase()}, ${String(mid).toUpperCase()}` : String(first).toUpperCase(), 79.2 - 11.52, 43.2 + 0.27 * 72);
            doc.text(existing.birthdate || '', 79.2 - 11.52, 43.2 + 0.32 * 72);
            const nin = String(existing.nin || '').replace(/\s+/g, '');
            const prettyNin = nin ? `${nin.slice(0, 4)}  ${nin.slice(4, 7)}  ${nin.slice(7, 11)}` : '';
            doc.fontSize(21.6).text(prettyNin, 43.2, 135 - 3.6);
            const backPath = path_1.default.join(process.cwd(), 'public', 'img', 'back.png');
            if (fs_1.default.existsSync(backPath)) {
                doc.addPage({ size: [243, 135], margin: 0 });
                doc.image(backPath, 0, 0, { width: 243, height: 135 });
            }
            doc.end();
        }
        catch (err) {
            console.error('[NinSlipController] Failed to generate standard slip:', err);
            res.status(500).json({ success: false, message: 'Failed to generate standard slip' });
        }
    }
    static async getRegularSlip(req, res) {
        try {
            const userId = req.user?.userId;
            const trackingId = req.params['trackingId'];
            if (!userId || !trackingId) {
                res.status(400).json({ success: false, message: 'Missing user or trackingId' });
                return;
            }
            const existing = await nin_slip_1.NinSlip.findOne({ userId, trackingId }).lean();
            if (!existing) {
                res.status(404).json({ success: false, message: 'Slip data not found. Verify first.' });
                return;
            }
            res.setHeader('Content-Type', 'application/pdf');
            res.setHeader('Content-Disposition', `attachment; filename="nin-regular-slip-${trackingId}.pdf"`);
            res.setHeader('Cache-Control', 'no-store');
            const doc = new pdfkit_1.default({ size: [892, 500], margin: 0 });
            doc.pipe(res);
            const bgPath = path_1.default.join(process.cwd(), 'public', 'img', 'regular.png');
            if (fs_1.default.existsSync(bgPath)) {
                doc.image(bgPath, 0, 0, { width: 892, height: 500 });
            }
            doc.font('Helvetica-Bold').fontSize(25).fillColor('#000').text('National Identity Management System', 214, 32);
            doc.font('Helvetica-Bold').fontSize(16).text('Federal Republic of Nigeria', 330, 65);
            doc.font('Helvetica-Bold').fontSize(13).text('National Identification Number Slip (NINS)', 308, 89);
            doc.font('Helvetica-Bold').fontSize(12).text('Tracking ID:', 36, 134);
            doc.font('Helvetica').fontSize(11).text(String(existing.trackingId || trackingId), 132, 135);
            doc.font('Helvetica-Bold').fontSize(12).text('NIN:', 36, 179);
            doc.font('Helvetica').fontSize(12).text(String(existing.nin || ''), 132, 178);
            doc.font('Helvetica-Bold').fontSize(12).text('Surname:', 254, 137);
            doc.font('Helvetica').fontSize(12).text(String(existing.surname || ''), 335, 136);
            doc.font('Helvetica-Bold').fontSize(12).text('First Name:', 254, 179);
            doc.font('Helvetica').fontSize(12).text(String(existing.firstname || ''), 335, 179);
            doc.font('Helvetica-Bold').fontSize(12).text('Middle Name:', 254, 221);
            doc.font('Helvetica').fontSize(12).text(String(existing.middlename || ''), 335, 221);
            doc.font('Helvetica-Bold').fontSize(12).text('Gender:', 254, 257);
            doc.font('Helvetica').fontSize(12).text(String(existing.gender || ''), 335, 257);
            doc.font('Helvetica-Bold').fontSize(12).text('Address:', 507, 128);
            doc.font('Helvetica').fontSize(11).text(String(existing.address || ''), 507, 146, { width: 220 });
            doc.font('Helvetica').fontSize(11).text(String(existing.lga || ''), 509, 242);
            doc.font('Helvetica').fontSize(11).text(String(existing.state || ''), 509, 266);
            try {
                const photo = existing.photo;
                if (photo) {
                    if (photo.startsWith('data:image')) {
                        const base64 = photo.split(',')[1] || '';
                        const buf = Buffer.from(base64, 'base64');
                        doc.image(buf, 736, 118, { width: 120, height: 160 });
                    }
                    else if (fs_1.default.existsSync(photo)) {
                        doc.image(photo, 736, 118, { width: 120, height: 160 });
                    }
                }
            }
            catch { }
            doc.end();
        }
        catch (err) {
            console.error('[NinSlipController] Failed to generate regular slip:', err);
            res.status(500).json({ success: false, message: 'Failed to generate regular slip' });
        }
    }
}
exports.NinSlipController = NinSlipController;
