"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NinSlipTestController = void 0;
const pdfkit_1 = __importDefault(require("pdfkit"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const MOCK_NIN_DATA = {
    nin: '12345678901',
    surname: 'ADEBAYO',
    firstname: 'OLUWASEUN',
    middlename: 'MICHAEL',
    gender: 'MALE',
    birthdate: '15/03/1990',
    photo: '',
    trackingId: 'TEST-TRACKING-001',
};
const MOCK_PHOTO_BASE64 = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==';
function writePremiumSlip(doc, details) {
    const bgPath = path_1.default.join(process.cwd(), 'public', 'img', 'premium7.png');
    if (fs_1.default.existsSync(bgPath)) {
        doc.image(bgPath, 0, 0, { width: 243, height: 153 });
    }
    const { nin = '', surname = '', firstname = '', middlename = '', gender = '', birthdate = '', photo = '' } = details || {};
    try {
        if (photo) {
            if (photo.startsWith('data:image')) {
                const base64 = photo.split(',')[1] || '';
                const buf = Buffer.from(base64, 'base64');
                doc.image(buf, 9, 50, { width: 80, height: 95 });
            }
            else if (fs_1.default.existsSync(photo)) {
                doc.image(photo, 9, 50, { width: 80, height: 95 });
            }
        }
    }
    catch (error) {
        console.warn('Failed to add photo to PDF:', error);
    }
    const contentLeft = 110;
    const contentTop = 25;
    doc.fillColor('#000');
    doc.fontSize(10).text(String(surname || '').toUpperCase(), contentLeft - 9, contentTop + 42, { width: 120, ellipsis: true });
    const firstMiddle = middlename ? `${String(firstname || '').toUpperCase()}, ${String(middlename || '').toUpperCase()}` : String(firstname || '').toUpperCase();
    doc.fontSize(10).text(firstMiddle, contentLeft - 9, contentTop + 70, { width: 120, ellipsis: true });
    doc.fontSize(10).text(String(birthdate || ''), contentLeft - 9, contentTop + 82, { width: 120, ellipsis: true });
    doc.fontSize(10).text(String(gender || ''), 85, 100, { width: 60, ellipsis: true });
    const ninRaw = String(nin || '');
    if (ninRaw) {
        doc.save().fillColor('#e3dcdc').fontSize(7).rotate(330, { origin: [0, 0] }).text(ninRaw, -95, 107).restore();
        doc.save().fillColor('#a6a1a150').fontSize(7).rotate(180, { origin: [0, 0] }).text(ninRaw, -133, -75).restore();
    }
    const tight = ninRaw.replace(/\s+/g, '');
    const pretty = tight ? `${tight.slice(0, 4)}  ${tight.slice(4, 7)}  ${tight.slice(7, 11)}` : '';
    if (pretty) {
        doc.fontSize(20).fillColor('#131111');
        const textWidth = doc.widthOfString(pretty);
        const centerX = 243 / 2 - textWidth / 2;
        doc.text(pretty, centerX, 153 - 24);
    }
    const now = new Date().toLocaleDateString('en-GB', { day: '2-digit', month: 'short', year: 'numeric' }).replaceAll('/', ' ');
    doc.fontSize(9).fillColor('#000');
    const dateWidth = doc.widthOfString(now);
    doc.text(now, 243 - 18 - dateWidth, 153 - 60);
}
function writeStandardSlip(doc, details) {
    const bgPath = path_1.default.join(process.cwd(), 'public', 'img', 'standard.png');
    if (fs_1.default.existsSync(bgPath)) {
        doc.image(bgPath, 0, 0, { width: 243, height: 153 });
    }
    const { nin = '', surname = '', firstname = '', middlename = '', gender = '', birthdate = '', photo = '' } = details || {};
    try {
        if (photo) {
            if (photo.startsWith('data:image')) {
                const base64 = photo.split(',')[1] || '';
                const buf = Buffer.from(base64, 'base64');
                doc.image(buf, 9, 50, { width: 80, height: 95 });
            }
            else if (fs_1.default.existsSync(photo)) {
                doc.image(photo, 9, 50, { width: 80, height: 95 });
            }
        }
    }
    catch (error) {
        console.warn('Failed to add photo to PDF:', error);
    }
    const contentLeft = 110;
    const contentTop = 25;
    doc.fillColor('#000');
    doc.fontSize(9).text(String(surname || '').toUpperCase(), contentLeft - 9, contentTop + 40, { width: 120, ellipsis: true });
    const firstMiddle = middlename ? `${String(firstname || '').toUpperCase()}, ${String(middlename || '').toUpperCase()}` : String(firstname || '').toUpperCase();
    doc.fontSize(9).text(firstMiddle, contentLeft - 9, contentTop + 65, { width: 120, ellipsis: true });
    doc.fontSize(9).text(String(birthdate || ''), contentLeft - 9, contentTop + 80, { width: 120, ellipsis: true });
    doc.fontSize(9).text(String(gender || ''), 85, 95, { width: 60, ellipsis: true });
    const ninRaw = String(nin || '');
    const tight = ninRaw.replace(/\s+/g, '');
    const pretty = tight ? `${tight.slice(0, 4)}  ${tight.slice(4, 7)}  ${tight.slice(7, 11)}` : '';
    if (pretty) {
        doc.fontSize(18).fillColor('#131111');
        const textWidth = doc.widthOfString(pretty);
        const centerX = 243 / 2 - textWidth / 2;
        doc.text(pretty, centerX, 153 - 20);
    }
    const now = new Date().toLocaleDateString('en-GB', { day: '2-digit', month: 'short', year: 'numeric' }).replaceAll('/', ' ');
    doc.fontSize(8).fillColor('#000');
    const dateWidth = doc.widthOfString(now);
    doc.text(now, 243 - 15 - dateWidth, 153 - 50);
}
function writeRegularSlip(doc, details) {
    const bgPath = path_1.default.join(process.cwd(), 'public', 'img', 'regular.png');
    if (fs_1.default.existsSync(bgPath)) {
        doc.image(bgPath, 0, 0, { width: 892, height: 500 });
    }
    const { nin = '', surname = '', firstname = '', middlename = '', gender = '', birthdate = '', photo = '', trackingId = '' } = details || {};
    doc.font('Helvetica-Bold').fontSize(25).fillColor('#000').text('National Identity Management System', 214, 32);
    doc.font('Helvetica-Bold').fontSize(16).text('Federal Republic of Nigeria', 330, 65);
    doc.font('Helvetica-Bold').fontSize(13).text('National Identification Number Slip (NINS)', 308, 89);
    doc.font('Helvetica-Bold').fontSize(12).text('Tracking ID:', 36, 134);
    doc.font('Helvetica').fontSize(11).text(String(trackingId || ''), 132, 135);
    doc.font('Helvetica-Bold').fontSize(12).text('NIN:', 36, 179);
    doc.font('Helvetica').fontSize(12).text(String(nin || ''), 132, 180);
    doc.font('Helvetica-Bold').fontSize(12).text('Surname:', 36, 224);
    doc.font('Helvetica').fontSize(12).text(String(surname || '').toUpperCase(), 132, 225);
    doc.font('Helvetica-Bold').fontSize(12).text('First Name:', 36, 269);
    doc.font('Helvetica').fontSize(12).text(String(firstname || '').toUpperCase(), 132, 270);
    if (middlename) {
        doc.font('Helvetica-Bold').fontSize(12).text('Middle Name:', 36, 314);
        doc.font('Helvetica').fontSize(12).text(String(middlename || '').toUpperCase(), 132, 315);
    }
    doc.font('Helvetica-Bold').fontSize(12).text('Gender:', 36, 359);
    doc.font('Helvetica').fontSize(12).text(String(gender || ''), 132, 360);
    doc.font('Helvetica-Bold').fontSize(12).text('Date of Birth:', 36, 404);
    doc.font('Helvetica').fontSize(12).text(String(birthdate || ''), 132, 405);
    if (photo) {
        try {
            if (photo.startsWith('data:image')) {
                const base64 = photo.split(',')[1] || '';
                const buf = Buffer.from(base64, 'base64');
                doc.image(buf, 600, 150, { width: 120, height: 150 });
            }
        }
        catch (error) {
            console.warn('Failed to add photo to regular slip:', error);
        }
    }
    const now = new Date().toLocaleDateString('en-GB', { day: '2-digit', month: 'short', year: 'numeric' }).replaceAll('/', ' ');
    doc.font('Helvetica').fontSize(10).text(`Issued: ${now}`, 36, 450);
}
class NinSlipTestController {
    static async getTestPremiumSlip(req, res) {
        try {
            res.setHeader('Content-Type', 'application/pdf');
            res.setHeader('Content-Disposition', 'attachment; filename="nin-test-premium-slip.pdf"');
            res.setHeader('Cache-Control', 'no-store');
            const doc = new pdfkit_1.default({ size: [243, 153], margin: 0 });
            doc.pipe(res);
            writePremiumSlip(doc, { ...MOCK_NIN_DATA, photo: MOCK_PHOTO_BASE64 });
            const backPath = path_1.default.join(process.cwd(), 'public', 'img', 'back.png');
            if (fs_1.default.existsSync(backPath)) {
                doc.addPage({ size: [243, 153], margin: 0 });
                doc.image(backPath, 0, 0, { width: 243, height: 153 });
            }
            doc.end();
        }
        catch (err) {
            console.error('[NinSlipTestController] Failed to generate test premium slip:', err);
            res.status(500).json({ success: false, message: 'Failed to generate test premium slip' });
        }
    }
    static async getTestStandardSlip(req, res) {
        try {
            res.setHeader('Content-Type', 'application/pdf');
            res.setHeader('Content-Disposition', 'attachment; filename="nin-test-standard-slip.pdf"');
            res.setHeader('Cache-Control', 'no-store');
            const doc = new pdfkit_1.default({ size: [243, 153], margin: 0 });
            doc.pipe(res);
            writeStandardSlip(doc, { ...MOCK_NIN_DATA, photo: MOCK_PHOTO_BASE64 });
            const backPath = path_1.default.join(process.cwd(), 'public', 'img', 'back.png');
            if (fs_1.default.existsSync(backPath)) {
                doc.addPage({ size: [243, 153], margin: 0 });
                doc.image(backPath, 0, 0, { width: 243, height: 153 });
            }
            doc.end();
        }
        catch (err) {
            console.error('[NinSlipTestController] Failed to generate test standard slip:', err);
            res.status(500).json({ success: false, message: 'Failed to generate test standard slip' });
        }
    }
    static async getTestRegularSlip(req, res) {
        try {
            res.setHeader('Content-Type', 'application/pdf');
            res.setHeader('Content-Disposition', 'attachment; filename="nin-test-regular-slip.pdf"');
            res.setHeader('Cache-Control', 'no-store');
            const doc = new pdfkit_1.default({ size: [892, 500], margin: 0 });
            doc.pipe(res);
            writeRegularSlip(doc, { ...MOCK_NIN_DATA, photo: MOCK_PHOTO_BASE64 });
            doc.end();
        }
        catch (err) {
            console.error('[NinSlipTestController] Failed to generate test regular slip:', err);
            res.status(500).json({ success: false, message: 'Failed to generate test regular slip' });
        }
    }
    static async getTestData(req, res) {
        try {
            res.json({
                success: true,
                data: MOCK_NIN_DATA,
                message: 'Mock NIN data for testing'
            });
        }
        catch (err) {
            console.error('[NinSlipTestController] Failed to get test data:', err);
            res.status(500).json({ success: false, message: 'Failed to get test data' });
        }
    }
    static async getCustomTestSlip(req, res) {
        try {
            const { type = 'premium', data = {} } = req.body;
            const slipData = { ...MOCK_NIN_DATA, ...data };
            res.setHeader('Content-Type', 'application/pdf');
            res.setHeader('Content-Disposition', `attachment; filename="nin-test-${type}-slip.pdf"`);
            res.setHeader('Cache-Control', 'no-store');
            let doc;
            if (type === 'regular') {
                doc = new pdfkit_1.default({ size: [892, 500], margin: 0 });
            }
            else {
                doc = new pdfkit_1.default({ size: [243, 153], margin: 0 });
            }
            doc.pipe(res);
            switch (type) {
                case 'premium':
                    writePremiumSlip(doc, slipData);
                    break;
                case 'standard':
                    writeStandardSlip(doc, slipData);
                    break;
                case 'regular':
                    writeRegularSlip(doc, slipData);
                    break;
                default:
                    writePremiumSlip(doc, slipData);
            }
            if (type !== 'regular') {
                const backPath = path_1.default.join(process.cwd(), 'public', 'img', 'back.png');
                if (fs_1.default.existsSync(backPath)) {
                    doc.addPage({ size: [243, 153], margin: 0 });
                    doc.image(backPath, 0, 0, { width: 243, height: 153 });
                }
            }
            doc.end();
        }
        catch (err) {
            console.error('[NinSlipTestController] Failed to generate custom test slip:', err);
            res.status(500).json({ success: false, message: 'Failed to generate custom test slip' });
        }
    }
}
exports.NinSlipTestController = NinSlipTestController;
