"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.userService = exports.UserService = void 0;
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const env_1 = require("../../config/env");
const user_repo_1 = require("./user.repo");
class UserService {
    hashPassword(password) {
        const salt = bcryptjs_1.default.genSaltSync(env_1.env.BCRYPT_ROUNDS);
        return bcryptjs_1.default.hashSync(password, salt);
    }
    verifyPassword(password, hash) {
        return bcryptjs_1.default.compareSync(password, hash);
    }
    async createUser(input) {
        const passwordHash = this.hashPassword(input.password);
        const user = await user_repo_1.userRepository.create({
            email: input.email,
            phone: input.phone,
            firstName: input.firstName,
            lastName: input.lastName,
            role: input.role || 'user',
            passwordHash,
        });
        return user;
    }
    async getUserById(userId) {
        return user_repo_1.userRepository.findById(userId);
    }
    async listUsers(params = {}) {
        return user_repo_1.userRepository.list({}, params);
    }
    async updateUser(userId, input) {
        return user_repo_1.userRepository.updateById(userId, { $set: input });
    }
    async deleteUser(userId) {
        await user_repo_1.userRepository.deleteById(userId);
    }
    async changePasswordSelf(userId, currentPassword, newPassword) {
        const user = await user_repo_1.userRepository.findById(userId);
        if (!user)
            throw new Error('User not found');
        const ok = this.verifyPassword(currentPassword, user.passwordHash);
        if (!ok)
            throw new Error('Current password is incorrect');
        user.passwordHash = this.hashPassword(newPassword);
        await user.save();
    }
    async changePasswordAdmin(userId, newPassword) {
        const user = await user_repo_1.userRepository.findById(userId);
        if (!user)
            throw new Error('User not found');
        user.passwordHash = this.hashPassword(newPassword);
        await user.save();
    }
    async changeRole(userId, role) {
        return user_repo_1.userRepository.updateById(userId, { $set: { role } });
    }
    async setSuspended(userId, isSuspended) {
        return user_repo_1.userRepository.updateById(userId, { $set: { isSuspended } });
    }
}
exports.UserService = UserService;
exports.userService = new UserService();
