"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const user_controller_1 = require("./user.controller");
const auth_1 = require("../../core/middleware/auth");
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const user_model_1 = require("./user.model");
const router = (0, express_1.Router)();
router.post("/", auth_1.requireAuth, (0, auth_1.requireRole)("admin"), user_controller_1.userController.create);
router.get("/", auth_1.requireAuth, (0, auth_1.requireRole)("admin"), user_controller_1.userController.list);
router.get("/:userId", auth_1.requireAuth, (0, auth_1.requireRole)("admin"), user_controller_1.userController.get);
router.patch("/:userId", auth_1.requireAuth, (0, auth_1.requireRole)("admin"), user_controller_1.userController.update);
router.delete("/:userId", auth_1.requireAuth, (0, auth_1.requireRole)("admin"), user_controller_1.userController.remove);
router.post("/:userId/role", auth_1.requireAuth, (0, auth_1.requireRole)("admin"), user_controller_1.userController.changeRole);
router.post("/:userId/suspend", auth_1.requireAuth, (0, auth_1.requireRole)("admin"), user_controller_1.userController.suspend);
router.post("/me/change-password", auth_1.requireAuth, user_controller_1.userController.changePasswordSelf);
router.post('/me/pin/set', auth_1.requireAuth, async (req, res) => {
    try {
        const { pin } = req.body || {};
        if (!pin || String(pin).length < 4) {
            res.status(400).json({ success: false, message: 'PIN must be at least 4 digits' });
            return;
        }
        const hash = await bcryptjs_1.default.hash(String(pin), 12);
        await user_model_1.User.updateOne({ _id: req.user.userId }, { $set: { transactionPinHash: hash } });
        res.status(200).json({ success: true, message: 'Transaction PIN set' });
    }
    catch (e) {
        res.status(400).json({ success: false, message: e.message || 'Failed to set PIN' });
    }
});
router.post('/me/pin/verify', auth_1.requireAuth, async (req, res) => {
    try {
        const { pin } = req.body || {};
        const user = await user_model_1.User.findById(req.user.userId).select('+transactionPinHash');
        if (!user || !user['transactionPinHash']) {
            res.status(400).json({ success: false, message: 'PIN not set' });
            return;
        }
        const ok = await bcryptjs_1.default.compare(String(pin), String(user['transactionPinHash']));
        res.status(200).json({ success: ok, message: ok ? 'PIN verified' : 'Invalid PIN' });
    }
    catch (e) {
        res.status(400).json({ success: false, message: e.message || 'Failed to verify PIN' });
    }
});
exports.default = router;
