"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.userController = void 0;
const user_service_1 = require("./user.service");
const user_repo_1 = require("./user.repo");
const user_model_1 = require("./user.model");
const user_schemas_1 = require("./user.schemas");
class UserController {
    async create(req, res) {
        const parsed = user_schemas_1.createUserSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        try {
            const user = await user_service_1.userService.createUser(parsed.data);
            res.status(201).json({ user: user.toJSON() });
        }
        catch (e) {
            res.status(400).json({ error: e.message || 'Create failed' });
        }
    }
    async list(req, res) {
        try {
            const { page = 1, limit = 50, search = '', role = 'all', status = 'all', sortBy = 'createdAt', sortOrder = 'desc' } = req.query;
            const searchParams = {};
            if (search && search !== '') {
                searchParams.$or = [
                    { email: { $regex: search, $options: 'i' } },
                    { phone: { $regex: search, $options: 'i' } },
                    { firstName: { $regex: search, $options: 'i' } },
                    { lastName: { $regex: search, $options: 'i' } }
                ];
            }
            if (role && role !== 'all') {
                searchParams.role = role;
            }
            if (status && status !== 'all') {
                if (status === 'active') {
                    searchParams.isSuspended = false;
                }
                else if (status === 'suspended') {
                    searchParams.isSuspended = true;
                }
            }
            const sort = {};
            sort[sortBy] = sortOrder === 'desc' ? -1 : 1;
            const skip = (Number(page) - 1) * Number(limit);
            const [users, total] = await Promise.all([
                user_repo_1.userRepository.list(searchParams, { limit: Number(limit), skip }),
                user_model_1.User.countDocuments({ ...searchParams, isDeleted: { $ne: true } })
            ]);
            res.json({
                success: true,
                data: users.map(user => user.toJSON()),
                pagination: {
                    page: Number(page),
                    limit: Number(limit),
                    total,
                    pages: Math.ceil(total / Number(limit))
                }
            });
        }
        catch (error) {
            res.status(400).json({ error: error.message || 'Failed to get users' });
        }
    }
    async get(req, res) {
        const userId = req.params['userId'];
        if (!userId) {
            res.status(400).json({ error: 'User ID is required' });
            return;
        }
        const user = await user_service_1.userService.getUserById(userId);
        if (!user) {
            res.status(404).json({ error: 'User not found' });
            return;
        }
        res.status(200).json({ user: user.toJSON() });
    }
    async update(req, res) {
        const parsed = user_schemas_1.updateUserSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const userId = req.params['userId'];
        if (!userId) {
            res.status(400).json({ error: 'User ID is required' });
            return;
        }
        const user = await user_service_1.userService.updateUser(userId, parsed.data);
        if (!user) {
            res.status(404).json({ error: 'User not found' });
            return;
        }
        res.status(200).json({ user: user.toJSON() });
    }
    async remove(req, res) {
        const userId = req.params['userId'];
        if (!userId) {
            res.status(400).json({ error: 'User ID is required' });
            return;
        }
        await user_service_1.userService.deleteUser(userId);
        res.status(204).send();
    }
    async changePasswordSelf(req, res) {
        const parsed = user_schemas_1.changePasswordSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const userId = req.user?.userId;
        if (!userId) {
            res.status(401).json({ error: 'Unauthorized' });
            return;
        }
        try {
            await user_service_1.userService.changePasswordSelf(userId, parsed.data.currentPassword, parsed.data.newPassword);
            res.status(204).send();
        }
        catch (e) {
            res.status(400).json({ error: e.message || 'Change password failed' });
        }
    }
    async adminChangePassword(req, res) {
        const parsed = user_schemas_1.adminChangePasswordSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const userId = req.params['userId'];
        if (!userId) {
            res.status(400).json({ error: 'User ID is required' });
            return;
        }
        await user_service_1.userService.changePasswordAdmin(userId, parsed.data.newPassword);
        res.status(204).send();
    }
    async changeRole(req, res) {
        const parsed = user_schemas_1.changeRoleSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const userId = req.params['userId'];
        if (!userId) {
            res.status(400).json({ error: 'User ID is required' });
            return;
        }
        const user = await user_service_1.userService.changeRole(userId, parsed.data.role);
        if (!user) {
            res.status(404).json({ error: 'User not found' });
            return;
        }
        res.status(200).json({ user: user.toJSON() });
    }
    async suspend(req, res) {
        const parsed = user_schemas_1.suspendUserSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const userId = req.params['userId'];
        if (!userId) {
            res.status(400).json({ error: 'User ID is required' });
            return;
        }
        const user = await user_service_1.userService.setSuspended(userId, parsed.data.isSuspended);
        if (!user) {
            res.status(404).json({ error: 'User not found' });
            return;
        }
        res.status(200).json({ user: user.toJSON() });
    }
}
exports.userController = new UserController();
