"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const auth_1 = require("../../core/middleware/auth");
const transactions_1 = require("../../database/repositories/transactions");
const router = (0, express_1.Router)();
router.get('/', auth_1.requireAuth, async (req, res) => {
    try {
        const { page = 1, limit = 50, search = '', status = 'all', type = 'all', category = 'all', sortBy = 'createdAt', sortOrder = 'desc' } = req.query;
        const searchParams = {};
        if (search && search !== '') {
            searchParams.$or = [
                { reference: { $regex: search, $options: 'i' } },
                { externalReference: { $regex: search, $options: 'i' } },
                { description: { $regex: search, $options: 'i' } }
            ];
        }
        if (status && status !== 'all') {
            searchParams.status = status;
        }
        if (type && type !== 'all') {
            searchParams.type = type;
        }
        if (category && category !== 'all') {
            searchParams.category = category;
        }
        const sort = {};
        sort[sortBy] = sortOrder === 'desc' ? -1 : 1;
        const result = await transactions_1.transactionRepository.searchTransactions(searchParams, Number(page), Number(limit));
        res.json({
            success: true,
            data: result.data,
            pagination: result.pagination
        });
    }
    catch (error) {
        res.status(400).json({ error: error.message || 'Failed to get transactions' });
    }
});
router.get('/search', auth_1.requireAuth, async (req, res) => {
    try {
        const { page = 1, limit = 20, userId, walletId, type, category, status, currency, dateFrom, dateTo, minAmount, maxAmount, reference, externalReference, journalEntryId, isReversal } = req.query;
        const searchParams = {
            userId: userId,
            walletId: walletId,
            type: type,
            category: category,
            status: status,
            currency: currency,
            ...(dateFrom && { dateFrom: new Date(dateFrom) }),
            ...(dateTo && { dateTo: new Date(dateTo) }),
            ...(minAmount && { minAmount: Number(minAmount) }),
            ...(maxAmount && { maxAmount: Number(maxAmount) }),
            ...(reference && { reference: reference }),
            ...(externalReference && { externalReference: externalReference }),
            ...(journalEntryId && { journalEntryId: journalEntryId }),
            ...(isReversal !== undefined && { isReversal: isReversal === 'true' })
        };
        const result = await transactions_1.transactionRepository.searchTransactions(searchParams, Number(page), Number(limit));
        res.json({
            success: true,
            data: result.data,
            pagination: result.pagination
        });
    }
    catch (error) {
        res.status(400).json({ error: error.message || 'Failed to search transactions' });
    }
});
router.get('/stats', auth_1.requireAuth, async (req, res) => {
    try {
        const { userId, walletId, type, category, status, currency, dateFrom, dateTo, minAmount, maxAmount } = req.query;
        const searchParams = {
            userId: userId,
            walletId: walletId,
            type: type,
            category: category,
            status: status,
            currency: currency,
            ...(dateFrom && { dateFrom: new Date(dateFrom) }),
            ...(dateTo && { dateTo: new Date(dateTo) }),
            ...(minAmount && { minAmount: Number(minAmount) }),
            ...(maxAmount && { maxAmount: Number(maxAmount) })
        };
        const stats = await transactions_1.transactionRepository.getTransactionStats(searchParams);
        res.json({ success: true, data: stats });
    }
    catch (error) {
        res.status(400).json({ error: error.message || 'Failed to get transaction stats' });
    }
});
router.get('/export', auth_1.requireAuth, async (req, res) => {
    try {
        const { format = 'json', includeMetadata = 'false', userId, walletId, type, category, status, currency, dateFrom, dateTo, minAmount, maxAmount, reference, externalReference, journalEntryId, isReversal } = req.query;
        const exportParams = {
            format: format,
            includeMetadata: includeMetadata === 'true',
            userId: userId,
            walletId: walletId,
            type: type,
            category: category,
            status: status,
            currency: currency,
            ...(dateFrom && { dateFrom: new Date(dateFrom) }),
            ...(dateTo && { dateTo: new Date(dateTo) }),
            ...(minAmount && { minAmount: Number(minAmount) }),
            ...(maxAmount && { maxAmount: Number(maxAmount) }),
            ...(reference && { reference: reference }),
            ...(externalReference && { externalReference: externalReference }),
            ...(journalEntryId && { journalEntryId: journalEntryId }),
            ...(isReversal !== undefined && { isReversal: isReversal === 'true' })
        };
        const exportData = await transactions_1.transactionRepository.exportTransactions(exportParams);
        if (exportParams.format === 'csv') {
            res.setHeader('Content-Type', 'text/csv');
            res.setHeader('Content-Disposition', `attachment; filename="transactions-${Date.now()}.csv"`);
        }
        else {
            res.setHeader('Content-Type', 'application/json');
            res.setHeader('Content-Disposition', `attachment; filename="transactions-${Date.now()}.json"`);
        }
        res.send(exportData);
    }
    catch (error) {
        res.status(400).json({ error: error.message || 'Failed to export transactions' });
    }
});
router.get('/journal/:journalEntryId', auth_1.requireAuth, async (req, res) => {
    try {
        const { journalEntryId } = req.params;
        const transactions = await transactions_1.transactionRepository.getJournalEntry(journalEntryId);
        res.json({
            success: true,
            data: {
                journalEntryId,
                transactions,
                count: transactions.length
            }
        });
    }
    catch (error) {
        res.status(400).json({ error: error.message || 'Failed to get journal entry' });
    }
});
router.get('/:transactionId/related', auth_1.requireAuth, async (req, res) => {
    try {
        const { transactionId } = req.params;
        const transactions = await transactions_1.transactionRepository.getRelatedTransactions(transactionId);
        res.json({
            success: true,
            data: {
                originalTransactionId: transactionId,
                relatedTransactions: transactions,
                count: transactions.length
            }
        });
    }
    catch (error) {
        res.status(400).json({ error: error.message || 'Failed to get related transactions' });
    }
});
exports.default = router;
