"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.queueController = exports.QueueController = void 0;
const queue_1 = require("../../config/queue");
const job_scheduler_1 = require("../../jobs/schedulers/job.scheduler");
const worker_manager_1 = require("../../jobs/workers/worker.manager");
const logger_1 = require("../../config/logger");
class QueueController {
    async getQueueStats(_req, res) {
        try {
            const stats = await queue_1.queueManager.getAllQueueStats();
            res.json({
                success: true,
                data: stats
            });
        }
        catch (error) {
            logger_1.logger.error('Failed to get queue stats:', error);
            res.status(500).json({
                error: error.message || 'Failed to get queue statistics'
            });
        }
    }
    async getQueueStat(req, res) {
        try {
            const { queueName } = req.params;
            const stats = await queue_1.queueManager.getQueueStats(queueName);
            if (!stats) {
                res.status(404).json({
                    error: `Queue not found: ${queueName}`
                });
                return;
            }
            res.json({
                success: true,
                data: stats
            });
        }
        catch (error) {
            logger_1.logger.error(`Failed to get queue stats for ${req.params['queueName']}:`, error);
            res.status(500).json({
                error: error.message || 'Failed to get queue statistics'
            });
        }
    }
    async getJobSchedulerStatus(_req, res) {
        try {
            const status = job_scheduler_1.jobScheduler.getStatus();
            const jobStatuses = job_scheduler_1.jobScheduler.getAllJobStatuses();
            res.json({
                success: true,
                data: {
                    scheduler: status,
                    jobs: jobStatuses
                }
            });
        }
        catch (error) {
            logger_1.logger.error('Failed to get job scheduler status:', error);
            res.status(500).json({
                error: error.message || 'Failed to get job scheduler status'
            });
        }
    }
    async getWorkerManagerStatus(_req, res) {
        try {
            const status = worker_manager_1.workerManager.getStatus();
            const workerStatuses = worker_manager_1.workerManager.getAllWorkerStatuses();
            res.json({
                success: true,
                data: {
                    manager: status,
                    workers: workerStatuses
                }
            });
        }
        catch (error) {
            logger_1.logger.error('Failed to get worker manager status:', error);
            res.status(500).json({
                error: error.message || 'Failed to get worker manager status'
            });
        }
    }
    async toggleJob(req, res) {
        try {
            const { jobId } = req.params;
            const { enabled } = req.body;
            if (typeof enabled !== 'boolean') {
                res.status(400).json({
                    error: 'enabled field must be a boolean'
                });
                return;
            }
            await job_scheduler_1.jobScheduler.toggleJob(jobId, enabled);
            res.json({
                success: true,
                message: `Job ${jobId} ${enabled ? 'enabled' : 'disabled'} successfully`
            });
        }
        catch (error) {
            logger_1.logger.error(`Failed to toggle job ${req.params['jobId']}:`, error);
            res.status(500).json({
                error: error.message || 'Failed to toggle job'
            });
        }
    }
    async updateJobSchedule(req, res) {
        try {
            const { jobId } = req.params;
            const { pattern } = req.body;
            if (!pattern || typeof pattern !== 'string') {
                res.status(400).json({
                    error: 'pattern field is required and must be a string'
                });
                return;
            }
            await job_scheduler_1.jobScheduler.updateJobSchedule(jobId, pattern);
            res.json({
                success: true,
                message: `Job ${jobId} schedule updated successfully`
            });
        }
        catch (error) {
            logger_1.logger.error(`Failed to update job schedule for ${req.params['jobId']}:`, error);
            res.status(500).json({
                error: error.message || 'Failed to update job schedule'
            });
        }
    }
    async triggerJob(req, res) {
        try {
            const { jobId } = req.params;
            const { data } = req.body;
            await job_scheduler_1.jobScheduler.triggerJob(jobId, data);
            res.json({
                success: true,
                message: `Job ${jobId} triggered successfully`
            });
        }
        catch (error) {
            logger_1.logger.error(`Failed to trigger job ${req.params['jobId']}:`, error);
            res.status(500).json({
                error: error.message || 'Failed to trigger job'
            });
        }
    }
    async pauseWorker(req, res) {
        try {
            const { queueName } = req.params;
            worker_manager_1.workerManager.pauseWorker(queueName);
            res.json({
                success: true,
                message: `Worker for queue ${queueName} paused successfully`
            });
        }
        catch (error) {
            logger_1.logger.error(`Failed to pause worker for queue ${req.params['queueName']}:`, error);
            res.status(500).json({
                error: error.message || 'Failed to pause worker'
            });
        }
    }
    async resumeWorker(req, res) {
        try {
            const { queueName } = req.params;
            worker_manager_1.workerManager.resumeWorker(queueName);
            res.json({
                success: true,
                message: `Worker for queue ${queueName} resumed successfully`
            });
        }
        catch (error) {
            logger_1.logger.error(`Failed to resume worker for queue ${req.params['queueName']}:`, error);
            res.status(500).json({
                error: error.message || 'Failed to resume worker'
            });
        }
    }
    async pauseAllWorkers(_req, res) {
        try {
            worker_manager_1.workerManager.pauseAllWorkers();
            res.json({
                success: true,
                message: 'All workers paused successfully'
            });
        }
        catch (error) {
            logger_1.logger.error('Failed to pause all workers:', error);
            res.status(500).json({
                error: error.message || 'Failed to pause all workers'
            });
        }
    }
    async resumeAllWorkers(_req, res) {
        try {
            worker_manager_1.workerManager.resumeAllWorkers();
            res.json({
                success: true,
                message: 'All workers resumed successfully'
            });
        }
        catch (error) {
            logger_1.logger.error('Failed to resume all workers:', error);
            res.status(500).json({
                error: error.message || 'Failed to resume all workers'
            });
        }
    }
    async cleanupOldJobs(req, res) {
        try {
            const { queueName } = req.params;
            const { olderThan = 24 * 60 * 60 * 1000 } = req.body;
            const cleanedCount = await queue_1.queueManager.cleanupOldJobs(queueName, olderThan);
            res.json({
                success: true,
                message: `Cleaned up ${cleanedCount} old jobs from queue ${queueName}`,
                data: { cleanedCount }
            });
        }
        catch (error) {
            logger_1.logger.error(`Failed to cleanup old jobs for queue ${req.params['queueName']}:`, error);
            res.status(500).json({
                error: error.message || 'Failed to cleanup old jobs'
            });
        }
    }
    async getSystemStatus(_req, res) {
        try {
            const [queueStats, schedulerStatus, workerStatus] = await Promise.all([
                queue_1.queueManager.getAllQueueStats(),
                Promise.resolve(job_scheduler_1.jobScheduler.getStatus()),
                Promise.resolve(worker_manager_1.workerManager.getStatus())
            ]);
            res.json({
                success: true,
                data: {
                    queues: queueStats,
                    scheduler: schedulerStatus,
                    workers: workerStatus,
                    timestamp: new Date().toISOString()
                }
            });
        }
        catch (error) {
            logger_1.logger.error('Failed to get system status:', error);
            res.status(500).json({
                error: error.message || 'Failed to get system status'
            });
        }
    }
}
exports.QueueController = QueueController;
exports.queueController = new QueueController();
