"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.paymentService = exports.PaymentService = void 0;
const transactions_1 = require("../../database/models/transactions");
const paystack_1 = require("../../integrations/paystack");
const flutterwave_1 = require("../../integrations/flutterwave");
const aspfiy_1 = require("../../integrations/aspfiy");
const redis_1 = require("../../config/redis");
const errors_1 = require("../../core/errors");
const crypto_1 = __importDefault(require("crypto"));
class PaymentService {
    static instance;
    constructor() { }
    static getInstance() {
        if (!PaymentService.instance) {
            PaymentService.instance = new PaymentService();
        }
        return PaymentService.instance;
    }
    async initializeCharge(request) {
        try {
            this.validateChargeRequest(request);
            const existingTransaction = await transactions_1.Transaction.findOne({ reference: request.reference });
            if (existingTransaction) {
                throw new errors_1.ValidationError('Payment reference already exists');
            }
            const reference = request.reference || this.generateReference(request.provider);
            let chargeResponse;
            if (request.provider === 'paystack') {
                const paystackRequest = {
                    amount: request.amount,
                    email: request.email,
                    reference,
                    callback_url: request.callbackUrl || '',
                    currency: (request.currency === 'EUR' || request.currency === 'GBP') ? 'NGN' : (request.currency || 'NGN'),
                    metadata: {
                        ...request.metadata,
                        paymentType: request.paymentType,
                        customerName: request.customerName,
                        customerPhone: request.customerPhone,
                    },
                };
                const response = await paystack_1.paystackAdapter.initializeCharge(paystackRequest);
                chargeResponse = {
                    success: response.status,
                    provider: 'paystack',
                    reference: response.data.reference,
                    authorizationUrl: response.data.authorization_url,
                    accessCode: response.data.access_code,
                    amount: response.data.amount / 100,
                    currency: response.data.currency,
                    metadata: response.data.metadata || {},
                };
            }
            else if (request.provider === 'flutterwave') {
                const flutterwaveRequest = {
                    tx_ref: reference,
                    amount: request.amount,
                    currency: request.currency || 'NGN',
                    redirect_url: request.callbackUrl || '',
                    customer: {
                        email: request.email,
                        phone_number: request.customerPhone || '',
                        name: request.customerName || '',
                    },
                    meta: {
                        ...request.metadata,
                        paymentType: request.paymentType,
                    },
                };
                const response = await flutterwave_1.flutterwaveAdapter.initializeCharge(flutterwaveRequest);
                chargeResponse = {
                    success: response.status === 'success',
                    provider: 'flutterwave',
                    reference: response.data.tx_ref,
                    authorizationUrl: response.data.link,
                    flwRef: response.data.flw_ref,
                    amount: response.data.amount,
                    currency: response.data.currency,
                    metadata: response.data.meta || {},
                };
            }
            else if (request.provider === 'aspfiy') {
                const aspfiyRequest = {
                    amount: request.amount,
                    email: request.email,
                    reference,
                    callback_url: request.callbackUrl || '',
                    currency: (request.currency === 'USD' || request.currency === 'EUR' || request.currency === 'GBP') ? 'NGN' : (request.currency || 'NGN'),
                    customer_name: request.customerName || '',
                    customer_phone: request.customerPhone || '',
                    metadata: {
                        ...request.metadata,
                        paymentType: request.paymentType,
                    },
                    payment_type: request.paymentType === 'bank_transfer' ? 'bank_transfer' : 'reserved_account',
                };
                const response = await aspfiy_1.aspfiyAdapter.initializeCharge(aspfiyRequest);
                chargeResponse = {
                    success: response.status,
                    provider: 'aspfiy',
                    reference: response.data.reference,
                    authorizationUrl: response.data.authorization_url || `https://checkout.aspfiy.com/${response.data.reference}`,
                    amount: response.data.amount,
                    currency: response.data.currency,
                    metadata: {
                        ...response.data.metadata,
                        account_number: response.data.account_number,
                        account_name: response.data.account_name,
                        bank_name: response.data.bank_name,
                        expires_at: response.data.expires_at,
                    },
                };
            }
            else {
                throw new errors_1.ValidationError('Invalid payment provider');
            }
            await this.storePaymentIntent(reference, {
                provider: request.provider,
                amount: request.amount,
                email: request.email,
                metadata: request.metadata,
                createdAt: new Date(),
            });
            return chargeResponse;
        }
        catch (error) {
            console.error('[PaymentService] Charge initialization failed:', error);
            throw error;
        }
    }
    async verifyPayment(request) {
        try {
            if (!request.reference || !request.provider) {
                throw new errors_1.ValidationError('Reference and provider are required');
            }
            let verificationResponse;
            if (request.provider === 'paystack') {
                const transaction = await paystack_1.paystackAdapter.verifyTransaction(request.reference);
                verificationResponse = {
                    success: transaction.status === 'success',
                    reference: transaction.reference || request.reference,
                    provider: 'paystack',
                    status: transaction.status,
                    amount: transaction.amount / 100,
                    currency: transaction.currency,
                    transactionId: transaction.id.toString(),
                };
            }
            else if (request.provider === 'flutterwave') {
                const transaction = await flutterwave_1.flutterwaveAdapter.verifyTransaction(request.reference);
                verificationResponse = {
                    success: transaction.status === 'successful',
                    reference: transaction.tx_ref,
                    provider: 'flutterwave',
                    status: transaction.status,
                    amount: transaction.amount,
                    currency: transaction.currency,
                    metadata: transaction.meta || {},
                    transactionId: transaction.id.toString(),
                };
            }
            else if (request.provider === 'aspfiy') {
                const transaction = await aspfiy_1.aspfiyAdapter.verifyTransaction(request.reference);
                verificationResponse = {
                    success: transaction.status === 'successful',
                    reference: transaction.reference,
                    provider: 'aspfiy',
                    status: transaction.status,
                    amount: transaction.amount,
                    currency: transaction.currency,
                    metadata: {
                        ...transaction.metadata,
                        account_number: transaction.account_number,
                        account_name: transaction.account_name,
                        bank_name: transaction.bank_name,
                        payment_type: transaction.payment_type,
                        fees: transaction.fees,
                        net_amount: transaction.net_amount,
                    },
                    transactionId: transaction.id,
                };
            }
            else {
                throw new errors_1.ValidationError('Invalid payment provider');
            }
            await this.updateTransactionStatus(request.reference, verificationResponse.status);
            return verificationResponse;
        }
        catch (error) {
            console.error('[PaymentService] Payment verification failed:', error);
            throw error;
        }
    }
    async processWebhook(request) {
        try {
            let isValid = false;
            if (request.provider === 'paystack') {
                isValid = paystack_1.paystackAdapter.verifyWebhookSignature(request.signature, request.rawBody);
            }
            else if (request.provider === 'flutterwave') {
                isValid = flutterwave_1.flutterwaveAdapter.verifyWebhookSignature(request.signature, request.rawBody);
            }
            else if (request.provider === 'aspfiy') {
                isValid = aspfiy_1.aspfiyAdapter.verifyWebhookSignature(request.signature, request.rawBody);
            }
            if (!isValid) {
                console.error(`[PaymentService] Invalid webhook signature for ${request.provider}`);
                throw new errors_1.AppError('Invalid webhook signature', 401);
            }
            let webhookResult;
            if (request.provider === 'paystack') {
                webhookResult = await paystack_1.paystackAdapter.processWebhookEvent(request.event);
            }
            else if (request.provider === 'flutterwave') {
                webhookResult = await flutterwave_1.flutterwaveAdapter.processWebhookEvent(request.event);
            }
            else if (request.provider === 'aspfiy') {
                webhookResult = await aspfiy_1.aspfiyAdapter.processWebhookEvent(request.event);
            }
            else {
                throw new errors_1.ValidationError('Invalid payment provider');
            }
            await this.updateTransactionFromWebhook(webhookResult.transactionId, webhookResult.status, webhookResult.amount, request.provider, webhookResult.metadata);
            if (request.provider === 'aspfiy' && webhookResult.status === 'successful') {
                try {
                    const md = webhookResult.metadata || {};
                    const accountNumber = md.account_number;
                    const email = md['customer_email'];
                    const reference = webhookResult.transactionId;
                    const amount = webhookResult.amount;
                    console.log('[PaymentService] Aspfiy webhook - attempting wallet credit:', {
                        accountNumber,
                        email,
                        reference,
                        amount,
                        metadata: md
                    });
                    let userId;
                    const { User } = await Promise.resolve().then(() => __importStar(require('../users/user.model')));
                    if (accountNumber) {
                        console.log('[PaymentService] Looking up user by virtual account:', accountNumber);
                        const userByVa = await User.findOne({ 'virtualAccount.accountNumber': accountNumber });
                        if (userByVa) {
                            userId = userByVa.id;
                            console.log('[PaymentService] Found user by virtual account:', userId);
                        }
                    }
                    if (!userId && email) {
                        console.log('[PaymentService] Looking up user by email:', email);
                        const userByEmail = await User.findOne({ email });
                        if (userByEmail) {
                            userId = userByEmail.id;
                            console.log('[PaymentService] Found user by email:', userId);
                        }
                    }
                    if (userId) {
                        console.log('[PaymentService] Crediting wallet for user:', userId);
                        try {
                            const { walletService } = await Promise.resolve().then(() => __importStar(require('../wallet/wallet.service')));
                            const creditResult = await walletService.credit({
                                userId,
                                currency: 'NGN',
                                amount,
                                reference,
                                category: 'funding',
                                description: `Aspfiy ${md['payment_type'] || 'payment'} - ${reference}`,
                                metadata: { provider: 'aspfiy', ...md },
                            });
                            console.log('[PaymentService] Wallet credited successfully for user:', userId);
                            console.log('[PaymentService] Credit result:', {
                                transactionId: creditResult.transaction.id,
                                newBalance: creditResult.wallet.availableBalance,
                                walletId: creditResult.wallet.id
                            });
                        }
                        catch (error) {
                            console.error('[PaymentService] Error during wallet credit:', {
                                error: error instanceof Error ? error.message : 'Unknown error',
                                stack: error instanceof Error ? error.stack : undefined,
                                userId,
                                accountNumber,
                                email,
                                reference,
                                amount
                            });
                            throw error;
                        }
                    }
                    else {
                        console.warn('[PaymentService] Aspfiy webhook: unable to map user for credit', {
                            accountNumber,
                            email,
                            reference,
                            availableMetadata: Object.keys(md)
                        });
                    }
                }
                catch (e) {
                    console.error('[PaymentService] Failed to credit wallet from Aspfiy webhook:', e);
                }
            }
            await this.clearPaymentIntent(webhookResult.transactionId);
            return {
                success: true,
                message: 'Webhook processed successfully',
                transactionId: webhookResult.transactionId,
            };
        }
        catch (error) {
            console.error('[PaymentService] Webhook processing failed:', error);
            throw error;
        }
    }
    async reconcileTransactions(request) {
        try {
            let providerTransactions = [];
            if (request.provider === 'paystack') {
                providerTransactions = await paystack_1.paystackAdapter.reconcileTransactions(request.startDate, request.endDate);
            }
            else if (request.provider === 'flutterwave') {
                providerTransactions = await flutterwave_1.flutterwaveAdapter.reconcileTransactions(request.startDate, request.endDate);
            }
            else if (request.provider === 'aspfiy') {
                providerTransactions = await aspfiy_1.aspfiyAdapter.reconcileTransactions(request.startDate, request.endDate);
            }
            else {
                throw new errors_1.ValidationError('Invalid payment provider');
            }
            const localTransactions = await transactions_1.Transaction.find({
                externalReference: {
                    $in: providerTransactions.map(t => t.reference || t.tx_ref || t.id)
                },
                createdAt: {
                    $gte: new Date(request.startDate),
                    $lte: new Date(request.endDate),
                },
            });
            const discrepancies = this.findDiscrepancies(providerTransactions, localTransactions);
            return {
                success: true,
                message: 'Reconciliation completed',
                reconciledCount: providerTransactions.length,
                discrepancies,
            };
        }
        catch (error) {
            console.error('[PaymentService] Transaction reconciliation failed:', error);
            throw error;
        }
    }
    async processRefund(request) {
        try {
            if (!request.reference || !request.provider) {
                throw new errors_1.ValidationError('Reference and provider are required');
            }
            let refundResult;
            if (request.provider === 'paystack') {
                refundResult = await paystack_1.paystackAdapter.refundTransaction(request.reference, request.amount);
            }
            else if (request.provider === 'flutterwave') {
                refundResult = await flutterwave_1.flutterwaveAdapter.refundTransaction(request.reference, request.amount);
            }
            else if (request.provider === 'aspfiy') {
                refundResult = await aspfiy_1.aspfiyAdapter.refundTransaction(request.reference, request.amount);
            }
            else {
                throw new errors_1.ValidationError('Invalid payment provider');
            }
            await this.createRefundTransaction(request.reference, refundResult.amount, request.provider, request.reason);
            return {
                success: refundResult.success,
                refundId: refundResult.refundId,
                amount: refundResult.amount,
                message: 'Refund processed successfully',
            };
        }
        catch (error) {
            console.error('[PaymentService] Refund processing failed:', error);
            throw error;
        }
    }
    async getPaymentStats(provider, startDate, endDate) {
        try {
            const filter = {};
            if (provider)
                filter.metadata = { provider };
            if (startDate && endDate) {
                filter.createdAt = {
                    $gte: new Date(startDate),
                    $lte: new Date(endDate),
                };
            }
            const transactions = await transactions_1.Transaction.find(filter);
            const totalTransactions = transactions.length;
            const successfulTransactions = transactions.filter(t => t.status === 'successful');
            const totalAmount = transactions.reduce((sum, t) => sum + t.amount, 0);
            const successRate = totalTransactions > 0 ? (successfulTransactions.length / totalTransactions) * 100 : 0;
            const averageAmount = totalTransactions > 0 ? totalAmount / totalTransactions : 0;
            const providerBreakdown = transactions.reduce((acc, t) => {
                const provider = t.metadata?.['provider'] || 'unknown';
                if (!acc[provider]) {
                    acc[provider] = { count: 0, amount: 0, successCount: 0 };
                }
                acc[provider].count++;
                acc[provider].amount += t.amount;
                if (t.status === 'successful')
                    acc[provider].successCount++;
                return acc;
            }, {});
            return {
                totalTransactions,
                totalAmount,
                successRate,
                averageAmount,
                providerBreakdown,
            };
        }
        catch (error) {
            console.error('[PaymentService] Failed to get payment stats:', error);
            throw error;
        }
    }
    validateChargeRequest(request) {
        if (!request.amount || request.amount <= 0) {
            throw new errors_1.ValidationError('Amount must be greater than 0');
        }
        if (!request.email) {
            throw new errors_1.ValidationError('Email is required');
        }
        if (!request.provider) {
            throw new errors_1.ValidationError('Payment provider is required');
        }
        if (request.amount > 1000000) {
            throw new errors_1.ValidationError('Amount exceeds maximum limit');
        }
    }
    generateReference(provider) {
        const timestamp = Date.now();
        const random = crypto_1.default.randomBytes(4).toString('hex');
        const prefix = provider === 'paystack' ? 'PSK' : 'FLW';
        return `${prefix}-${timestamp}-${random}`;
    }
    async storePaymentIntent(reference, data) {
        const key = `payment_intent:${reference}`;
        await redis_1.redisManager.setex(key, 3600, JSON.stringify(data));
    }
    async clearPaymentIntent(reference) {
        const key = `payment_intent:${reference}`;
        await redis_1.redisManager.getClient('cache').then(client => client.del(key));
    }
    async updateTransactionStatus(reference, status) {
        await transactions_1.Transaction.findOneAndUpdate({ reference }, {
            status,
            updatedAt: new Date(),
        });
    }
    async updateTransactionFromWebhook(reference, status, amount, provider, metadata) {
        if (provider === 'aspfiy') {
            await transactions_1.Transaction.updateOne({ reference }, {
                $set: {
                    status,
                    updatedAt: new Date(),
                },
            });
            return;
        }
        await transactions_1.Transaction.updateOne({ reference }, {
            $set: {
                status,
                updatedAt: new Date(),
            },
            $setOnInsert: {
                reference,
                amount,
                metadata: { ...metadata, provider },
            },
            $setOnInsert__noop: undefined
        }, { upsert: true });
    }
    async createRefundTransaction(originalReference, refundAmount, provider, reason) {
        const originalTransaction = await transactions_1.Transaction.findOne({ reference: originalReference });
        if (!originalTransaction) {
            throw new errors_1.ValidationError('Original transaction not found');
        }
        const refundReference = `REF-${originalReference}-${Date.now()}`;
        await transactions_1.Transaction.create({
            reference: refundReference,
            type: 'credit',
            category: 'refund',
            amount: refundAmount,
            currency: originalTransaction.currency,
            status: 'successful',
            description: `Refund for ${originalReference}${reason ? ` - ${reason}` : ''}`,
            metadata: {
                provider,
                originalReference,
                reason,
            },
            externalReference: refundReference,
        });
    }
    findDiscrepancies(providerTransactions, localTransactions) {
        const discrepancies = [];
        for (const providerTx of providerTransactions) {
            const providerRef = providerTx.reference || providerTx.tx_ref;
            const localTx = localTransactions.find(t => t.reference === providerRef);
            if (!localTx) {
                discrepancies.push({
                    type: 'missing_local',
                    providerReference: providerRef,
                    providerAmount: providerTx.amount,
                    providerStatus: providerTx.status,
                });
            }
            else {
                const providerAmount = providerTx.amount / 100 || providerTx.amount;
                if (Math.abs(providerAmount - localTx.amount) > 0.01) {
                    discrepancies.push({
                        type: 'amount_mismatch',
                        reference: providerRef,
                        providerAmount,
                        localAmount: localTx.amount,
                    });
                }
                if (providerTx.status !== localTx.status) {
                    discrepancies.push({
                        type: 'status_mismatch',
                        reference: providerRef,
                        providerStatus: providerTx.status,
                        localStatus: localTx.status,
                    });
                }
            }
        }
        return discrepancies;
    }
}
exports.PaymentService = PaymentService;
exports.paymentService = PaymentService.getInstance();
