"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.PaymentController = void 0;
const errors_1 = require("../../core/errors");
const payment_service_1 = require("./payment.service");
const transactions_1 = require("../../database/models/transactions");
const errors_2 = require("../../core/errors");
class PaymentController {
    static initializeCharge = (0, errors_1.asyncHandler)(async (req, res) => {
        const { amount, email, reference, provider, callbackUrl, currency, metadata, paymentType, customerName, customerPhone, } = req.body;
        if (!amount || !email || !provider) {
            throw new errors_2.ValidationError('Amount, email, and provider are required');
        }
        if (!['paystack', 'flutterwave'].includes(provider)) {
            throw new errors_2.ValidationError('Invalid payment provider. Must be paystack or flutterwave');
        }
        const chargeResponse = await payment_service_1.paymentService.initializeCharge({
            amount: Number(amount),
            email,
            reference,
            provider: provider,
            callbackUrl,
            currency,
            metadata,
            paymentType,
            customerName,
            customerPhone,
        });
        res.status(200).json({
            success: true,
            message: 'Payment charge initialized successfully',
            data: chargeResponse,
        });
    });
    static verifyPayment = (0, errors_1.asyncHandler)(async (req, res) => {
        const { reference, provider } = req.params;
        if (!reference || !provider) {
            throw new errors_2.ValidationError('Reference and provider are required');
        }
        if (!['paystack', 'flutterwave'].includes(provider)) {
            throw new errors_2.ValidationError('Invalid payment provider');
        }
        const verificationResponse = await payment_service_1.paymentService.verifyPayment({
            reference,
            provider: provider,
        });
        res.status(200).json({
            success: true,
            message: 'Payment verification completed',
            data: verificationResponse,
        });
    });
    static getTransaction = (0, errors_1.asyncHandler)(async (req, res) => {
        const { reference } = req.params;
        if (!reference) {
            throw new errors_2.ValidationError('Transaction reference is required');
        }
        const transaction = await transactions_1.Transaction.findOne({ reference });
        if (!transaction) {
            throw new errors_2.ValidationError('Transaction not found');
        }
        res.status(200).json({
            success: true,
            message: 'Transaction retrieved successfully',
            data: transaction,
        });
    });
    static getUserTransactions = (0, errors_1.asyncHandler)(async (req, res) => {
        const userId = req.user?.userId;
        const { page = 1, limit = 10, status, category, provider } = req.query;
        if (!userId) {
            throw new errors_2.ValidationError('User ID is required');
        }
        const filter = { userId };
        if (status)
            filter.status = status;
        if (category)
            filter.category = category;
        if (provider)
            filter['metadata.provider'] = provider;
        const skip = (Number(page) - 1) * Number(limit);
        const transactions = await transactions_1.Transaction.find(filter)
            .sort({ createdAt: -1 })
            .skip(skip)
            .limit(Number(limit))
            .populate('userId', 'email firstName lastName');
        const total = await transactions_1.Transaction.countDocuments(filter);
        res.status(200).json({
            success: true,
            message: 'User transactions retrieved successfully',
            data: {
                transactions,
                pagination: {
                    page: Number(page),
                    limit: Number(limit),
                    total,
                    pages: Math.ceil(total / Number(limit)),
                },
            },
        });
    });
    static processRefund = (0, errors_1.asyncHandler)(async (req, res) => {
        const { reference, provider } = req.params;
        const { amount, reason } = req.body;
        if (!reference || !provider) {
            throw new errors_2.ValidationError('Reference and provider are required');
        }
        if (!['paystack', 'flutterwave'].includes(provider)) {
            throw new errors_2.ValidationError('Invalid payment provider');
        }
        const refundResult = await payment_service_1.paymentService.processRefund({
            reference,
            provider: provider,
            amount: amount ? Number(amount) : 0,
            reason,
        });
        res.status(200).json({
            success: true,
            message: 'Refund processed successfully',
            data: refundResult,
        });
    });
    static reconcileTransactions = (0, errors_1.asyncHandler)(async (req, res) => {
        const { provider } = req.params;
        const { startDate, endDate } = req.query;
        if (!provider || !startDate || !endDate) {
            throw new errors_2.ValidationError('Provider, startDate, and endDate are required');
        }
        if (!['paystack', 'flutterwave'].includes(provider)) {
            throw new errors_2.ValidationError('Invalid payment provider');
        }
        const reconciliationResult = await payment_service_1.paymentService.reconcileTransactions({
            provider: provider,
            startDate: startDate,
            endDate: endDate,
        });
        res.status(200).json({
            success: true,
            message: 'Transaction reconciliation completed',
            data: reconciliationResult,
        });
    });
    static getPaymentStats = (0, errors_1.asyncHandler)(async (req, res) => {
        const { provider, startDate, endDate } = req.query;
        const stats = await payment_service_1.paymentService.getPaymentStats(provider, startDate, endDate);
        res.status(200).json({
            success: true,
            message: 'Payment statistics retrieved successfully',
            data: stats,
        });
    });
    static getPaymentProviders = (0, errors_1.asyncHandler)(async (_req, res) => {
        const providers = [
            {
                id: 'paystack',
                name: 'Paystack',
                description: 'Paystack payment gateway',
                supportedCurrencies: ['NGN', 'USD'],
                supportedPaymentTypes: ['card', 'bank_transfer', 'ussd', 'qr'],
                logo: 'https://paystack.com/static/img/logo-blue.svg',
            },
            {
                id: 'flutterwave',
                name: 'Flutterwave',
                description: 'Flutterwave payment gateway',
                supportedCurrencies: ['NGN', 'USD', 'EUR', 'GBP'],
                supportedPaymentTypes: ['card', 'bank_transfer', 'ussd', 'qr', 'mobile_money'],
                logo: 'https://flutterwave.com/images/logo.svg',
            },
            {
                id: 'aspfiy',
                name: 'Aspfiy',
                description: 'Aspfiy payment gateway - Reserved accounts, dynamic accounts, and bank transfers',
                supportedCurrencies: ['NGN'],
                supportedPaymentTypes: ['bank_transfer', 'reserved_account', 'dynamic_account'],
                logo: 'https://aspfiy.com/logo.svg',
                features: [
                    'Reserved Account Numbers',
                    'Dynamic Account Creation',
                    'Bank Transfer Payments',
                    'Account Resolution',
                    'Direct Bank Transfers'
                ],
            },
        ];
        res.status(200).json({
            success: true,
            message: 'Payment providers retrieved successfully',
            data: providers,
        });
    });
    static validateBankAccount = (0, errors_1.asyncHandler)(async (req, res) => {
        const { accountNumber, accountBank } = req.body;
        if (!accountNumber || !accountBank) {
            throw new errors_2.ValidationError('Account number and bank are required');
        }
        const { flutterwaveAdapter } = await Promise.resolve().then(() => __importStar(require('../../integrations/flutterwave')));
        const validationResult = await flutterwaveAdapter.validateBankAccount(accountNumber, accountBank);
        res.status(200).json({
            success: true,
            message: 'Bank account validation completed',
            data: validationResult,
        });
    });
    static createAspfiyReservedAccount = (0, errors_1.asyncHandler)(async (req, res) => {
        const { customer_email, customer_name, customer_phone, reference, metadata, provider } = req.body;
        if (!customer_email || !reference) {
            throw new errors_2.ValidationError('Customer email and reference are required');
        }
        if (provider && !['paga', 'palmpay'].includes(String(provider))) {
            throw new errors_2.ValidationError('Invalid Aspfiy provider. Must be paga or palmpay');
        }
        const { aspfiyAdapter } = await Promise.resolve().then(() => __importStar(require('../../integrations/aspfiy/')));
        const result = await aspfiyAdapter.createReservedAccount({
            customer_email,
            customer_name,
            customer_phone,
            reference,
            metadata,
            provider: provider || 'paga',
        });
        res.status(200).json({
            success: true,
            message: 'Reserved account created successfully',
            data: result.data,
        });
    });
    static createAspfiyReservedAccountDirect = (0, errors_1.asyncHandler)(async (req, res) => {
        const { customer_email, customer_name, customer_phone, provider } = req.body;
        const authEmail = req.user?.email;
        const email = (authEmail || customer_email || '').trim();
        if (!email) {
            throw new errors_2.BadRequestError('Customer email is required', 'EMAIL_REQUIRED');
        }
        if (provider && !['paga', 'palmpay'].includes(String(provider))) {
            throw new errors_2.ValidationError('Invalid Aspfiy provider. Must be paga or palmpay');
        }
        const reference = `DIRECT-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
        const { aspfiyAdapter } = await Promise.resolve().then(() => __importStar(require('../../integrations/aspfiy')));
        const result = await aspfiyAdapter.createReservedAccount({
            customer_email: email,
            customer_name,
            customer_phone,
            reference,
            metadata: {
                created_via: 'direct',
                timestamp: new Date().toISOString(),
                kyc_bypassed: true,
            },
            provider: provider || 'paga',
        });
        let accountData = (result && result.data) || {};
        if (accountData.account && accountData.account.account_number) {
            const acc = accountData.account;
            accountData = {
                ...accountData,
                account_number: acc.account_number,
                account_name: acc.account_name,
                bank_name: acc.bank_name,
                bank_code: acc.bank_code,
            };
        }
        if (!accountData.account_number) {
            try {
                const tx = await aspfiyAdapter.verifyTransaction(reference);
                if (tx) {
                    accountData = {
                        ...accountData,
                        account_number: tx.account_number || accountData.account_number,
                        account_name: tx.account_name || accountData.account_name,
                        bank_name: tx.bank_name || accountData.bank_name,
                    };
                }
            }
            catch (_e) {
            }
        }
        res.status(200).json({
            success: true,
            message: 'Reserved account created successfully without KYC verification',
            data: {
                ...(accountData.account_number ? { account_number: accountData.account_number } : {}),
                ...(accountData.account_name ? { account_name: accountData.account_name } : {}),
                ...(accountData.bank_name ? { bank_name: accountData.bank_name } : {}),
                ...(accountData.bank_code ? { bank_code: accountData.bank_code } : {}),
                reference,
                kyc_required: false,
                created_directly: true,
            },
        });
    });
    static createAspfiyDynamicAccount = (0, errors_1.asyncHandler)(async (req, res) => {
        const { amount, customer_email, customer_name, customer_phone, reference, expires_in, metadata } = req.body;
        if (!amount || !customer_email || !reference) {
            throw new errors_2.ValidationError('Amount, customer email, and reference are required');
        }
        const { aspfiyAdapter } = await Promise.resolve().then(() => __importStar(require('../../integrations/aspfiy')));
        const result = await aspfiyAdapter.createDynamicAccount({
            amount,
            customer_email,
            customer_name,
            customer_phone,
            reference,
            expires_in: expires_in || 60,
            metadata,
        });
        res.status(200).json({
            success: true,
            message: 'Dynamic account created successfully',
            data: result.data,
        });
    });
    static getAspfiyBanks = (0, errors_1.asyncHandler)(async (_req, res) => {
        const { aspfiyAdapter } = await Promise.resolve().then(() => __importStar(require('../../integrations/aspfiy')));
        const banks = await aspfiyAdapter.getBanks();
        res.status(200).json({
            success: true,
            message: 'Banks retrieved successfully',
            data: banks,
        });
    });
    static resolveAspfiyAccount = (0, errors_1.asyncHandler)(async (req, res) => {
        const { account_number, bank_code } = req.body;
        if (!account_number || !bank_code) {
            throw new errors_2.ValidationError('Account number and bank code are required');
        }
        const { aspfiyAdapter } = await Promise.resolve().then(() => __importStar(require('../../integrations/aspfiy')));
        const result = await aspfiyAdapter.resolveAccount({
            account_number,
            bank_code,
        });
        res.status(200).json({
            success: true,
            message: 'Account resolved successfully',
            data: result.data,
        });
    });
    static initiateAspfiyTransfer = (0, errors_1.asyncHandler)(async (req, res) => {
        const { amount, reference, recipient_account, recipient_bank_code, recipient_name, narration, metadata } = req.body;
        if (!amount || !reference || !recipient_account || !recipient_bank_code) {
            throw new errors_2.ValidationError('Amount, reference, recipient account, and recipient bank code are required');
        }
        const { aspfiyAdapter } = await Promise.resolve().then(() => __importStar(require('../../integrations/aspfiy')));
        const result = await aspfiyAdapter.initiateTransfer({
            amount,
            reference,
            recipient_account,
            recipient_bank_code,
            recipient_name,
            narration,
            metadata,
        });
        res.status(200).json({
            success: true,
            message: 'Transfer initiated successfully',
            data: result.data,
        });
    });
    static getWebhookStatus = (0, errors_1.asyncHandler)(async (req, res) => {
        const { reference } = req.params;
        if (!reference) {
            throw new errors_2.ValidationError('Reference is required');
        }
        const { redisManager } = await Promise.resolve().then(() => __importStar(require('../../config/redis')));
        const paymentIntent = await redisManager.get(`payment_intent:${reference}`);
        res.status(200).json({
            success: true,
            message: 'Webhook status retrieved successfully',
            data: {
                reference,
                hasPaymentIntent: !!paymentIntent,
                paymentIntent: paymentIntent ? JSON.parse(paymentIntent) : null,
            },
        });
    });
}
exports.PaymentController = PaymentController;
