"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const auth_1 = require("../../core/middleware/auth");
const jamb_request_1 = require("../../database/models/jamb-request");
const service_configuration_repo_1 = require("../../database/repositories/service-configuration.repo");
const router = (0, express_1.Router)();
router.post('/requests', auth_1.requireAuth, async (req, res) => {
    try {
        const { serviceType, payload } = req.body || {};
        if (!serviceType || !payload) {
            res.status(400).json({ success: false, message: 'serviceType and payload are required' });
            return;
        }
        try {
            const globalCfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey('jamb_enabled', 'service', { serviceType: 'jamb' });
            if (globalCfg && typeof globalCfg.value === 'boolean' && globalCfg.value === false) {
                res.status(403).json({ success: false, message: 'JAMB services are disabled' });
                return;
            }
            const svcCfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(`jamb_${String(serviceType)}_enabled`, 'service', { serviceType: 'jamb' });
            if (svcCfg && typeof svcCfg.value === 'boolean' && svcCfg.value === false) {
                res.status(403).json({ success: false, message: `${serviceType} service is disabled` });
                return;
            }
        }
        catch { }
        const cfgKey = `jamb_price_${String(serviceType)}`;
        const cfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(cfgKey, 'service', { serviceType: 'jamb' });
        const price = Number(cfg?.value || 0);
        const doc = await jamb_request_1.JambRequest.create({ userId: req.user.userId, serviceType, payload, price });
        res.json({ success: true, data: { id: doc.id, price } });
    }
    catch (e) {
        res.status(500).json({ success: false, message: e?.message || 'Failed to create request' });
    }
});
router.get('/admin/requests', auth_1.requireAuth, async (req, res) => {
    if (req.user?.role !== 'admin') {
        res.status(403).json({ success: false, message: 'Admin access required' });
        return;
    }
    const status = req.query['status'];
    const filter = {};
    if (status)
        filter.status = status;
    const items = await jamb_request_1.JambRequest.find(filter).sort({ createdAt: -1 }).limit(200);
    res.json({ success: true, data: items });
});
router.patch('/admin/requests/:id', auth_1.requireAuth, async (req, res) => {
    if (req.user?.role !== 'admin') {
        res.status(403).json({ success: false, message: 'Admin access required' });
        return;
    }
    const { id } = req.params;
    const { status, resultUrl, notes } = req.body || {};
    const doc = await jamb_request_1.JambRequest.findByIdAndUpdate(id, { $set: { status, resultUrl, notes } }, { new: true });
    if (!doc) {
        res.status(404).json({ success: false, message: 'Not found' });
        return;
    }
    res.json({ success: true, data: doc });
});
exports.default = router;
