"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.disputeService = exports.DisputeService = void 0;
const dispute_1 = require("../../database/repositories/dispute");
const transactions_1 = require("../../database/repositories/transactions");
const wallet_service_1 = require("../wallet/wallet.service");
class DisputeService {
    async createDispute(request) {
        const transaction = await transactions_1.transactionRepository.findById(request.transactionId);
        if (!transaction) {
            throw new Error('Transaction not found');
        }
        const existingDispute = await dispute_1.disputeRepository.findOne({
            transactionId: request.transactionId,
            status: { $nin: ['closed', 'resolved'] }
        });
        if (existingDispute) {
            throw new Error('Active dispute already exists for this transaction');
        }
        const dispute = await dispute_1.disputeRepository.create({
            userId: request.userId,
            transactionId: request.transactionId,
            type: request.type,
            reason: request.reason,
            description: request.description,
            priority: request.priority,
            evidence: request.evidence || [],
            metadata: request.metadata || {},
            status: 'opened',
            openedAt: new Date()
        });
        return dispute;
    }
    async updateDispute(disputeId, updates) {
        const dispute = await dispute_1.disputeRepository.findById(disputeId);
        if (!dispute) {
            throw new Error('Dispute not found');
        }
        const updateData = { ...updates };
        if (updates.status === 'resolved' || updates.status === 'closed') {
            updateData.resolvedAt = new Date();
        }
        const updatedDispute = await dispute_1.disputeRepository.update(disputeId, updateData);
        if (!updatedDispute) {
            throw new Error('Failed to update dispute');
        }
        if (updates.status === 'resolved' && updates.resolutionAmount && updates.resolutionAmount > 0) {
            await this.processDisputeRefund(dispute, updates.resolutionAmount);
        }
        return updatedDispute;
    }
    async processChargeback(request) {
        const transaction = await transactions_1.transactionRepository.findById(request.transactionId);
        if (!transaction) {
            throw new Error('Transaction not found');
        }
        const dispute = await this.createDispute({
            userId: request.userId,
            transactionId: request.transactionId,
            type: 'chargeback',
            reason: request.reason,
            description: `Chargeback request for transaction ${request.transactionId}`,
            priority: 'urgent',
            evidence: request.evidence,
            metadata: { chargebackAmount: request.amount || transaction.amount }
        });
        await dispute_1.disputeRepository.update(dispute.id, {
            status: 'under_review',
        });
        return dispute;
    }
    async processDisputeRefund(dispute, refundAmount) {
        const transaction = await transactions_1.transactionRepository.findById(dispute.transactionId.toString());
        if (!transaction) {
            throw new Error('Transaction not found');
        }
        const refundReference = `refund-${transaction.reference}-${Date.now()}`;
        await wallet_service_1.walletService.credit({
            userId: dispute.userId.toString(),
            currency: transaction.currency,
            amount: refundAmount,
            reference: refundReference,
            category: 'refund',
            description: `Refund for dispute: ${dispute.reason}`,
            metadata: {
                disputeId: dispute.id,
                originalTransaction: transaction.reference,
                disputeType: dispute.type
            }
        });
    }
    async getDisputeById(disputeId) {
        if (!dispute_1.disputeRepository.isValidId(disputeId))
            return null;
        return await dispute_1.disputeRepository.findById(disputeId);
    }
    async getDisputes(filters, page = 1, limit = 20) {
        const filter = {};
        if (filters.userId)
            filter.userId = filters.userId;
        if (filters.type)
            filter.type = filters.type;
        if (filters.status)
            filter.status = filters.status;
        if (filters.priority)
            filter.priority = filters.priority;
        if (filters.dateFrom || filters.dateTo) {
            filter.openedAt = {};
            if (filters.dateFrom)
                filter.openedAt.$gte = filters.dateFrom;
            if (filters.dateTo)
                filter.openedAt.$lte = filters.dateTo;
        }
        return dispute_1.disputeRepository.findPaginated(filter, page, limit);
    }
    async getDisputeStats() {
        const stats = await dispute_1.disputeRepository.getDisputeStatsAggregation();
        return stats[0] || {
            totalDisputes: 0,
            openDisputes: 0,
            underReview: 0,
            resolved: 0,
            closed: 0,
            urgentPriority: 0,
            chargebacks: 0
        };
    }
    async escalateUrgentDisputes() {
        const urgentDisputes = await dispute_1.disputeRepository.find({
            priority: 'urgent',
            status: { $in: ['opened', 'under_review'] },
            openedAt: { $lte: new Date(Date.now() - 24 * 60 * 60 * 1000) }
        });
        for (const dispute of urgentDisputes) {
            await dispute_1.disputeRepository.update(dispute.id, {
                priority: 'urgent',
                metadata: {
                    ...dispute.metadata,
                    escalatedAt: new Date(),
                    escalationReason: 'Automatic escalation for urgent disputes older than 24 hours'
                }
            });
        }
    }
}
exports.DisputeService = DisputeService;
exports.disputeService = new DisputeService();
