"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.disputeController = exports.DisputeController = void 0;
const dispute_service_1 = require("./dispute.service");
class DisputeController {
    async createDispute(req, res) {
        try {
            const { transactionId, type, reason, description, priority, evidence, metadata } = req.body;
            if (!transactionId || !type || !reason || !description || !priority) {
                res.status(400).json({
                    error: 'Missing required fields: transactionId, type, reason, description, priority'
                });
                return;
            }
            const dispute = await dispute_service_1.disputeService.createDispute({
                userId: req.user.userId,
                transactionId,
                type,
                reason,
                description,
                priority,
                evidence,
                metadata
            });
            res.status(201).json({
                success: true,
                message: 'Dispute created successfully',
                data: dispute
            });
        }
        catch (error) {
            res.status(400).json({
                error: error.message || 'Failed to create dispute'
            });
        }
    }
    async updateDispute(req, res) {
        try {
            const { disputeId } = req.params;
            const { status, resolution, resolutionAmount, adminNotes, metadata } = req.body;
            const updatedDispute = await dispute_service_1.disputeService.updateDispute(disputeId, {
                status,
                resolution,
                resolutionAmount,
                adminNotes,
                metadata
            });
            res.json({
                success: true,
                message: 'Dispute updated successfully',
                data: updatedDispute
            });
        }
        catch (error) {
            res.status(400).json({
                error: error.message || 'Failed to update dispute'
            });
        }
    }
    async processChargeback(req, res) {
        try {
            const { transactionId, reason, evidence, amount } = req.body;
            if (!transactionId || !reason || !evidence) {
                res.status(400).json({
                    error: 'Missing required fields: transactionId, reason, evidence'
                });
                return;
            }
            const dispute = await dispute_service_1.disputeService.processChargeback({
                userId: req.user.userId,
                transactionId,
                reason,
                evidence,
                amount
            });
            res.status(201).json({
                success: true,
                message: 'Chargeback request processed successfully',
                data: dispute
            });
        }
        catch (error) {
            res.status(400).json({
                error: error.message || 'Failed to process chargeback'
            });
        }
    }
    async getDisputes(req, res) {
        try {
            const { page = 1, limit = 20, } = req.query;
            const result = await dispute_service_1.disputeService.getDisputes({
                userId: req.user.userId,
                type: req.query['type'],
                status: req.query['status'],
                priority: req.query['priority'],
                ...(req.query['dateFrom'] && { dateFrom: new Date(req.query['dateFrom']) }),
                ...(req.query['dateTo'] && { dateTo: new Date(req.query['dateTo']) }),
            }, Number(page), Number(limit));
            res.json({
                success: true,
                data: result.data,
                pagination: result.pagination
            });
        }
        catch (error) {
            res.status(400).json({
                error: error.message || 'Failed to get disputes'
            });
        }
    }
    async getDisputeStats(_req, res) {
        try {
            const stats = await dispute_service_1.disputeService.getDisputeStats();
            res.json({
                success: true,
                data: stats
            });
        }
        catch (error) {
            res.status(400).json({
                error: error.message || 'Failed to get dispute statistics'
            });
        }
    }
    async getDisputeById(req, res) {
        try {
            const { disputeId } = req.params;
            const dispute = await dispute_service_1.disputeService.getDisputeById(disputeId);
            if (!dispute) {
                res.status(404).json({
                    error: 'Dispute not found'
                });
                return;
            }
            res.json({
                success: true,
                data: dispute
            });
        }
        catch (error) {
            res.status(400).json({
                error: error.message || 'Failed to get dispute'
            });
        }
    }
    async escalateUrgentDisputes(req, res) {
        try {
            if (req.user.role !== 'admin') {
                res.status(403).json({
                    error: 'Insufficient permissions to escalate disputes'
                });
                return;
            }
            await dispute_service_1.disputeService.escalateUrgentDisputes();
            res.json({
                success: true,
                message: 'Urgent disputes escalated successfully'
            });
        }
        catch (error) {
            res.status(400).json({
                error: error.message || 'Failed to escalate urgent disputes'
            });
        }
    }
}
exports.DisputeController = DisputeController;
exports.disputeController = new DisputeController();
