"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.billsService = exports.BillsService = void 0;
const wallet_service_1 = require("../wallet/wallet.service");
const bills_1 = require("../../database/repositories/bills");
const transactions_1 = require("../../database/repositories/transactions");
const registry_1 = require("../../integrations/registry");
class BillsService {
    async callProvider(order) {
        const adapter = (0, registry_1.getBillAdapter)();
        const payload = {
            amount: order.amount,
            customerId: order.customerId,
            reference: String(order.id),
        };
        if (order.productCode !== undefined) {
            payload.productCode = order.productCode;
        }
        return adapter.payBill(payload);
    }
    async payBill(params) {
        const existingTx = await transactions_1.transactionRepository.findByReference(params.reference);
        if (existingTx) {
            const existingOrder = await bills_1.billPaymentRepository.findOne({ transactionId: existingTx.id });
            if (existingOrder)
                return { order: existingOrder };
        }
        const order = await bills_1.billPaymentRepository.create({
            userId: params.userId,
            type: params.type,
            biller: params.biller,
            customerId: params.customerId,
            amount: params.amount,
            productCode: params.productCode,
            status: 'pending',
            provider: 'others'
        });
        const { transaction } = await wallet_service_1.walletService.debit({
            userId: params.userId,
            currency: params.currency,
            amount: params.amount,
            reference: params.reference,
            category: 'bill_payment',
            description: `${params.type} payment to ${params.biller}`,
            metadata: { orderId: order.id }
        });
        const updatedOrder = await bills_1.billPaymentRepository.updateById(order.id, { transactionId: transaction.id, status: 'processing' });
        if (!updatedOrder)
            throw new Error('Failed to update bill order');
        const provider = await this.callProvider(updatedOrder);
        if (provider.success) {
            await bills_1.billPaymentRepository.updateById(updatedOrder.id, {
                status: 'successful',
                providerReference: provider.providerReference,
                token: provider.token,
                metadata: provider.meta,
            });
        }
        else {
            await bills_1.billPaymentRepository.updateById(updatedOrder.id, { status: 'failed' });
            const reversalRef = `${params.reference}-reversal`;
            await wallet_service_1.walletService.credit({
                userId: params.userId,
                currency: params.currency,
                amount: params.amount,
                reference: reversalRef,
                category: 'reversal',
                description: `Reversal for failed bill payment ${params.reference}`,
                metadata: { originalReference: params.reference, orderId: updatedOrder.id }
            });
        }
        const finalOrder = await bills_1.billPaymentRepository.findById(updatedOrder.id);
        if (!finalOrder)
            throw new Error('Bill order not found after update');
        return { order: finalOrder };
    }
}
exports.BillsService = BillsService;
exports.billsService = new BillsService();
