"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.authService = exports.AuthService = void 0;
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const crypto_1 = __importDefault(require("crypto"));
const env_1 = require("../../config/env");
const redis_1 = require("../../config/redis");
const user_model_1 = require("../users/user.model");
const notifier_1 = require("../../core/utils/notifier");
class AuthService {
    hashPassword(password) {
        const salt = bcryptjs_1.default.genSaltSync(env_1.env.BCRYPT_ROUNDS);
        return bcryptjs_1.default.hashSync(password, salt);
    }
    verifyPassword(password, passwordHash) {
        return bcryptjs_1.default.compareSync(password, passwordHash);
    }
    async blacklistRefreshToken(jti, ttlSeconds) {
        const client = await redis_1.redisManager.getClient('sessions');
        await client.setEx(`rt:blacklist:${jti}`, ttlSeconds, '1');
    }
    async isRefreshTokenBlacklisted(jti) {
        const client = await redis_1.redisManager.getClient('sessions');
        const exists = await client.exists(`rt:blacklist:${jti}`);
        return exists === 1;
    }
    generateTokens(payload) {
        const userPayload = { ...payload };
        const accessToken = jsonwebtoken_1.default.sign(userPayload, env_1.env.JWT_SECRET, { expiresIn: env_1.env.JWT_EXPIRES_IN });
        const refreshJti = crypto_1.default.randomUUID();
        const refreshToken = jsonwebtoken_1.default.sign({ ...payload, jti: refreshJti }, env_1.env.JWT_SECRET, { expiresIn: env_1.env.JWT_REFRESH_EXPIRES_IN });
        return { accessToken, refreshToken };
    }
    async signup(input) {
        const { email, phone, password, firstName, lastName } = input;
        console.log('[AuthService] Starting signup process:', {
            email: email?.toLowerCase().trim(),
            phone: phone?.trim(),
            firstName,
            lastName
        });
        if (email) {
            const normalizedEmail = email.toLowerCase().trim();
            console.log('[AuthService] Checking email uniqueness:', normalizedEmail);
            const exists = await user_model_1.User.findOne({
                email: normalizedEmail,
                isDeleted: { $ne: true }
            });
            if (exists) {
                console.log('[AuthService] Email already exists:', {
                    existingUserId: exists._id,
                    existingEmail: exists.email,
                    isDeleted: exists.isDeleted,
                    createdAt: exists.createdAt
                });
                throw new Error('Email already in use');
            }
            console.log('[AuthService] Email is available');
        }
        if (phone) {
            const normalizedPhone = phone.trim();
            console.log('[AuthService] Checking phone uniqueness:', normalizedPhone);
            const exists = await user_model_1.User.findOne({
                phone: normalizedPhone,
                isDeleted: { $ne: true }
            });
            if (exists) {
                console.log('[AuthService] Phone already exists:', {
                    existingUserId: exists._id,
                    existingPhone: exists.phone,
                    isDeleted: exists.isDeleted,
                    createdAt: exists.createdAt
                });
                throw new Error('Phone already in use');
            }
            console.log('[AuthService] Phone is available');
        }
        console.log('[AuthService] Creating user...');
        const passwordHash = this.hashPassword(password);
        try {
            const user = await user_model_1.User.create({
                email: email?.toLowerCase().trim(),
                phone: phone?.trim(),
                passwordHash,
                firstName,
                lastName
            });
            console.log('[AuthService] User created successfully:', {
                userId: user._id,
                email: user.email,
                phone: user.phone
            });
            console.log('[AuthService] Generating tokens...');
            const tokens = this.generateTokens({ userId: user.id, role: user.role });
            console.log('[AuthService] Signup process completed successfully');
            return { user, tokens };
        }
        catch (error) {
            console.error('[AuthService] User creation failed:', {
                error: error instanceof Error ? error.message : 'Unknown error',
                email: email?.toLowerCase().trim(),
                phone: phone?.trim()
            });
            throw error;
        }
    }
    async login(input) {
        const { email, phone, password } = input;
        const user = await user_model_1.User.findOne(email ? { email } : { phone });
        if (!user)
            throw new Error('Invalid credentials');
        const ok = this.verifyPassword(password, user.passwordHash);
        if (!ok)
            throw new Error('Invalid credentials');
        const tokens = this.generateTokens({ userId: user.id, role: user.role });
        try {
            const { sessionService } = await Promise.resolve().then(() => __importStar(require('../users/session.service')));
            const ip = input.ipAddress;
            const ua = input.userAgent;
            const meta = {};
            if (ip)
                meta.ipAddress = ip;
            if (ua)
                meta.userAgent = ua;
            await sessionService.create(user.id, meta);
            user.lastLoginAt = new Date();
            if (ip)
                user.lastLoginIp = ip;
            await user.save();
        }
        catch { }
        return { user, tokens };
    }
    async refresh(input) {
        try {
            const decoded = jsonwebtoken_1.default.verify(input.refreshToken, env_1.env.JWT_SECRET);
            const jti = decoded['jti'];
            const userId = decoded['userId'];
            if (!jti || !userId)
                throw new Error('Invalid token');
            const isBlacklisted = await this.isRefreshTokenBlacklisted(jti);
            if (isBlacklisted)
                throw new Error('Token revoked');
            const exp = decoded['exp'];
            const nowSec = Math.floor(Date.now() / 1000);
            const remainingTtl = typeof exp === 'number' ? Math.max(exp - nowSec, 0) : 0;
            if (remainingTtl > 0) {
                await this.blacklistRefreshToken(jti, remainingTtl);
            }
            const user = await user_model_1.User.findById(userId);
            if (!user)
                throw new Error('Invalid token');
            return this.generateTokens({ userId, role: user.role });
        }
        catch (e) {
            throw new Error('Invalid refresh token');
        }
    }
    async logout(input) {
        try {
            const decoded = jsonwebtoken_1.default.verify(input.refreshToken, env_1.env.JWT_SECRET);
            const jti = decoded['jti'];
            if (!jti)
                return;
            const exp = decoded['exp'];
            const nowSec = Math.floor(Date.now() / 1000);
            const remainingTtl = typeof exp === 'number' ? Math.max(exp - nowSec, 0) : 0;
            if (remainingTtl > 0) {
                await this.blacklistRefreshToken(jti, remainingTtl);
            }
        }
        catch (_e) {
        }
    }
    async storeCode(key, code, ttlSeconds) {
        const client = await redis_1.redisManager.getClient('sessions');
        await client.setEx(key, ttlSeconds, code);
    }
    async getCode(key) {
        const client = await redis_1.redisManager.getClient('sessions');
        return client.get(key);
    }
    async sendVerificationCode(user, channel) {
        const code = String(Math.floor(100000 + Math.random() * 900000));
        const key = `verify:${channel}:${user.id}`;
        await this.storeCode(key, code, 10 * 60);
        if (channel === 'email' && user.email) {
            await notifier_1.notifier.sendEmail(user.email, 'Your verification code', `Your code is ${code}`);
        }
        if (channel === 'phone' && user.phone) {
            await notifier_1.notifier.sendSms(user.phone, `Your verification code is ${code}`);
        }
    }
    async confirmVerification(user, channel, code) {
        const key = `verify:${channel}:${user.id}`;
        const saved = await this.getCode(key);
        if (!saved || saved !== code)
            return false;
        if (channel === 'email') {
            user.isEmailVerified = true;
        }
        else {
            user.isPhoneVerified = true;
        }
        await user.save();
        const client = await redis_1.redisManager.getClient('sessions');
        await client.del(key);
        return true;
    }
    async sendPasswordReset(user) {
        const jti = crypto_1.default.randomUUID();
        const payload = { userId: user.id, purpose: 'password_reset', jti };
        const token = jsonwebtoken_1.default.sign(payload, env_1.env.JWT_SECRET, { expiresIn: '15m' });
        const client = await redis_1.redisManager.getClient('sessions');
        await client.setEx(`pwdreset:jti:${jti}`, 15 * 60, '1');
        const resetLink = `unifyverifier://auth/reset-password?token=${encodeURIComponent(token)}`;
        if (user.email) {
            const emailSubject = 'Reset Your Password - Unify';
            const emailContent = this.generatePasswordResetEmail(user.firstName || 'User', resetLink);
            const htmlContent = this.generatePasswordResetEmailHTML(user.firstName || 'User', resetLink);
            await notifier_1.notifier.sendEmail(user.email, emailSubject, emailContent, htmlContent);
        }
        else if (user.phone) {
            await notifier_1.notifier.sendSms(user.phone, `Your password reset link: ${resetLink}`);
        }
    }
    generatePasswordResetEmail(userName, resetLink) {
        return `
Hello ${userName},

You requested to reset your password for your Unify account.

Click the link below to reset your password:
${resetLink}

This link will expire in 15 minutes for security reasons.

If you didn't request this password reset, please ignore this email. Your password will remain unchanged.

For security reasons, please do not share this link with anyone.

Best regards,
The Unify Team

---
This is an automated message. Please do not reply to this email.
    `.trim();
    }
    generatePasswordResetEmailHTML(userName, resetLink) {
        return `
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Your Password - Unify</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 600px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f8f9fa;
        }
        .container {
            background-color: #ffffff;
            border-radius: 8px;
            padding: 40px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        .logo {
            font-size: 24px;
            font-weight: bold;
            color: #2563eb;
            margin-bottom: 10px;
        }
        .title {
            font-size: 20px;
            font-weight: 600;
            color: #1f2937;
            margin-bottom: 20px;
        }
        .content {
            margin-bottom: 30px;
        }
        .reset-button {
            display: inline-block;
            background-color: #2563eb;
            color: #ffffff;
            padding: 12px 24px;
            text-decoration: none;
            border-radius: 6px;
            font-weight: 600;
            margin: 20px 0;
            transition: background-color 0.2s;
        }
        .reset-button:hover {
            background-color: #1d4ed8;
        }
        .warning {
            background-color: #fef3c7;
            border: 1px solid #f59e0b;
            border-radius: 6px;
            padding: 15px;
            margin: 20px 0;
            color: #92400e;
        }
        .footer {
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #e5e7eb;
            font-size: 14px;
            color: #6b7280;
            text-align: center;
        }
        .link {
            color: #2563eb;
            word-break: break-all;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div class="logo">Unify</div>
            <h1 class="title">Reset Your Password</h1>
        </div>
        
        <div class="content">
            <p>Hello <strong>${userName}</strong>,</p>
            
            <p>You requested to reset your password for your Unify account.</p>
            
            <p>Click the button below to reset your password:</p>
            
            <div style="text-align: center;">
                <a href="${resetLink}" class="reset-button">Reset Password</a>
            </div>
            
            <p>Or copy and paste this link into your browser:</p>
            <p><a href="${resetLink}" class="link">${resetLink}</a></p>
            
            <div class="warning">
                <strong>⚠️ Important:</strong> This link will expire in 15 minutes for security reasons.
            </div>
            
            <p>If you didn't request this password reset, please ignore this email. Your password will remain unchanged.</p>
            
            <p>For security reasons, please do not share this link with anyone.</p>
        </div>
        
        <div class="footer">
            <p>Best regards,<br>The Unify Team</p>
            <p><em>This is an automated message. Please do not reply to this email.</em></p>
        </div>
    </div>
</body>
</html>
    `.trim();
    }
    async resetPassword(token, newPassword) {
        try {
            const decoded = jsonwebtoken_1.default.verify(token, env_1.env.JWT_SECRET);
            if (decoded['purpose'] !== 'password_reset')
                return false;
            const userId = decoded['userId'];
            const jti = decoded['jti'];
            if (!userId || !jti)
                return false;
            const client = await redis_1.redisManager.getClient('sessions');
            const exists = await client.exists(`pwdreset:jti:${jti}`);
            if (exists !== 1)
                return false;
            const user = await user_model_1.User.findById(userId);
            if (!user)
                return false;
            user.passwordHash = this.hashPassword(newPassword);
            await user.save();
            await client.del(`pwdreset:jti:${jti}`);
            return true;
        }
        catch (_e) {
            return false;
        }
    }
}
exports.AuthService = AuthService;
exports.authService = new AuthService();
