"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.authController = void 0;
const auth_schemas_1 = require("./auth.schemas");
const auth_service_1 = require("./auth.service");
class AuthController {
    async signup(req, res) {
        const parsed = auth_schemas_1.signupSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const timeoutId = setTimeout(() => {
            if (!res.headersSent) {
                console.error('[AuthController] Signup request timed out');
                res.status(408).json({ error: 'Request timeout - please try again' });
            }
        }, 25000);
        try {
            console.log('[AuthController] Starting signup for:', {
                email: parsed.data.email,
                phone: parsed.data.phone,
                firstName: parsed.data.firstName,
                lastName: parsed.data.lastName
            });
            const result = await auth_service_1.authService.signup(parsed.data);
            clearTimeout(timeoutId);
            console.log('[AuthController] Signup successful, sending verification codes in background...');
            if (!res.headersSent) {
                res.status(201).json({ user: result.user.toJSON(), tokens: result.tokens });
                console.log('[AuthController] Signup response sent successfully');
            }
            setImmediate(async () => {
                try {
                    const tasks = [];
                    if (result.user.email) {
                        tasks.push(auth_service_1.authService.sendVerificationCode(result.user, 'email'));
                    }
                    if (result.user.phone) {
                        tasks.push(auth_service_1.authService.sendVerificationCode(result.user, 'phone'));
                    }
                    if (tasks.length > 0) {
                        await Promise.allSettled(tasks);
                        console.log('[AuthController] Background verification codes sent');
                    }
                }
                catch (e) {
                    console.warn('[AuthController] Background verification code sending failed:', e);
                }
            });
        }
        catch (error) {
            clearTimeout(timeoutId);
            console.error('[AuthController] Signup failed:', {
                error: error.message,
                email: parsed.data.email,
                phone: parsed.data.phone
            });
            if (!res.headersSent) {
                res.status(400).json({ error: error.message || 'Signup failed' });
            }
        }
    }
    async login(req, res) {
        const parsed = auth_schemas_1.loginSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        try {
            const result = await auth_service_1.authService.login({ ...parsed.data, ipAddress: req.ip, userAgent: req.headers['user-agent'] });
            res.status(200).json({ user: result.user.toJSON(), tokens: result.tokens });
        }
        catch (error) {
            res.status(401).json({ error: error.message || 'Login failed' });
        }
    }
    async refresh(req, res) {
        const parsed = auth_schemas_1.refreshSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        try {
            const tokens = await auth_service_1.authService.refresh(parsed.data);
            res.status(200).json({ tokens });
        }
        catch (error) {
            res.status(401).json({ error: error.message || 'Refresh failed' });
        }
    }
    async logout(req, res) {
        const parsed = auth_schemas_1.logoutSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        await auth_service_1.authService.logout(parsed.data);
        res.status(204).send();
    }
    async requestVerification(req, res) {
        const parsed = auth_schemas_1.verificationRequestSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const { channel, userId } = parsed.data;
        const user = await (await Promise.resolve().then(() => __importStar(require('../users/user.model')))).User.findById(userId);
        if (!user) {
            res.status(404).json({ error: 'User not found' });
            return;
        }
        await (await Promise.resolve().then(() => __importStar(require('./auth.service')))).authService.sendVerificationCode(user, channel);
        res.status(200).json({ message: 'Verification code sent' });
    }
    async confirmVerification(req, res) {
        const parsed = auth_schemas_1.verificationConfirmSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const { channel, userId, code } = parsed.data;
        const user = await (await Promise.resolve().then(() => __importStar(require('../users/user.model')))).User.findById(userId);
        if (!user) {
            res.status(404).json({ error: 'User not found' });
            return;
        }
        const ok = await (await Promise.resolve().then(() => __importStar(require('./auth.service')))).authService.confirmVerification(user, channel, code);
        if (!ok) {
            res.status(400).json({ error: 'Invalid or expired code' });
            return;
        }
        res.status(200).json({ message: 'Verification successful' });
    }
    async forgotPassword(req, res) {
        const parsed = auth_schemas_1.forgotPasswordSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const { email, phone } = parsed.data;
        const User = (await Promise.resolve().then(() => __importStar(require('../users/user.model')))).User;
        const user = await User.findOne(email ? { email } : { phone });
        if (!user) {
            res.status(200).json({ success: true, message: 'Password reset token sent' });
            return;
        }
        await (await Promise.resolve().then(() => __importStar(require('./auth.service')))).authService.sendPasswordReset(user);
        res.status(200).json({ success: true, message: 'Password reset token sent' });
    }
    async resetPassword(req, res) {
        const parsed = auth_schemas_1.resetPasswordSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const { token, newPassword } = parsed.data;
        const ok = await (await Promise.resolve().then(() => __importStar(require('./auth.service')))).authService.resetPassword(token, newPassword);
        if (!ok) {
            res.status(400).json({ success: false, error: 'Invalid or expired token' });
            return;
        }
        res.status(200).json({ success: true, message: 'Password has been reset' });
    }
    async introspect(req, res) {
        const parsed = auth_schemas_1.introspectSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const { token, type } = parsed.data;
        try {
            const jsonwebtoken = await Promise.resolve().then(() => __importStar(require('jsonwebtoken')));
            const decoded = jsonwebtoken.verify(token, (await Promise.resolve().then(() => __importStar(require('../../config/env')))).env.JWT_SECRET);
            if (type === 'refresh') {
                const jti = decoded['jti'];
                if (jti) {
                    const { redisManager } = await Promise.resolve().then(() => __importStar(require('../../config/redis')));
                    const client = await redisManager.getClient('sessions');
                    const blacklisted = await client.exists(`rt:blacklist:${jti}`);
                    res.status(200).json({ active: blacklisted !== 1, payload: decoded });
                    return;
                }
            }
            res.status(200).json({ active: true, payload: decoded });
        }
        catch (_e) {
            res.status(200).json({ active: false });
        }
    }
    async changeRole(req, res) {
        const parsed = auth_schemas_1.changeRoleSchema.safeParse(req.body);
        if (!parsed.success) {
            res.status(400).json({ errors: parsed.error.flatten() });
            return;
        }
        const { userId, role } = parsed.data;
        const User = (await Promise.resolve().then(() => __importStar(require('../users/user.model')))).User;
        const user = await User.findById(userId);
        if (!user) {
            res.status(404).json({ error: 'User not found' });
            return;
        }
        user.role = role;
        await user.save();
        res.status(200).json({ user: user.toJSON() });
    }
}
exports.authController = new AuthController();
