"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WalletConfigController = void 0;
const zod_1 = require("zod");
const errors_1 = require("../../../core/errors");
const service_configuration_repo_1 = require("../../../database/repositories/service-configuration.repo");
const validateRequest = (schema) => (req, res, next) => {
    try {
        const result = schema.parse({ body: req.body, params: req.params, query: req.query });
        req.body = result.body;
        req.params = result.params;
        req.query = result.query;
        next();
    }
    catch (error) {
        next(new errors_1.AppError(error.errors ? error.errors[0].message : error.message, 400));
    }
};
const updateWalletConfigSchema = zod_1.z.object({
    body: zod_1.z.object({
        transactionLimits: zod_1.z.object({
            dailyLimit: zod_1.z.number().min(0).optional(),
            monthlyLimit: zod_1.z.number().min(0).optional(),
            singleTransactionLimit: zod_1.z.number().min(0).optional(),
        }).optional(),
        fees: zod_1.z.object({
            transferFee: zod_1.z.number().min(0).max(100).optional(),
            withdrawalFee: zod_1.z.number().min(0).max(100).optional(),
            minimumFee: zod_1.z.number().min(0).optional(),
        }).optional(),
        currencies: zod_1.z.object({
            supportedCurrencies: zod_1.z.array(zod_1.z.string()).optional(),
            defaultCurrency: zod_1.z.string().optional(),
        }).optional(),
        security: zod_1.z.object({
            requirePinForTransactions: zod_1.z.boolean().optional(),
            requirePinForWithdrawals: zod_1.z.boolean().optional(),
            maxFailedAttempts: zod_1.z.number().min(1).max(10).optional(),
        }).optional(),
    }),
});
const setTransactionLimitsSchema = zod_1.z.object({
    body: zod_1.z.object({
        dailyLimit: zod_1.z.number().min(0),
        monthlyLimit: zod_1.z.number().min(0),
        singleTransactionLimit: zod_1.z.number().min(0),
    }),
});
const setFeesSchema = zod_1.z.object({
    body: zod_1.z.object({
        transferFee: zod_1.z.number().min(0).max(100),
        withdrawalFee: zod_1.z.number().min(0).max(100),
        minimumFee: zod_1.z.number().min(0),
    }),
});
class WalletConfigController {
    static getWalletConfig = (0, errors_1.asyncHandler)(async (_req, res) => {
        const configs = await service_configuration_repo_1.serviceConfigurationRepository.listByCategory('wallet_config');
        const config = {
            transactionLimits: {
                dailyLimit: 1000000,
                monthlyLimit: 10000000,
                singleTransactionLimit: 500000,
            },
            fees: {
                transferFee: 0,
                withdrawalFee: 1.5,
                minimumFee: 10,
            },
            currencies: {
                supportedCurrencies: ['NGN', 'USD'],
                defaultCurrency: 'NGN',
            },
            security: {
                requirePinForTransactions: false,
                requirePinForWithdrawals: true,
                maxFailedAttempts: 3,
            },
        };
        for (const cfg of configs) {
            if (cfg.key === 'wallet_transaction_limits') {
                config.transactionLimits = { ...config.transactionLimits, ...cfg.value };
            }
            else if (cfg.key === 'wallet_fees') {
                config.fees = { ...config.fees, ...cfg.value };
            }
            else if (cfg.key === 'wallet_currencies') {
                config.currencies = { ...config.currencies, ...cfg.value };
            }
            else if (cfg.key === 'wallet_security') {
                config.security = { ...config.security, ...cfg.value };
            }
        }
        res.json({
            success: true,
            data: config,
        });
    });
    static updateWalletConfig = [
        validateRequest(updateWalletConfigSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            const { transactionLimits, fees, currencies, security } = req.body;
            const updates = [];
            if (transactionLimits) {
                updates.push(service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key: 'wallet_transaction_limits',
                    scope: 'global',
                    serviceType: 'wallet',
                    category: 'wallet_config',
                    value: transactionLimits,
                    isDeleted: false,
                }));
            }
            if (fees) {
                updates.push(service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key: 'wallet_fees',
                    scope: 'global',
                    serviceType: 'wallet',
                    category: 'wallet_config',
                    value: fees,
                    isDeleted: false,
                }));
            }
            if (currencies) {
                updates.push(service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key: 'wallet_currencies',
                    scope: 'global',
                    serviceType: 'wallet',
                    category: 'wallet_config',
                    value: currencies,
                    isDeleted: false,
                }));
            }
            if (security) {
                updates.push(service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key: 'wallet_security',
                    scope: 'global',
                    serviceType: 'wallet',
                    category: 'wallet_config',
                    value: security,
                    isDeleted: false,
                }));
            }
            await Promise.all(updates);
            res.json({
                success: true,
                message: 'Wallet configuration updated successfully',
            });
        }),
    ];
    static setTransactionLimits = [
        validateRequest(setTransactionLimitsSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            const { dailyLimit, monthlyLimit, singleTransactionLimit } = req.body;
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: 'wallet_transaction_limits',
                scope: 'global',
                serviceType: 'wallet',
                category: 'wallet_config',
                value: { dailyLimit, monthlyLimit, singleTransactionLimit },
                isDeleted: false,
            });
            res.json({
                success: true,
                message: 'Transaction limits updated successfully',
                data: { dailyLimit, monthlyLimit, singleTransactionLimit },
            });
        }),
    ];
    static setFees = [
        validateRequest(setFeesSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            const { transferFee, withdrawalFee, minimumFee } = req.body;
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: 'wallet_fees',
                scope: 'global',
                serviceType: 'wallet',
                category: 'wallet_config',
                value: { transferFee, withdrawalFee, minimumFee },
                isDeleted: false,
            });
            res.json({
                success: true,
                message: 'Fees updated successfully',
                data: { transferFee, withdrawalFee, minimumFee },
            });
        }),
    ];
    static getWalletStats = (0, errors_1.asyncHandler)(async (_req, res) => {
        const stats = {
            totalWallets: 1250,
            totalBalance: 15000000,
            activeWallets: 1100,
            totalTransactions: 45000,
            averageTransactionAmount: 2500,
            topCurrencies: [
                { currency: 'NGN', count: 1200, totalBalance: 14000000 },
                { currency: 'USD', count: 50, totalBalance: 100000 },
            ],
        };
        res.json({
            success: true,
            data: stats,
        });
    });
}
exports.WalletConfigController = WalletConfigController;
