"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VerificationProviderController = void 0;
const zod_1 = require("zod");
const errors_1 = require("../../../core/errors");
const service_configuration_repo_1 = require("../../../database/repositories/service-configuration.repo");
const provider_manager_1 = require("../../../integrations/verification/provider-manager");
const validateRequest = (schema) => (req, res, next) => {
    try {
        const result = schema.parse({ body: req.body, params: req.params, query: req.query });
        req.body = result.body;
        req.params = result.params;
        req.query = result.query;
        next();
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            res.status(400).json({
                success: false,
                message: 'Validation error',
                errors: error.errors,
            });
            return;
        }
        next(error);
    }
};
const setProviderSchema = zod_1.z.object({
    body: zod_1.z.object({
        provider: zod_1.z.enum(['unify', 'mock', 's8v']),
        config: zod_1.z.object({
            baseUrl: zod_1.z.string().url().optional(),
            apiKey: zod_1.z.string().min(1).optional(),
            timeout: zod_1.z.number().min(1000).max(120000).optional(),
            retries: zod_1.z.number().min(1).max(10).optional(),
        }).optional(),
    }),
});
const updateProviderConfigSchema = zod_1.z.object({
    params: zod_1.z.object({
        provider: zod_1.z.enum(['unify', 'mock', 's8v']),
    }),
    body: zod_1.z.object({
        baseUrl: zod_1.z.string().url().optional(),
        apiKey: zod_1.z.string().min(1).optional(),
        timeout: zod_1.z.number().min(1000).max(120000).optional(),
        retries: zod_1.z.number().min(1).max(10).optional(),
    }),
});
const testProviderSchema = zod_1.z.object({
    params: zod_1.z.object({
        provider: zod_1.z.enum(['unify', 'mock', 's8v']),
    }),
    body: zod_1.z.object({
        nin: zod_1.z.string().regex(/^\d{11}$/, 'NIN must be 11 digits'),
    }),
});
class VerificationProviderController {
    static getCurrentProvider = async (req, res) => {
        try {
            const currentProvider = provider_manager_1.verificationProviderManager.getCurrentProvider();
            const status = await provider_manager_1.verificationProviderManager.getProviderStatus();
            res.json({
                success: true,
                data: {
                    currentProvider,
                    status,
                },
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to get current provider', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static getConfigStatus = async (req, res) => {
        try {
            const subservices = [
                { service: 'bvn', sub: 'by_bvn', label: 'BVN by BVN' },
                { service: 'nin', sub: 'by_nin', label: 'NIN by NIN' },
                { service: 'nin', sub: 'by_phone', label: 'NIN by Phone' },
                { service: 'nin', sub: 'by_details', label: 'NIN by Details' },
            ];
            const results = {};
            for (const { service, sub, label } of subservices) {
                const nameKey = `verification_${service}_${sub}_provider`;
                const baseUrlKey = `verification_${service}_${sub}_base_url`;
                const apiKeyKey = `verification_${service}_${sub}_api_key`;
                const timeoutKey = `verification_${service}_${sub}_timeout`;
                const [nameCfg, baseUrlCfg, apiKeyCfg, timeoutCfg] = await Promise.all([
                    service_configuration_repo_1.serviceConfigurationRepository.getByKey(nameKey, 'service', { serviceType: 'verification' }),
                    service_configuration_repo_1.serviceConfigurationRepository.getByKey(baseUrlKey, 'service', { serviceType: 'verification' }),
                    service_configuration_repo_1.serviceConfigurationRepository.getByKey(apiKeyKey, 'service', { serviceType: 'verification' }),
                    service_configuration_repo_1.serviceConfigurationRepository.getByKey(timeoutKey, 'service', { serviceType: 'verification' }),
                ]);
                const fallbackProvider = provider_manager_1.verificationProviderManager.isSandboxMode() ? 'mock' : 'unify';
                const providerName = nameCfg?.value || fallbackProvider;
                results[`${service}:${sub}`] = {
                    label,
                    keys: {
                        provider: nameKey,
                        baseUrl: baseUrlKey,
                        apiKey: apiKeyKey,
                        timeout: timeoutKey,
                    },
                    present: {
                        provider: Boolean(nameCfg?.value),
                        baseUrl: Boolean(baseUrlCfg?.value),
                        apiKey: Boolean(apiKeyCfg?.value),
                        timeout: typeof timeoutCfg?.value === 'number',
                    },
                    values: {
                        provider: providerName,
                        baseUrl: baseUrlCfg?.value || null,
                        apiKeyPresent: Boolean(apiKeyCfg?.value),
                        timeout: typeof timeoutCfg?.value === 'number' ? timeoutCfg?.value : null,
                    },
                    effectiveProvider: providerName,
                    fallbackUsed: !nameCfg?.value,
                };
            }
            res.json({
                success: true,
                data: {
                    currentProvider: provider_manager_1.verificationProviderManager.getCurrentProviderName(),
                    sandboxMode: provider_manager_1.verificationProviderManager.isSandboxMode(),
                    subservices: results,
                },
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to get verification config status', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static getAvailableProviders = async (req, res) => {
        try {
            const providers = [
                {
                    name: 'unify',
                    displayName: 'Unify Verifier',
                    description: 'Primary NIN/BVN verification provider',
                    supportedServices: ['nin', 'bvn'],
                    isHealthy: true,
                },
                {
                    name: 's8v',
                    displayName: 'S8V Verification',
                    description: 'Alternative NIN verification provider',
                    supportedServices: ['nin'],
                    isHealthy: true,
                },
                {
                    name: 'mock',
                    displayName: 'Mock Provider',
                    description: 'Development and testing provider',
                    supportedServices: ['nin', 'bvn'],
                    isHealthy: true,
                },
            ];
            res.json({
                success: true,
                data: providers,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to get available providers', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static setProvider = [
        validateRequest(setProviderSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            try {
                const { provider, config } = req.body;
                if (config) {
                    if (config.baseUrl) {
                        await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                            key: `verification_${provider}_base_url`,
                            scope: 'service',
                            type: 'string',
                            serviceType: 'verification',
                            value: config.baseUrl,
                            name: `${provider.toUpperCase()} Base URL`,
                            description: `Base URL for ${provider} verification provider`,
                            category: 'verification_providers',
                            isPublic: false,
                            isEditable: true,
                        });
                    }
                    if (config.apiKey) {
                        await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                            key: `verification_${provider}_api_key`,
                            scope: 'service',
                            type: 'string',
                            serviceType: 'verification',
                            value: config.apiKey,
                            name: `${provider.toUpperCase()} API Key`,
                            description: `API key for ${provider} verification provider`,
                            category: 'verification_providers',
                            isPublic: false,
                            isEditable: true,
                        });
                    }
                    if (config.timeout) {
                        await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                            key: `verification_${provider}_timeout`,
                            scope: 'service',
                            type: 'number',
                            serviceType: 'verification',
                            value: config.timeout,
                            name: `${provider.toUpperCase()} Timeout`,
                            description: `Timeout for ${provider} verification provider`,
                            category: 'verification_providers',
                            isPublic: false,
                            isEditable: true,
                        });
                    }
                    if (config.retries) {
                        await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                            key: `verification_${provider}_retries`,
                            scope: 'service',
                            type: 'number',
                            serviceType: 'verification',
                            value: config.retries,
                            name: `${provider.toUpperCase()} Retries`,
                            description: `Retry count for ${provider} verification provider`,
                            category: 'verification_providers',
                            isPublic: false,
                            isEditable: true,
                        });
                    }
                }
                provider_manager_1.verificationProviderManager.setCurrentProvider(provider);
                await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key: 'verification_current_provider',
                    scope: 'service',
                    type: 'string',
                    serviceType: 'verification',
                    value: provider,
                    name: 'Current Verification Provider',
                    description: 'Currently active verification provider',
                    category: 'verification_providers',
                    isPublic: false,
                    isEditable: true,
                });
                res.json({
                    success: true,
                    message: `Verification provider switched to ${provider}`,
                    data: {
                        currentProvider: provider,
                        config,
                    },
                });
            }
            catch (error) {
                throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to set verification provider', error instanceof errors_1.AppError ? error.statusCode : 500);
            }
        })
    ];
    static updateProviderConfig = [
        validateRequest(updateProviderConfigSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            try {
                const { provider } = req.params;
                const config = req.body;
                provider_manager_1.verificationProviderManager.updateCurrentProviderConfig(config);
                for (const [key, value] of Object.entries(config)) {
                    if (value !== undefined) {
                        await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                            key: `verification_${provider}_${key}`,
                            scope: 'service',
                            type: typeof value === 'number' ? 'number' : 'string',
                            serviceType: 'verification',
                            value,
                            name: `${provider.toUpperCase()} ${key.charAt(0).toUpperCase() + key.slice(1)}`,
                            description: `${key} configuration for ${provider} verification provider`,
                            category: 'verification_providers',
                            isPublic: false,
                            isEditable: true,
                        });
                    }
                }
                res.json({
                    success: true,
                    message: `${provider} provider configuration updated`,
                    data: {
                        provider,
                        config,
                    },
                });
            }
            catch (error) {
                throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to update provider configuration', error instanceof errors_1.AppError ? error.statusCode : 500);
            }
        })
    ];
    static testProvider = [
        validateRequest(testProviderSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            try {
                const { provider } = req.params;
                const { nin } = req.body;
                const testProvider = provider_manager_1.verificationProviderManager.createProviderInstance(provider);
                const result = await testProvider.verifyNin({ nin });
                res.json({
                    success: true,
                    message: `${provider} provider test successful`,
                    data: {
                        provider,
                        testResult: result,
                    },
                });
            }
            catch (error) {
                res.status(400).json({
                    success: false,
                    message: `Provider test failed: ${error instanceof Error ? error.message : 'Unknown error'}`,
                    data: {
                        provider: req.params.provider,
                        error: error instanceof Error ? error.message : 'Unknown error',
                    },
                });
            }
        })
    ];
    static getProviderHealth = async (req, res) => {
        try {
            const { provider } = req.params;
            if (!['unify', 'mock', 's8v'].includes(provider)) {
                throw new errors_1.AppError('Invalid provider', 400);
            }
            const testProvider = provider_manager_1.verificationProviderManager.createProviderInstance(provider);
            const isHealthy = await testProvider.healthCheck();
            res.json({
                success: true,
                data: {
                    provider,
                    isHealthy,
                    timestamp: new Date().toISOString(),
                },
            });
        }
        catch (error) {
            res.status(400).json({
                success: false,
                message: `Health check failed: ${error instanceof Error ? error.message : 'Unknown error'}`,
                data: {
                    provider: req.params.provider,
                    isHealthy: false,
                    error: error instanceof Error ? error.message : 'Unknown error',
                },
            });
        }
    };
    static verifyProviderCredentials = (0, errors_1.asyncHandler)(async (req, res) => {
        const { provider, subService } = req.params;
        if (!provider || !subService) {
            return res.status(400).json({
                success: false,
                message: 'Provider and subService are required'
            });
        }
        try {
            const baseUrlKey = `verification_${subService}_base_url`;
            const apiKeyKey = `verification_${subService}_api_key`;
            const baseUrl = await service_configuration_repo_1.serviceConfigurationRepository.findOne({
                key: baseUrlKey,
                scope: 'service',
                serviceType: 'verification'
            });
            const apiKey = await service_configuration_repo_1.serviceConfigurationRepository.findOne({
                key: apiKeyKey,
                scope: 'service',
                serviceType: 'verification'
            });
            if (!baseUrl || !apiKey) {
                return res.status(404).json({
                    success: false,
                    message: 'Credentials not found for this provider/subservice combination',
                    data: {
                        provider,
                        subService,
                        baseUrlFound: !!baseUrl,
                        apiKeyFound: !!apiKey
                    }
                });
            }
            const axios = require('axios');
            const startTime = Date.now();
            try {
                const testUrl = `${baseUrl.value}/pricing`;
                const response = await axios.get(testUrl, {
                    headers: {
                        'Authorization': `Bearer ${apiKey.value}`,
                        'Content-Type': 'application/json'
                    },
                    timeout: 10000,
                    validateStatus: (status) => status < 500
                });
                const responseTime = Date.now() - startTime;
                res.json({
                    success: true,
                    message: 'Credentials verification completed',
                    data: {
                        provider,
                        subService,
                        baseUrl: baseUrl.value,
                        apiKey: `${apiKey.value.substring(0, 8)}...`,
                        isValid: response.status === 200,
                        status: response.status,
                        responseTime,
                        error: response.status !== 200 ? `HTTP ${response.status}` : null
                    }
                });
            }
            catch (error) {
                const responseTime = Date.now() - startTime;
                res.json({
                    success: true,
                    message: 'Credentials verification completed',
                    data: {
                        provider,
                        subService,
                        baseUrl: baseUrl.value,
                        apiKey: `${apiKey.value.substring(0, 8)}...`,
                        isValid: false,
                        responseTime,
                        error: error.message
                    }
                });
            }
        }
        catch (error) {
            res.status(500).json({
                success: false,
                message: 'Failed to verify credentials',
                error: error.message
            });
        }
    });
}
exports.VerificationProviderController = VerificationProviderController;
