"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VerificationPricingController = void 0;
const service_configuration_repo_1 = require("../../../database/repositories/service-configuration.repo");
const CONFIG_CATEGORY = 'verification_pricing';
const SERVICE_TYPE = 'verification';
function buildConfigKey(service) {
    return `verification_price_${service}`;
}
function buildSubServiceConfigKey(subService) {
    return `verification_price_${subService}`;
}
const SUB_SERVICE_MAPPING = {
    'nin_search_by_nin': 'nin',
    'nin_search_by_name_dob': 'nin',
    'nin_search_by_phone': 'nin',
    'nin_validation': 'nin',
    'bvn_verification': 'bvn',
    'bvn_by_phone': 'bvn'
};
class VerificationPricingController {
    static async list(_req, res) {
        const configs = await service_configuration_repo_1.serviceConfigurationRepository.listByCategory(CONFIG_CATEGORY);
        const result = {};
        const subServices = {};
        for (const c of configs) {
            if (typeof c.value === 'number' && c.key.startsWith('verification_price_')) {
                const name = c.key.replace('verification_price_', '');
                if (Object.keys(SUB_SERVICE_MAPPING).includes(name)) {
                    const mainService = SUB_SERVICE_MAPPING[name];
                    if (!subServices[mainService]) {
                        subServices[mainService] = {};
                    }
                    subServices[mainService][name] = c.value;
                }
                else {
                    result[name] = c.value;
                }
            }
        }
        res.json({
            success: true,
            data: result,
            subServices: subServices
        });
    }
    static async getByService(req, res) {
        const service = req.params['service'];
        const key = buildConfigKey(service);
        const cfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(key, 'service', { serviceType: SERVICE_TYPE });
        res.json({ success: true, data: { service, price: cfg?.value ?? null } });
    }
    static async upsert(req, res) {
        const service = req.params['service'];
        const { price } = req.body;
        if (typeof price !== 'number' || !Number.isFinite(price) || price < 0) {
            res.status(400).json({ success: false, message: 'price must be a non-negative number' });
            return;
        }
        const key = buildConfigKey(service);
        const saved = await service_configuration_repo_1.serviceConfigurationRepository.upsert({
            key,
            scope: 'service',
            type: 'number',
            serviceType: SERVICE_TYPE,
            value: Math.round(price * 100) / 100,
            defaultValue: undefined,
            name: `${service.toUpperCase()} verification price`,
            description: `Customer selling price for ${service.toUpperCase()} verification`,
            category: CONFIG_CATEGORY,
            validation: { required: true, min: 0 },
            isPublic: false,
            isEditable: true,
        });
        res.json({ success: true, data: { service, price: saved.value } });
    }
    static async listSubServices(_req, res) {
        const configs = await service_configuration_repo_1.serviceConfigurationRepository.listByCategory(CONFIG_CATEGORY);
        const result = {};
        for (const c of configs) {
            if (typeof c.value === 'number' && c.key.startsWith('verification_price_')) {
                const name = c.key.replace('verification_price_', '');
                if (Object.keys(SUB_SERVICE_MAPPING).includes(name)) {
                    result[name] = c.value;
                }
            }
        }
        res.json({ success: true, data: result });
    }
    static async getBySubService(req, res) {
        const subService = req.params['subService'];
        const key = buildSubServiceConfigKey(subService);
        const cfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(key, 'service', { serviceType: SERVICE_TYPE });
        res.json({ success: true, data: { subService, price: cfg?.value ?? null } });
    }
    static async upsertSubService(req, res) {
        const subService = req.params['subService'];
        const { price } = req.body;
        if (typeof price !== 'number' || !Number.isFinite(price) || price < 0) {
            res.status(400).json({ success: false, message: 'price must be a non-negative number' });
            return;
        }
        const key = buildSubServiceConfigKey(subService);
        const mainService = SUB_SERVICE_MAPPING[subService];
        const saved = await service_configuration_repo_1.serviceConfigurationRepository.upsert({
            key,
            scope: 'service',
            type: 'number',
            serviceType: SERVICE_TYPE,
            value: Math.round(price * 100) / 100,
            defaultValue: undefined,
            name: `${subService.replace(/_/g, ' ').toUpperCase()} verification price`,
            description: `Customer selling price for ${subService.replace(/_/g, ' ')} verification`,
            category: CONFIG_CATEGORY,
            validation: { required: true, min: 0 },
            isPublic: false,
            isEditable: true,
        });
        res.json({ success: true, data: { subService, mainService, price: saved.value } });
    }
    static async bulkUpdateSubServices(req, res) {
        const { prices } = req.body;
        if (!prices || typeof prices !== 'object') {
            res.status(400).json({ success: false, message: 'prices must be an object' });
            return;
        }
        const results = {};
        const errors = [];
        for (const [subService, price] of Object.entries(prices)) {
            if (!Object.keys(SUB_SERVICE_MAPPING).includes(subService)) {
                errors.push(`Invalid subservice: ${subService}`);
                continue;
            }
            if (typeof price !== 'number' || !Number.isFinite(price) || price < 0) {
                errors.push(`Invalid price for ${subService}: ${price}`);
                continue;
            }
            try {
                const key = buildSubServiceConfigKey(subService);
                const saved = await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key,
                    scope: 'service',
                    type: 'number',
                    serviceType: SERVICE_TYPE,
                    value: Math.round(price * 100) / 100,
                    defaultValue: undefined,
                    name: `${subService.replace(/_/g, ' ').toUpperCase()} verification price`,
                    description: `Customer selling price for ${subService.replace(/_/g, ' ')} verification`,
                    category: CONFIG_CATEGORY,
                    validation: { required: true, min: 0 },
                    isPublic: false,
                    isEditable: true,
                });
                results[subService] = saved.value;
            }
            catch (error) {
                errors.push(`Failed to update ${subService}: ${error.message}`);
            }
        }
        res.json({
            success: errors.length === 0,
            data: results,
            errors: errors.length > 0 ? errors : undefined
        });
    }
}
exports.VerificationPricingController = VerificationPricingController;
