"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VatConfigController = void 0;
const service_configuration_repo_1 = require("../../../database/repositories/service-configuration.repo");
const VAT_KEY = 'vat_rate';
const VAT_CATEGORY = 'taxes';
class VatConfigController {
    static async list(_req, res) {
        const items = await service_configuration_repo_1.serviceConfigurationRepository.listByCategory(VAT_CATEGORY);
        return res.json({ success: true, data: items });
    }
    static async getByServiceCategory(req, res) {
        const serviceCategory = (req.params.serviceCategory || '').toLowerCase();
        const cfg = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(VAT_KEY, 'service', { serviceType: serviceCategory });
        return res.json({ success: true, data: cfg });
    }
    static async createOrUpdate(req, res) {
        const { serviceCategory, rate, enabled } = req.body;
        if (!serviceCategory)
            return res.status(400).json({ success: false, message: 'serviceCategory is required' });
        if (rate == null || isNaN(Number(rate)) || Number(rate) < 0) {
            return res.status(400).json({ success: false, message: 'rate must be a non-negative number (e.g., 0.075 for 7.5%)' });
        }
        const doc = await service_configuration_repo_1.serviceConfigurationRepository.upsert({
            key: VAT_KEY,
            scope: 'service',
            type: 'object',
            serviceType: String(serviceCategory).toLowerCase(),
            value: { rate: Number(rate), enabled: Boolean(enabled) },
            defaultValue: { rate: 0, enabled: false },
            name: `VAT for ${serviceCategory}`,
            description: 'Configurable VAT rate per service category',
            category: VAT_CATEGORY,
            validation: { min: 0, max: 1 },
            isPublic: false,
            isEditable: true,
        });
        return res.status(200).json({ success: true, data: doc });
    }
    static async disable(req, res) {
        const serviceCategory = (req.params.serviceCategory || '').toLowerCase();
        const existing = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(VAT_KEY, 'service', { serviceType: serviceCategory });
        if (!existing)
            return res.status(404).json({ success: false, message: 'VAT config not found' });
        const doc = await service_configuration_repo_1.serviceConfigurationRepository.upsert({
            key: VAT_KEY,
            scope: 'service',
            type: 'object',
            serviceType: serviceCategory,
            value: { ...(existing.value || {}), enabled: false },
            defaultValue: existing.defaultValue ?? { rate: 0, enabled: false },
            name: existing.name,
            description: existing.description,
            category: VAT_CATEGORY,
            validation: existing.validation,
            isPublic: existing.isPublic,
            isEditable: existing.isEditable,
            providerId: existing.providerId,
            planId: existing.planId,
            lastModifiedBy: undefined,
            isDeleted: false,
        });
        return res.json({ success: true, data: doc });
    }
}
exports.VatConfigController = VatConfigController;
