"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceProviderController = void 0;
const zod_1 = require("zod");
const service_provider_repo_1 = require("../../../database/repositories/service-provider.repo");
const errors_1 = require("../../../core/errors");
const validate_1 = require("../../../core/validate");
const validateRequest = (schema) => (0, validate_1.validate)(schema, 'body');
const createServiceProviderSchema = zod_1.z.object({
    body: zod_1.z.object({
        name: zod_1.z.string().min(1).max(100),
        slug: zod_1.z.string().min(1).max(50).regex(/^[a-z0-9-]+$/),
        displayName: zod_1.z.string().min(1).max(100),
        serviceType: zod_1.z.enum(['vtu', 'bills', 'verification', 'payment']),
        baseUrl: zod_1.z.string().url(),
        apiKey: zod_1.z.string().min(1),
        secretKey: zod_1.z.string().optional(),
        webhookUrl: zod_1.z.string().url().optional(),
        timeout: zod_1.z.number().min(1000).max(120000).optional(),
        retryAttempts: zod_1.z.number().min(1).max(10).optional(),
        sandboxMode: zod_1.z.boolean().optional(),
        priority: zod_1.z.number().min(1).max(100).optional(),
        defaultMarkup: zod_1.z.number().min(0).max(100).optional(),
        supportedServices: zod_1.z.array(zod_1.z.string()).min(1),
        supportedNetworks: zod_1.z.array(zod_1.z.string()).optional(),
        rateLimitPerMinute: zod_1.z.number().min(1).optional(),
        rateLimitPerHour: zod_1.z.number().min(1).optional(),
        minimumBalance: zod_1.z.number().min(0).optional(),
    }),
});
const updateServiceProviderSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().regex(/^[0-9a-fA-F]{24}$/),
    }),
    body: zod_1.z.object({
        name: zod_1.z.string().min(1).max(100).optional(),
        displayName: zod_1.z.string().min(1).max(100).optional(),
        status: zod_1.z.enum(['active', 'inactive', 'maintenance']).optional(),
        baseUrl: zod_1.z.string().url().optional(),
        apiKey: zod_1.z.string().min(1).optional(),
        secretKey: zod_1.z.string().optional(),
        webhookUrl: zod_1.z.string().url().optional(),
        timeout: zod_1.z.number().min(1000).max(120000).optional(),
        retryAttempts: zod_1.z.number().min(1).max(10).optional(),
        sandboxMode: zod_1.z.boolean().optional(),
        priority: zod_1.z.number().min(1).max(100).optional(),
        defaultMarkup: zod_1.z.number().min(0).max(100).optional(),
        supportedServices: zod_1.z.array(zod_1.z.string()).optional(),
        supportedNetworks: zod_1.z.array(zod_1.z.string()).optional(),
        rateLimitPerMinute: zod_1.z.number().min(1).optional(),
        rateLimitPerHour: zod_1.z.number().min(1).optional(),
        minimumBalance: zod_1.z.number().min(0).optional(),
    }),
});
const queryServiceProvidersSchema = zod_1.z.object({
    query: zod_1.z.object({
        serviceType: zod_1.z.enum(['vtu', 'bills', 'verification', 'payment']).optional(),
        status: zod_1.z.enum(['active', 'inactive', 'maintenance']).optional(),
        isHealthy: zod_1.z.string().transform(val => val === 'true').optional(),
        sandboxMode: zod_1.z.string().transform(val => val === 'true').optional(),
        search: zod_1.z.string().optional(),
        page: zod_1.z.string().transform(val => parseInt(val) || 1).optional(),
        limit: zod_1.z.string().transform(val => Math.min(parseInt(val) || 20, 100)).optional(),
    }),
});
class ServiceProviderController {
    static createProvider = [
        validateRequest(createServiceProviderSchema),
        async (req, res) => {
            try {
                const data = req.body;
                const existingProvider = await service_provider_repo_1.serviceProviderRepository.findBySlug(data.slug);
                if (existingProvider) {
                    throw new errors_1.AppError('Provider with this slug already exists', 400);
                }
                const provider = await service_provider_repo_1.serviceProviderRepository.create(data);
                res.status(201).json({
                    success: true,
                    message: 'Service provider created successfully',
                    data: provider,
                });
            }
            catch (error) {
                throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to create service provider', error instanceof errors_1.AppError ? error.statusCode : 500);
            }
        },
    ];
    static getProviders = [
        validateRequest(queryServiceProvidersSchema),
        async (req, res) => {
            try {
                const { serviceType, status, isHealthy, sandboxMode, search, page = 1, limit = 20 } = req.query;
                let providers;
                if (search) {
                    providers = await service_provider_repo_1.serviceProviderRepository.searchProviders(search, {
                        serviceType,
                        status,
                        isHealthy,
                        sandboxMode,
                    });
                }
                else {
                    const filters = {};
                    if (serviceType)
                        filters.serviceType = serviceType;
                    if (status)
                        filters.status = status;
                    if (isHealthy !== undefined)
                        filters.isHealthy = isHealthy;
                    if (sandboxMode !== undefined)
                        filters.sandboxMode = sandboxMode;
                    providers = await service_provider_repo_1.serviceProviderRepository.find(filters);
                }
                const startIndex = (page - 1) * limit;
                const endIndex = startIndex + limit;
                const paginatedProviders = providers.slice(startIndex, endIndex);
                const stats = await service_provider_repo_1.serviceProviderRepository.getProviderStats();
                res.json({
                    success: true,
                    data: {
                        providers: paginatedProviders,
                        pagination: {
                            page,
                            limit,
                            total: providers.length,
                            pages: Math.ceil(providers.length / limit),
                        },
                        stats,
                    },
                });
            }
            catch (error) {
                throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch service providers', 500);
            }
        },
    ];
    static getProvider = async (req, res) => {
        try {
            const { id } = req.params;
            const provider = await service_provider_repo_1.serviceProviderRepository.findById(id);
            if (!provider) {
                throw new errors_1.AppError('Service provider not found', 404);
            }
            res.json({
                success: true,
                data: provider,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch service provider', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static updateProvider = [
        validateRequest(updateServiceProviderSchema),
        async (req, res) => {
            try {
                const { id } = req.params;
                const updateData = req.body;
                const provider = await service_provider_repo_1.serviceProviderRepository.updateById(id, updateData);
                if (!provider) {
                    throw new errors_1.AppError('Service provider not found', 404);
                }
                res.json({
                    success: true,
                    message: 'Service provider updated successfully',
                    data: provider,
                });
            }
            catch (error) {
                throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to update service provider', error instanceof errors_1.AppError ? error.statusCode : 500);
            }
        },
    ];
    static deleteProvider = async (req, res) => {
        try {
            const { id } = req.params;
            const provider = await service_provider_repo_1.serviceProviderRepository.softDelete(id);
            if (!provider) {
                throw new errors_1.AppError('Service provider not found', 404);
            }
            res.json({
                success: true,
                message: 'Service provider deleted successfully',
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to delete service provider', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static testProviderHealth = async (req, res) => {
        try {
            const { id } = req.params;
            const provider = await service_provider_repo_1.serviceProviderRepository.findById(id);
            if (!provider) {
                throw new errors_1.AppError('Service provider not found', 404);
            }
            const healthStatus = {
                isHealthy: true,
                responseTime: Math.random() * 1000,
                lastCheck: new Date(),
                message: 'Health check successful',
            };
            await service_provider_repo_1.serviceProviderRepository.updateHealthStatus(id, {
                isHealthy: healthStatus.isHealthy,
                lastHealthCheck: healthStatus.lastCheck,
            });
            res.json({
                success: true,
                data: healthStatus,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to test provider health', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static getProvidersByServiceType = async (req, res) => {
        try {
            const { serviceType } = req.params;
            if (!['vtu', 'bills', 'verification', 'payment'].includes(serviceType)) {
                throw new errors_1.AppError('Invalid service type', 400);
            }
            const providers = await service_provider_repo_1.serviceProviderRepository.findByServiceType(serviceType);
            res.json({
                success: true,
                data: providers,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch providers by service type', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static getActiveProviders = async (req, res) => {
        try {
            const { serviceType } = req.params;
            if (!['vtu', 'bills', 'verification', 'payment'].includes(serviceType)) {
                throw new errors_1.AppError('Invalid service type', 400);
            }
            const providers = await service_provider_repo_1.serviceProviderRepository.findActiveProviders(serviceType);
            res.json({
                success: true,
                data: providers,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch active providers', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static updateProviderPriority = async (req, res) => {
        try {
            const { id } = req.params;
            const { priority } = req.body;
            if (!priority || priority < 1 || priority > 100) {
                throw new errors_1.AppError('Priority must be between 1 and 100', 400);
            }
            const provider = await service_provider_repo_1.serviceProviderRepository.updateById(id, { priority });
            if (!provider) {
                throw new errors_1.AppError('Service provider not found', 404);
            }
            res.json({
                success: true,
                message: 'Provider priority updated successfully',
                data: provider,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to update provider priority', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static toggleProviderStatus = async (req, res) => {
        try {
            const { id } = req.params;
            const provider = await service_provider_repo_1.serviceProviderRepository.findById(id);
            if (!provider) {
                throw new errors_1.AppError('Service provider not found', 404);
            }
            const newStatus = provider.status === 'active' ? 'inactive' : 'active';
            const updatedProvider = await service_provider_repo_1.serviceProviderRepository.updateById(id, { status: newStatus });
            res.json({
                success: true,
                message: `Provider ${newStatus === 'active' ? 'activated' : 'deactivated'} successfully`,
                data: updatedProvider,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to toggle provider status', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
}
exports.ServiceProviderController = ServiceProviderController;
