"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServicePlanController = void 0;
const zod_1 = require("zod");
const service_plan_repo_1 = require("../../../database/repositories/service-plan.repo");
const service_provider_repo_1 = require("../../../database/repositories/service-provider.repo");
const errors_1 = require("../../../core/errors");
const validate_1 = require("../../../core/validate");
const validateRequest = (schema) => (0, validate_1.validate)(schema, 'body');
const createServicePlanSchema = zod_1.z.object({
    body: zod_1.z.object({
        providerId: zod_1.z.string().regex(/^[0-9a-fA-F]{24}$/),
        planId: zod_1.z.string().min(1).max(100),
        vendorPlanId: zod_1.z.string().optional(),
        name: zod_1.z.string().min(1).max(200),
        description: zod_1.z.string().optional(),
        type: zod_1.z.enum(['data', 'airtime', 'electricity', 'cable', 'education', 'exam-pin']),
        category: zod_1.z.string().optional(),
        network: zod_1.z.enum(['mtn', 'airtel', 'glo', '9mobile']).optional(),
        dataPlanType: zod_1.z.enum(['sme', 'sme2', 'corporate', 'awoof', 'gifting']).optional(),
        size: zod_1.z.string().optional(),
        validity: zod_1.z.string().optional(),
        amount: zod_1.z.number().min(0),
        sellingPrice: zod_1.z.number().min(0),
        status: zod_1.z.enum(['active', 'inactive', 'out-of-stock']).optional(),
        isVisible: zod_1.z.boolean().optional(),
        stockQuantity: zod_1.z.number().min(0).optional(),
        minPurchase: zod_1.z.number().min(0).optional(),
        maxPurchase: zod_1.z.number().min(0).optional(),
        dailyLimit: zod_1.z.number().min(0).optional(),
        metadata: zod_1.z.record(zod_1.z.any()).optional(),
    }),
});
const updateServicePlanSchema = zod_1.z.object({
    params: zod_1.z.object({
        id: zod_1.z.string().regex(/^[0-9a-fA-F]{24}$/),
    }),
    body: zod_1.z.object({
        vendorPlanId: zod_1.z.string().optional(),
        name: zod_1.z.string().min(1).max(200).optional(),
        description: zod_1.z.string().optional(),
        category: zod_1.z.string().optional(),
        size: zod_1.z.string().optional(),
        validity: zod_1.z.string().optional(),
        amount: zod_1.z.number().min(0).optional(),
        sellingPrice: zod_1.z.number().min(0).optional(),
        status: zod_1.z.enum(['active', 'inactive', 'out-of-stock']).optional(),
        isVisible: zod_1.z.boolean().optional(),
        stockQuantity: zod_1.z.number().min(0).optional(),
        minPurchase: zod_1.z.number().min(0).optional(),
        maxPurchase: zod_1.z.number().min(0).optional(),
        dailyLimit: zod_1.z.number().min(0).optional(),
        metadata: zod_1.z.record(zod_1.z.any()).optional(),
    }),
});
const queryServicePlansSchema = zod_1.z.object({
    query: zod_1.z.object({
        type: zod_1.z.enum(['data', 'airtime', 'electricity', 'cable', 'education', 'exam-pin']).optional(),
        network: zod_1.z.enum(['mtn', 'airtel', 'glo', '9mobile']).optional(),
        status: zod_1.z.enum(['active', 'inactive', 'out-of-stock']).optional(),
        isVisible: zod_1.z.string().transform(val => val === 'true').optional(),
        providerId: zod_1.z.string().regex(/^[0-9a-fA-F]{24}$/).optional(),
        minPrice: zod_1.z.string().transform(val => parseFloat(val)).optional(),
        maxPrice: zod_1.z.string().transform(val => parseFloat(val)).optional(),
        category: zod_1.z.string().optional(),
        search: zod_1.z.string().optional(),
        page: zod_1.z.string().transform(val => parseInt(val) || 1).optional(),
        limit: zod_1.z.string().transform(val => Math.min(parseInt(val) || 20, 100)).optional(),
    }),
});
class ServicePlanController {
    static createPlan = [
        validateRequest(createServicePlanSchema),
        async (req, res) => {
            try {
                const data = req.body;
                const provider = await service_provider_repo_1.serviceProviderRepository.findById(data.providerId);
                if (!provider) {
                    throw new errors_1.AppError('Service provider not found', 404);
                }
                const existingPlan = await service_plan_repo_1.servicePlanRepository.findByPlanId(data.planId, data.providerId);
                if (existingPlan) {
                    throw new errors_1.AppError('Plan with this ID already exists for this provider', 400);
                }
                const plan = await service_plan_repo_1.servicePlanRepository.create(data);
                res.status(201).json({
                    success: true,
                    message: 'Service plan created successfully',
                    data: plan,
                });
            }
            catch (error) {
                throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to create service plan', error instanceof errors_1.AppError ? error.statusCode : 500);
            }
        },
    ];
    static getPlans = [
        validateRequest(queryServicePlansSchema),
        async (req, res) => {
            try {
                const { type, network, status, isVisible, providerId, minPrice, maxPrice, category, search, page = 1, limit = 20 } = req.query;
                let plans;
                if (search) {
                    plans = await service_plan_repo_1.servicePlanRepository.searchPlans(search, {
                        type, network, status, isVisible, providerId,
                        minPrice, maxPrice, category,
                    });
                }
                else {
                    const filters = {};
                    if (type)
                        filters.type = type;
                    if (network)
                        filters.network = network;
                    if (status)
                        filters.status = status;
                    if (isVisible !== undefined)
                        filters.isVisible = isVisible;
                    if (providerId)
                        filters.providerId = providerId;
                    if (minPrice)
                        filters.minPrice = minPrice;
                    if (maxPrice)
                        filters.maxPrice = maxPrice;
                    if (category)
                        filters.category = category;
                    plans = await service_plan_repo_1.servicePlanRepository.findActivePlans(filters);
                }
                const startIndex = (page - 1) * limit;
                const endIndex = startIndex + limit;
                const paginatedPlans = plans.slice(startIndex, endIndex);
                const stats = await service_plan_repo_1.servicePlanRepository.getPlanStats();
                res.json({
                    success: true,
                    data: {
                        plans: paginatedPlans,
                        pagination: {
                            page,
                            limit,
                            total: plans.length,
                            pages: Math.ceil(plans.length / limit),
                        },
                        stats,
                    },
                });
            }
            catch (error) {
                throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch service plans', 500);
            }
        },
    ];
    static getPlan = async (req, res) => {
        try {
            const { id } = req.params;
            const plan = await service_plan_repo_1.servicePlanRepository.findById(id);
            if (!plan) {
                throw new errors_1.AppError('Service plan not found', 404);
            }
            res.json({
                success: true,
                data: plan,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch service plan', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static updatePlan = [
        validateRequest(updateServicePlanSchema),
        async (req, res) => {
            try {
                const { id } = req.params;
                const updateData = req.body;
                const plan = await service_plan_repo_1.servicePlanRepository.updateById(id, updateData);
                if (!plan) {
                    throw new errors_1.AppError('Service plan not found', 404);
                }
                res.json({
                    success: true,
                    message: 'Service plan updated successfully',
                    data: plan,
                });
            }
            catch (error) {
                throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to update service plan', error instanceof errors_1.AppError ? error.statusCode : 500);
            }
        },
    ];
    static deletePlan = async (req, res) => {
        try {
            const { id } = req.params;
            const plan = await service_plan_repo_1.servicePlanRepository.softDelete(id);
            if (!plan) {
                throw new errors_1.AppError('Service plan not found', 404);
            }
            res.json({
                success: true,
                message: 'Service plan deleted successfully',
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to delete service plan', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static getPlansByType = async (req, res) => {
        try {
            const { type } = req.params;
            if (!['data', 'airtime', 'electricity', 'cable', 'education', 'exam-pin'].includes(type)) {
                throw new errors_1.AppError('Invalid plan type', 400);
            }
            const plans = await service_plan_repo_1.servicePlanRepository.findByType(type);
            res.json({
                success: true,
                data: plans,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch plans by type', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static getDataPlansByNetwork = async (req, res) => {
        try {
            const { network } = req.params;
            const { dataPlanType } = req.query;
            if (!['mtn', 'airtel', 'glo', '9mobile'].includes(network)) {
                throw new errors_1.AppError('Invalid network', 400);
            }
            const plans = await service_plan_repo_1.servicePlanRepository.findDataPlans(network, dataPlanType);
            res.json({
                success: true,
                data: plans,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch data plans by network', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static getElectricityPlans = async (req, res) => {
        try {
            const { discoName } = req.query;
            const plans = await service_plan_repo_1.servicePlanRepository.findElectricityPlans(discoName);
            res.json({
                success: true,
                data: plans,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch electricity plans', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static getCablePlans = async (req, res) => {
        try {
            const { cableProvider } = req.query;
            const plans = await service_plan_repo_1.servicePlanRepository.findCablePlans(cableProvider);
            res.json({
                success: true,
                data: plans,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to fetch cable plans', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static bulkUpdatePrices = async (req, res) => {
        try {
            const { providerId, markupPercentage } = req.body;
            if (!providerId || markupPercentage === undefined) {
                throw new errors_1.AppError('Provider ID and markup percentage are required', 400);
            }
            if (markupPercentage < 0 || markupPercentage > 100) {
                throw new errors_1.AppError('Markup percentage must be between 0 and 100', 400);
            }
            const provider = await service_provider_repo_1.serviceProviderRepository.findById(providerId);
            if (!provider) {
                throw new errors_1.AppError('Service provider not found', 404);
            }
            const result = await service_plan_repo_1.servicePlanRepository.bulkUpdatePrices(providerId, markupPercentage);
            res.json({
                success: true,
                message: `Updated prices for ${result.modifiedCount} plans`,
                data: { modifiedCount: result.modifiedCount },
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to bulk update prices', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static togglePlanVisibility = async (req, res) => {
        try {
            const { id } = req.params;
            const plan = await service_plan_repo_1.servicePlanRepository.findById(id);
            if (!plan) {
                throw new errors_1.AppError('Service plan not found', 404);
            }
            const updatedPlan = await service_plan_repo_1.servicePlanRepository.updateById(id, {
                isVisible: !plan.isVisible
            });
            res.json({
                success: true,
                message: `Plan ${updatedPlan?.isVisible ? 'shown' : 'hidden'} successfully`,
                data: updatedPlan,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to toggle plan visibility', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
    static togglePlanStatus = async (req, res) => {
        try {
            const { id } = req.params;
            const plan = await service_plan_repo_1.servicePlanRepository.findById(id);
            if (!plan) {
                throw new errors_1.AppError('Service plan not found', 404);
            }
            const newStatus = plan.status === 'active' ? 'inactive' : 'active';
            const updatedPlan = await service_plan_repo_1.servicePlanRepository.updateById(id, { status: newStatus });
            res.json({
                success: true,
                message: `Plan ${newStatus === 'active' ? 'activated' : 'deactivated'} successfully`,
                data: updatedPlan,
            });
        }
        catch (error) {
            throw new errors_1.AppError(error instanceof Error ? error.message : 'Failed to toggle plan status', error instanceof errors_1.AppError ? error.statusCode : 500);
        }
    };
}
exports.ServicePlanController = ServicePlanController;
