"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PaymentProviderController = void 0;
const zod_1 = require("zod");
const errors_1 = require("../../../core/errors");
const service_configuration_repo_1 = require("../../../database/repositories/service-configuration.repo");
const validateRequest = (schema) => (req, res, next) => {
    try {
        const result = schema.parse({ body: req.body, params: req.params, query: req.query });
        req.body = result.body;
        req.params = result.params;
        req.query = result.query;
        next();
    }
    catch (error) {
        next(new errors_1.AppError(error.errors ? error.errors[0].message : error.message, 400));
    }
};
const setPaymentProviderSchema = zod_1.z.object({
    body: zod_1.z.object({
        provider: zod_1.z.enum(['paystack', 'flutterwave', 'aspfiy']),
        config: zod_1.z.object({
            apiKey: zod_1.z.string().min(1),
            secretKey: zod_1.z.string().optional(),
            baseUrl: zod_1.z.string().url().optional(),
            webhookUrl: zod_1.z.string().url().optional(),
            timeout: zod_1.z.number().min(1000).max(120000).optional(),
            retryAttempts: zod_1.z.number().min(1).max(10).optional(),
            sandboxMode: zod_1.z.boolean().optional(),
            priority: zod_1.z.number().min(1).max(100).optional(),
        }),
    }),
});
const updatePaymentProviderConfigSchema = zod_1.z.object({
    params: zod_1.z.object({
        provider: zod_1.z.enum(['paystack', 'flutterwave', 'aspfiy']),
    }),
    body: zod_1.z.object({
        apiKey: zod_1.z.string().min(1).optional(),
        secretKey: zod_1.z.string().optional(),
        baseUrl: zod_1.z.string().url().optional(),
        webhookUrl: zod_1.z.string().url().optional(),
        timeout: zod_1.z.number().min(1000).max(120000).optional(),
        retryAttempts: zod_1.z.number().min(1).max(10).optional(),
        sandboxMode: zod_1.z.boolean().optional(),
        priority: zod_1.z.number().min(1).max(100).optional(),
    }),
});
const testPaymentProviderSchema = zod_1.z.object({
    params: zod_1.z.object({
        provider: zod_1.z.enum(['paystack', 'flutterwave', 'aspfiy']),
    }),
    body: zod_1.z.object({
        amount: zod_1.z.number().min(1).optional(),
        email: zod_1.z.string().email().optional(),
    }),
});
class PaymentProviderController {
    static getAvailableProviders = (0, errors_1.asyncHandler)(async (_req, res) => {
        const providers = [
            {
                name: 'paystack',
                displayName: 'Paystack',
                description: 'Paystack payment gateway',
                supportedCurrencies: ['NGN', 'USD'],
                supportedPaymentTypes: ['card', 'bank_transfer', 'ussd', 'qr'],
            },
            {
                name: 'flutterwave',
                displayName: 'Flutterwave',
                description: 'Flutterwave payment gateway',
                supportedCurrencies: ['NGN', 'USD', 'EUR', 'GBP'],
                supportedPaymentTypes: ['card', 'bank_transfer', 'ussd', 'qr', 'mobile_money'],
            },
            {
                name: 'aspfiy',
                displayName: 'Aspfiy',
                description: 'Aspfiy payment gateway - Reserved accounts and bank transfers',
                supportedCurrencies: ['NGN'],
                supportedPaymentTypes: ['bank_transfer', 'reserved_account', 'dynamic_account'],
            },
        ];
        res.json({ success: true, data: providers });
    });
    static getCurrentProvider = (0, errors_1.asyncHandler)(async (_req, res) => {
        const currentProvider = await service_configuration_repo_1.serviceConfigurationRepository.getByKey('payment_default_provider', 'service', { serviceType: 'payment' });
        const providerName = currentProvider?.value || 'paystack';
        const config = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(`payment_provider_config_${providerName}`, 'service', { serviceType: 'payment' });
        res.json({
            success: true,
            data: {
                name: providerName,
                config: config?.value || {},
            },
        });
    });
    static setProvider = [
        validateRequest(setPaymentProviderSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            const { provider, config } = req.body;
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: 'payment_default_provider',
                scope: 'service',
                serviceType: 'payment',
                category: 'payment_provider',
                value: provider,
                isDeleted: false,
            });
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: `payment_provider_config_${provider}`,
                scope: 'service',
                serviceType: 'payment',
                category: 'payment_provider',
                value: config,
                isDeleted: false,
            });
            res.json({
                success: true,
                message: `Payment provider switched to ${provider}`,
                data: { currentProvider: provider, config },
            });
        }),
    ];
    static updateProviderConfig = [
        validateRequest(updatePaymentProviderConfigSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            const { provider } = req.params;
            const config = req.body;
            const existingConfig = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(`payment_provider_config_${provider}`, 'service', { serviceType: 'payment' });
            const updatedConfig = {
                ...(existingConfig?.value || {}),
                ...config,
            };
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: `payment_provider_config_${provider}`,
                scope: 'service',
                serviceType: 'payment',
                category: 'payment_provider',
                value: updatedConfig,
                isDeleted: false,
            });
            res.json({
                success: true,
                message: `Configuration for ${provider} updated`,
                data: { provider, config: updatedConfig },
            });
        }),
    ];
    static testProvider = [
        validateRequest(testPaymentProviderSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            const { provider } = req.params;
            const { amount = 100, email = 'test@example.com' } = req.body;
            const config = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(`payment_provider_config_${provider}`, 'service', { serviceType: 'payment' });
            if (!config?.value) {
                throw new errors_1.AppError(`No configuration found for provider: ${provider}`, 404);
            }
            const testResult = {
                provider,
                status: 'healthy',
                responseTime: Math.random() * 1000,
                lastChecked: new Date().toISOString(),
                testAmount: amount,
                testEmail: email,
            };
            res.json({
                success: true,
                message: 'Provider test completed',
                data: testResult,
            });
        }),
    ];
    static getProviderHealth = (0, errors_1.asyncHandler)(async (req, res) => {
        const { provider } = req.params;
        const config = await service_configuration_repo_1.serviceConfigurationRepository.getByKey(`payment_provider_config_${provider}`, 'service', { serviceType: 'payment' });
        if (!config?.value) {
            throw new errors_1.AppError(`No configuration found for provider: ${provider}`, 404);
        }
        const isHealthy = Math.random() > 0.1;
        res.json({
            success: true,
            data: {
                provider,
                isHealthy,
                lastChecked: new Date().toISOString(),
                status: isHealthy ? 'online' : 'offline',
            },
        });
    });
}
exports.PaymentProviderController = PaymentProviderController;
