"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthConfigController = void 0;
const zod_1 = require("zod");
const errors_1 = require("../../../core/errors");
const service_configuration_repo_1 = require("../../../database/repositories/service-configuration.repo");
const validateRequest = (schema) => (req, res, next) => {
    try {
        const result = schema.parse({ body: req.body, params: req.params, query: req.query });
        req.body = result.body;
        req.params = result.params;
        req.query = result.query;
        next();
    }
    catch (error) {
        next(new errors_1.AppError(error.errors ? error.errors[0].message : error.message, 400));
    }
};
const updateAuthConfigSchema = zod_1.z.object({
    body: zod_1.z.object({
        jwt: zod_1.z.object({
            accessTokenExpiry: zod_1.z.string().optional(),
            refreshTokenExpiry: zod_1.z.string().optional(),
        }).optional(),
        rateLimiting: zod_1.z.object({
            loginAttempts: zod_1.z.number().min(1).max(100).optional(),
            windowMinutes: zod_1.z.number().min(1).max(1440).optional(),
            signupAttempts: zod_1.z.number().min(1).max(100).optional(),
        }).optional(),
        passwordPolicy: zod_1.z.object({
            minLength: zod_1.z.number().min(6).max(50).optional(),
            requireUppercase: zod_1.z.boolean().optional(),
            requireLowercase: zod_1.z.boolean().optional(),
            requireNumbers: zod_1.z.boolean().optional(),
            requireSpecialChars: zod_1.z.boolean().optional(),
            maxAge: zod_1.z.number().min(30).max(365).optional(),
        }).optional(),
        security: zod_1.z.object({
            requireEmailVerification: zod_1.z.boolean().optional(),
            requirePhoneVerification: zod_1.z.boolean().optional(),
            maxFailedLoginAttempts: zod_1.z.number().min(3).max(10).optional(),
            lockoutDuration: zod_1.z.number().min(5).max(60).optional(),
        }).optional(),
    }),
});
const setPasswordPolicySchema = zod_1.z.object({
    body: zod_1.z.object({
        minLength: zod_1.z.number().min(6).max(50),
        requireUppercase: zod_1.z.boolean(),
        requireLowercase: zod_1.z.boolean(),
        requireNumbers: zod_1.z.boolean(),
        requireSpecialChars: zod_1.z.boolean(),
        maxAge: zod_1.z.number().min(30).max(365),
    }),
});
const setRateLimitingSchema = zod_1.z.object({
    body: zod_1.z.object({
        loginAttempts: zod_1.z.number().min(1).max(100),
        windowMinutes: zod_1.z.number().min(1).max(1440),
        signupAttempts: zod_1.z.number().min(1).max(100),
    }),
});
class AuthConfigController {
    static getAuthConfig = (0, errors_1.asyncHandler)(async (_req, res) => {
        const configs = await service_configuration_repo_1.serviceConfigurationRepository.listByCategory('auth_config');
        const config = {
            jwt: {
                accessTokenExpiry: '15m',
                refreshTokenExpiry: '7d',
            },
            rateLimiting: {
                loginAttempts: 5,
                windowMinutes: 15,
                signupAttempts: 3,
            },
            passwordPolicy: {
                minLength: 8,
                requireUppercase: true,
                requireLowercase: true,
                requireNumbers: true,
                requireSpecialChars: false,
                maxAge: 90,
            },
            security: {
                requireEmailVerification: true,
                requirePhoneVerification: false,
                maxFailedLoginAttempts: 5,
                lockoutDuration: 15,
            },
        };
        for (const cfg of configs) {
            if (cfg.key === 'auth_jwt_config') {
                config.jwt = { ...config.jwt, ...cfg.value };
            }
            else if (cfg.key === 'auth_rate_limiting') {
                config.rateLimiting = { ...config.rateLimiting, ...cfg.value };
            }
            else if (cfg.key === 'auth_password_policy') {
                config.passwordPolicy = { ...config.passwordPolicy, ...cfg.value };
            }
            else if (cfg.key === 'auth_security') {
                config.security = { ...config.security, ...cfg.value };
            }
        }
        res.json({
            success: true,
            data: config,
        });
    });
    static updateAuthConfig = [
        validateRequest(updateAuthConfigSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            const { jwt, rateLimiting, passwordPolicy, security } = req.body;
            const updates = [];
            if (jwt) {
                updates.push(service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key: 'auth_jwt_config',
                    scope: 'global',
                    serviceType: 'auth',
                    category: 'auth_config',
                    value: jwt,
                    isDeleted: false,
                }));
            }
            if (rateLimiting) {
                updates.push(service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key: 'auth_rate_limiting',
                    scope: 'global',
                    serviceType: 'auth',
                    category: 'auth_config',
                    value: rateLimiting,
                    isDeleted: false,
                }));
            }
            if (passwordPolicy) {
                updates.push(service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key: 'auth_password_policy',
                    scope: 'global',
                    serviceType: 'auth',
                    category: 'auth_config',
                    value: passwordPolicy,
                    isDeleted: false,
                }));
            }
            if (security) {
                updates.push(service_configuration_repo_1.serviceConfigurationRepository.upsert({
                    key: 'auth_security',
                    scope: 'global',
                    serviceType: 'auth',
                    category: 'auth_config',
                    value: security,
                    isDeleted: false,
                }));
            }
            await Promise.all(updates);
            res.json({
                success: true,
                message: 'Authentication configuration updated successfully',
            });
        }),
    ];
    static setPasswordPolicy = [
        validateRequest(setPasswordPolicySchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            const passwordPolicy = req.body;
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: 'auth_password_policy',
                scope: 'global',
                serviceType: 'auth',
                category: 'auth_config',
                value: passwordPolicy,
                isDeleted: false,
            });
            res.json({
                success: true,
                message: 'Password policy updated successfully',
                data: passwordPolicy,
            });
        }),
    ];
    static setRateLimiting = [
        validateRequest(setRateLimitingSchema),
        (0, errors_1.asyncHandler)(async (req, res) => {
            const rateLimiting = req.body;
            await service_configuration_repo_1.serviceConfigurationRepository.upsert({
                key: 'auth_rate_limiting',
                scope: 'global',
                serviceType: 'auth',
                category: 'auth_config',
                value: rateLimiting,
                isDeleted: false,
            });
            res.json({
                success: true,
                message: 'Rate limiting updated successfully',
                data: rateLimiting,
            });
        }),
    ];
    static getAuthStats = (0, errors_1.asyncHandler)(async (_req, res) => {
        const stats = {
            totalUsers: 2500,
            activeUsers: 1800,
            verifiedUsers: 2200,
            totalLogins: 15000,
            failedLogins: 450,
            averageSessionDuration: 25,
            topLoginMethods: [
                { method: 'email', count: 12000 },
                { method: 'phone', count: 3000 },
            ],
        };
        res.json({
            success: true,
            data: stats,
        });
    });
}
exports.AuthConfigController = AuthConfigController;
