"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.initializeWorkers = exports.workerManager = exports.WorkerManager = void 0;
const logger_1 = require("../../config/logger");
const queue_1 = require("../../config/queue");
const reconciliation_processor_1 = require("../processors/reconciliation.processor");
const cleanup_processor_1 = require("../processors/cleanup.processor");
const reports_processor_1 = require("../processors/reports.processor");
const dispute_service_1 = require("../../modules/disputes/dispute.service");
const retention_scheduler_1 = require("../schedulers/retention.scheduler");
class WorkerManager {
    static instance;
    workers = new Map();
    isInitialized = false;
    constructor() { }
    static getInstance() {
        if (!WorkerManager.instance) {
            WorkerManager.instance = new WorkerManager();
        }
        return WorkerManager.instance;
    }
    async initializeWorkers() {
        if (this.isInitialized) {
            logger_1.logger.info('Worker manager already initialized');
            return;
        }
        logger_1.logger.info('Initializing worker manager...');
        try {
            await this.initializeReconciliationWorkers();
            await this.initializeCleanupWorkers();
            await this.initializeReportWorkers();
            await this.initializeDisputeWorkers();
            await this.initializeNotificationWorkers();
            await this.initializeDataRetentionWorkers();
            this.isInitialized = true;
            logger_1.logger.info('Worker manager initialized successfully');
        }
        catch (error) {
            logger_1.logger.error('Failed to initialize worker manager:', error);
            throw error;
        }
    }
    async initializeReconciliationWorkers() {
        logger_1.logger.info('Initializing reconciliation workers...');
        const worker = await queue_1.queueManager.getWorker(queue_1.QUEUE_NAMES.RECONCILIATION, async (job) => {
            try {
                const processor = (0, reconciliation_processor_1.getReconciliationProcessor)(job.name);
                return await processor(job);
            }
            catch (error) {
                logger_1.logger.error(`Reconciliation job ${job.id} failed:`, error);
                throw error;
            }
        });
        this.workers.set(queue_1.QUEUE_NAMES.RECONCILIATION, worker);
        logger_1.logger.info('Reconciliation workers initialized');
    }
    async initializeCleanupWorkers() {
        logger_1.logger.info('Initializing cleanup workers...');
        const worker = await queue_1.queueManager.getWorker(queue_1.QUEUE_NAMES.CLEANUP, async (job) => {
            try {
                const processor = (0, cleanup_processor_1.getCleanupProcessor)(job.name);
                return await processor(job);
            }
            catch (error) {
                logger_1.logger.error(`Cleanup job ${job.id} failed:`, error);
                throw error;
            }
        });
        this.workers.set(queue_1.QUEUE_NAMES.CLEANUP, worker);
        logger_1.logger.info('Cleanup workers initialized');
    }
    async initializeReportWorkers() {
        logger_1.logger.info('Initializing report workers...');
        const worker = await queue_1.queueManager.getWorker(queue_1.QUEUE_NAMES.REPORTS, async (job) => {
            try {
                const processor = (0, reports_processor_1.getReportProcessor)(job.name);
                return await processor(job);
            }
            catch (error) {
                logger_1.logger.error(`Report job ${job.id} failed:`, error);
                throw error;
            }
        });
        this.workers.set(queue_1.QUEUE_NAMES.REPORTS, worker);
        logger_1.logger.info('Report workers initialized');
    }
    async initializeDisputeWorkers() {
        logger_1.logger.info('Initializing dispute workers...');
        const worker = await queue_1.queueManager.getWorker(queue_1.QUEUE_NAMES.DISPUTE_ESCALATION, async (job) => {
            try {
                switch (job.name) {
                    case queue_1.JOB_TYPES.ESCALATE_URGENT_DISPUTES:
                        await dispute_service_1.disputeService.escalateUrgentDisputes();
                        return { success: true, message: 'Urgent disputes escalated' };
                    case queue_1.JOB_TYPES.DISPUTE_REMINDERS:
                        return { success: true, message: 'Dispute reminders processed' };
                    default:
                        throw new Error(`Unknown dispute job type: ${job.name}`);
                }
            }
            catch (error) {
                logger_1.logger.error(`Dispute job ${job.id} failed:`, error);
                throw error;
            }
        });
        this.workers.set(queue_1.QUEUE_NAMES.DISPUTE_ESCALATION, worker);
        logger_1.logger.info('Dispute workers initialized');
    }
    async initializeNotificationWorkers() {
        logger_1.logger.info('Initializing notification workers...');
        const worker = await queue_1.queueManager.getWorker(queue_1.QUEUE_NAMES.NOTIFICATIONS, async (job) => {
            try {
                switch (job.name) {
                    case queue_1.JOB_TYPES.SEND_EMAIL:
                        return { success: true, message: 'Email sent' };
                    case queue_1.JOB_TYPES.SEND_SMS:
                        return { success: true, message: 'SMS sent' };
                    case queue_1.JOB_TYPES.SEND_PUSH:
                        return { success: true, message: 'Push notification sent' };
                    default:
                        throw new Error(`Unknown notification job type: ${job.name}`);
                }
            }
            catch (error) {
                logger_1.logger.error(`Notification job ${job.id} failed:`, error);
                throw error;
            }
        });
        this.workers.set(queue_1.QUEUE_NAMES.NOTIFICATIONS, worker);
        logger_1.logger.info('Notification workers initialized');
    }
    async initializeDataRetentionWorkers() {
        logger_1.logger.info('Initializing data retention workers...');
        const worker = await queue_1.queueManager.getWorker(queue_1.QUEUE_NAMES.DATA_RETENTION, async (job) => {
            try {
                switch (job.name) {
                    case queue_1.JOB_TYPES.RUN_DATA_RETENTION:
                        const { sessionsDays, verificationDays } = job.data;
                        await (0, retention_scheduler_1.runDataRetention)({ sessionsDays, verificationDays });
                        return { success: true, message: 'Data retention completed' };
                    default:
                        throw new Error(`Unknown data retention job type: ${job.name}`);
                }
            }
            catch (error) {
                logger_1.logger.error(`Data retention job ${job.id} failed:`, error);
                throw error;
            }
        });
        this.workers.set(queue_1.QUEUE_NAMES.DATA_RETENTION, worker);
        logger_1.logger.info('Data retention workers initialized');
    }
    getWorkerStatus(queueName) {
        const worker = this.workers.get(queueName);
        if (!worker) {
            return null;
        }
        return {
            isActive: worker.isRunning(),
            concurrency: worker.concurrency
        };
    }
    getAllWorkerStatuses() {
        const statuses = {};
        for (const [queueName, worker] of this.workers) {
            statuses[queueName] = {
                isActive: worker.isRunning(),
                concurrency: worker.concurrency
            };
        }
        return statuses;
    }
    pauseWorker(queueName) {
        const worker = this.workers.get(queueName);
        if (!worker) {
            throw new Error(`Worker not found for queue: ${queueName}`);
        }
        worker.pause();
        logger_1.logger.info(`Worker paused for queue: ${queueName}`);
    }
    resumeWorker(queueName) {
        const worker = this.workers.get(queueName);
        if (!worker) {
            throw new Error(`Worker not found for queue: ${queueName}`);
        }
        worker.resume();
        logger_1.logger.info(`Worker resumed for queue: ${queueName}`);
    }
    pauseAllWorkers() {
        logger_1.logger.info('Pausing all workers...');
        for (const [queueName, worker] of this.workers) {
            try {
                worker.pause();
                logger_1.logger.info(`Worker paused for queue: ${queueName}`);
            }
            catch (error) {
                logger_1.logger.error(`Failed to pause worker for queue ${queueName}:`, error);
            }
        }
    }
    resumeAllWorkers() {
        logger_1.logger.info('Resuming all workers...');
        for (const [queueName, worker] of this.workers) {
            try {
                worker.resume();
                logger_1.logger.info(`Worker resumed for queue: ${queueName}`);
            }
            catch (error) {
                logger_1.logger.error(`Failed to resume worker for queue ${queueName}:`, error);
            }
        }
    }
    getStatus() {
        const totalWorkers = this.workers.size;
        let activeWorkers = 0;
        let pausedWorkers = 0;
        for (const [queueName, worker] of this.workers) {
            try {
                if (worker.isRunning()) {
                    activeWorkers++;
                }
                else {
                    pausedWorkers++;
                }
            }
            catch (error) {
                logger_1.logger.error(`Error checking worker status for queue ${queueName}:`, error);
            }
        }
        return {
            isInitialized: this.isInitialized,
            totalWorkers,
            activeWorkers,
            pausedWorkers
        };
    }
    async shutdown() {
        logger_1.logger.info('Shutting down worker manager...');
        for (const [queueName, worker] of this.workers) {
            try {
                await worker.close();
                logger_1.logger.info(`Worker closed for queue: ${queueName}`);
            }
            catch (error) {
                logger_1.logger.error(`Error closing worker for queue ${queueName}:`, error);
            }
        }
        this.workers.clear();
        this.isInitialized = false;
        logger_1.logger.info('Worker manager shutdown completed');
    }
}
exports.WorkerManager = WorkerManager;
exports.workerManager = WorkerManager.getInstance();
const initializeWorkers = () => {
    return exports.workerManager.initializeWorkers();
};
exports.initializeWorkers = initializeWorkers;
