"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.startReconciliationScheduler = exports.reconciliationScheduler = exports.ReconciliationScheduler = void 0;
const node_cron_1 = __importDefault(require("node-cron"));
const logger_1 = require("../../config/logger");
const payment_service_1 = require("../../modules/payments/payment.service");
const defaultConfig = {
    enabled: true,
    schedule: '0 2 * * *',
    providers: ['paystack', 'flutterwave'],
    retentionDays: 90
};
class ReconciliationScheduler {
    config;
    isRunning = false;
    constructor(config = {}) {
        this.config = { ...defaultConfig, ...config };
    }
    start() {
        if (!this.config.enabled) {
            logger_1.logger.info('Reconciliation scheduler is disabled');
            return;
        }
        logger_1.logger.info(`Starting reconciliation scheduler with schedule: ${this.config.schedule}`);
        node_cron_1.default.schedule(this.config.schedule, async () => {
            await this.runDailyReconciliation();
        }, {
            timezone: 'Africa/Lagos'
        });
    }
    stop() {
        logger_1.logger.info('Stopping reconciliation scheduler');
    }
    async runDailyReconciliation() {
        if (this.isRunning) {
            logger_1.logger.warn('Reconciliation already running, skipping this cycle');
            return;
        }
        this.isRunning = true;
        const startTime = Date.now();
        try {
            logger_1.logger.info('Starting daily reconciliation process');
            const yesterday = new Date();
            yesterday.setDate(yesterday.getDate() - 1);
            const startDate = yesterday.toISOString().split('T')[0];
            const endDate = new Date().toISOString().split('T')[0];
            const results = [];
            for (const provider of this.config.providers) {
                try {
                    logger_1.logger.info(`Running reconciliation for provider: ${provider}`);
                    const result = await payment_service_1.paymentService.reconcileTransactions({
                        provider: provider,
                        startDate: startDate,
                        endDate: endDate
                    });
                    results.push({
                        provider: provider,
                        success: result.success,
                        reconciledCount: result.reconciledCount,
                        discrepancies: result.discrepancies.length
                    });
                    logger_1.logger.info(`Reconciliation completed for ${provider}: ${result.reconciledCount} transactions, ${result.discrepancies.length} discrepancies`);
                    if (result.discrepancies.length > 0) {
                        logger_1.logger.warn(`Found ${result.discrepancies.length} discrepancies for ${provider}:`, result.discrepancies);
                    }
                }
                catch (error) {
                    logger_1.logger.error(`Reconciliation failed for provider ${provider}:`, error);
                    results.push({
                        provider,
                        success: false,
                        error: error instanceof Error ? error.message : 'Unknown error'
                    });
                }
            }
            const successful = results.filter(r => r.success).length;
            const total = results.length;
            const totalDiscrepancies = results.reduce((sum, r) => sum + (r.discrepancies || 0), 0);
            logger_1.logger.info(`Daily reconciliation completed: ${successful}/${total} providers successful, ${totalDiscrepancies} total discrepancies`);
            await this.cleanupOldReconciliations();
        }
        catch (error) {
            logger_1.logger.error('Daily reconciliation process failed:', error);
        }
        finally {
            this.isRunning = false;
            const duration = Date.now() - startTime;
            logger_1.logger.info(`Daily reconciliation process completed in ${duration}ms`);
        }
    }
    async cleanupOldReconciliations() {
        try {
            const cutoffDate = new Date();
            cutoffDate.setDate(cutoffDate.getDate() - this.config.retentionDays);
            logger_1.logger.info(`Cleaning up reconciliation records older than ${cutoffDate.toISOString()}`);
        }
        catch (error) {
            logger_1.logger.error('Failed to cleanup old reconciliation records:', error);
        }
    }
    async runManualReconciliation(provider, startDate, endDate) {
        try {
            logger_1.logger.info(`Running manual reconciliation for ${provider} from ${startDate} to ${endDate}`);
            const result = await payment_service_1.paymentService.reconcileTransactions({
                provider: provider,
                startDate: startDate,
                endDate: endDate
            });
            logger_1.logger.info(`Manual reconciliation completed: ${result.reconciledCount} transactions, ${result.discrepancies.length} discrepancies`);
            return result;
        }
        catch (error) {
            logger_1.logger.error(`Manual reconciliation failed for ${provider}:`, error);
            throw error;
        }
    }
    getStatus() {
        return {
            enabled: this.config.enabled,
            isRunning: this.isRunning,
        };
    }
}
exports.ReconciliationScheduler = ReconciliationScheduler;
exports.reconciliationScheduler = new ReconciliationScheduler();
const startReconciliationScheduler = () => {
    exports.reconciliationScheduler.start();
};
exports.startReconciliationScheduler = startReconciliationScheduler;
