"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.startJobScheduler = exports.jobScheduler = exports.JobScheduler = void 0;
const logger_1 = require("../../config/logger");
const queue_1 = require("../../config/queue");
class JobScheduler {
    static instance;
    scheduledJobs = new Map();
    isInitialized = false;
    constructor() {
        this.initializeScheduledJobs();
    }
    static getInstance() {
        if (!JobScheduler.instance) {
            JobScheduler.instance = new JobScheduler();
        }
        return JobScheduler.instance;
    }
    initializeScheduledJobs() {
        this.scheduledJobs.set('daily-reconciliation', {
            name: 'Daily Reconciliation',
            queue: queue_1.QUEUE_NAMES.RECONCILIATION,
            jobType: queue_1.JOB_TYPES.DAILY_RECONCILIATION,
            pattern: '0 2 * * *',
            data: { providers: ['paystack', 'flutterwave'] },
            options: { priority: 1 },
            enabled: true
        });
        this.scheduledJobs.set('daily-report', {
            name: 'Daily Report',
            queue: queue_1.QUEUE_NAMES.REPORTS,
            jobType: queue_1.JOB_TYPES.DAILY_REPORT,
            pattern: '0 3 * * *',
            data: {
                includeTransactions: true,
                includeDisputes: true,
                includeReconciliation: true
            },
            options: { priority: 2 },
            enabled: true
        });
        this.scheduledJobs.set('daily-cleanup', {
            name: 'Daily Cleanup',
            queue: queue_1.QUEUE_NAMES.CLEANUP,
            jobType: queue_1.JOB_TYPES.OLD_LOGS_CLEANUP,
            pattern: '0 4 * * *',
            data: {
                olderThanDays: 90,
                logLevels: ['debug', 'info'],
                batchSize: 1000,
                dryRun: false
            },
            options: { priority: 3 },
            enabled: true
        });
        this.scheduledJobs.set('weekly-report', {
            name: 'Weekly Report',
            queue: queue_1.QUEUE_NAMES.REPORTS,
            jobType: queue_1.JOB_TYPES.WEEKLY_REPORT,
            pattern: '0 5 * * 1',
            data: {
                includeTransactions: true,
                includeDisputes: true,
                includeReconciliation: true
            },
            options: { priority: 1 },
            enabled: true
        });
        this.scheduledJobs.set('weekly-cleanup', {
            name: 'Weekly Cleanup',
            queue: queue_1.QUEUE_NAMES.CLEANUP,
            jobType: queue_1.JOB_TYPES.OLD_SESSIONS_CLEANUP,
            pattern: '0 6 * * 1',
            data: {
                olderThanDays: 30,
                batchSize: 1000,
                dryRun: false
            },
            options: { priority: 3 },
            enabled: true
        });
        this.scheduledJobs.set('monthly-report', {
            name: 'Monthly Report',
            queue: queue_1.QUEUE_NAMES.REPORTS,
            jobType: queue_1.JOB_TYPES.MONTHLY_REPORT,
            pattern: '0 7 1 * *',
            data: {
                includeTransactions: true,
                includeDisputes: true,
                includeReconciliation: true
            },
            options: { priority: 1 },
            enabled: true
        });
        this.scheduledJobs.set('monthly-cleanup', {
            name: 'Monthly Cleanup',
            queue: queue_1.QUEUE_NAMES.CLEANUP,
            jobType: queue_1.JOB_TYPES.OLD_TRANSACTIONS_CLEANUP,
            pattern: '0 8 1 * *',
            data: {
                olderThanDays: 365,
                batchSize: 1000,
                dryRun: false
            },
            options: { priority: 2 },
            enabled: true
        });
        this.scheduledJobs.set('dispute-escalation', {
            name: 'Dispute Escalation',
            queue: queue_1.QUEUE_NAMES.DISPUTE_ESCALATION,
            jobType: queue_1.JOB_TYPES.ESCALATE_URGENT_DISPUTES,
            pattern: '0 */4 * * *',
            data: {},
            options: { priority: 1 },
            enabled: true
        });
        this.scheduledJobs.set('data-retention', {
            name: 'Data Retention',
            queue: queue_1.QUEUE_NAMES.DATA_RETENTION,
            jobType: queue_1.JOB_TYPES.RUN_DATA_RETENTION,
            pattern: '0 1 * * *',
            data: {
                sessionsDays: Number(process.env['SESSION_RETENTION_DAYS'] || 30),
                verificationDays: Number(process.env['VERIFICATION_RETENTION_DAYS'] || 90)
            },
            options: { priority: 1 },
            enabled: true
        });
        logger_1.logger.info(`Initialized ${this.scheduledJobs.size} scheduled jobs`);
    }
    async startAllJobs() {
        if (this.isInitialized) {
            logger_1.logger.info('Job scheduler already initialized');
            return;
        }
        logger_1.logger.info('Starting job scheduler...');
        try {
            for (const [jobId, job] of this.scheduledJobs) {
                if (job.enabled) {
                    await this.scheduleJob(jobId, job);
                }
                else {
                    logger_1.logger.info(`Skipping disabled job: ${job.name}`);
                }
            }
            this.isInitialized = true;
            logger_1.logger.info('Job scheduler started successfully');
        }
        catch (error) {
            logger_1.logger.error('Failed to start job scheduler:', error);
            throw error;
        }
    }
    async scheduleJob(_jobId, job) {
        try {
            if (!job.enabled) {
                logger_1.logger.info(`Skipping disabled job: ${job.name}`);
                return;
            }
            await queue_1.queueManager.addRecurringJob(job.queue, job.jobType, job.data, job.pattern, job.options);
            logger_1.logger.info(`Scheduled job: ${job.name} (${job.pattern})`);
        }
        catch (error) {
            logger_1.logger.error(`Failed to schedule job ${job.name}:`, error);
            throw error;
        }
    }
    async toggleJob(jobId, enabled) {
        const job = this.scheduledJobs.get(jobId);
        if (!job) {
            throw new Error(`Job not found: ${jobId}`);
        }
        job.enabled = enabled;
        if (enabled && this.isInitialized) {
            await this.scheduleJob(jobId, job);
        }
        logger_1.logger.info(`Job ${job.name} ${enabled ? 'enabled' : 'disabled'}`);
    }
    async updateJobSchedule(jobId, newPattern) {
        const job = this.scheduledJobs.get(jobId);
        if (!job) {
            throw new Error(`Job not found: ${jobId}`);
        }
        if (!this.isValidCronPattern(newPattern)) {
            throw new Error(`Invalid cron pattern: ${newPattern}`);
        }
        job.pattern = newPattern;
        if (job.enabled && this.isInitialized) {
            await this.scheduleJob(jobId, job);
        }
        logger_1.logger.info(`Updated schedule for job ${job.name}: ${newPattern}`);
    }
    isValidCronPattern(pattern) {
        const cronRegex = /^(\*|([0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9])|\*\/([0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9])) (\*|([0-9]|1[0-9]|2[0-3])|\*\/([0-9]|1[0-9]|2[0-3])) (\*|([1-9]|1[0-9]|2[0-9]|3[0-1])|\*\/([1-9]|1[0-9]|2[0-9]|3[0-1])) (\*|([1-9]|1[0-2])|\*\/([1-9]|1[0-2])) (\*|([0-6])|\*\/([0-6]))$/;
        return cronRegex.test(pattern);
    }
    getJobStatus(jobId) {
        const job = this.scheduledJobs.get(jobId);
        if (!job) {
            return null;
        }
        return {
            enabled: job.enabled,
            pattern: job.pattern,
        };
    }
    getAllJobStatuses() {
        const statuses = {};
        for (const [jobId, job] of this.scheduledJobs) {
            statuses[jobId] = {
                enabled: job.enabled,
                pattern: job.pattern,
                queue: job.queue,
                jobType: job.jobType
            };
        }
        return statuses;
    }
    async triggerJob(jobId, data) {
        const job = this.scheduledJobs.get(jobId);
        if (!job) {
            throw new Error(`Job not found: ${jobId}`);
        }
        if (!job.enabled) {
            throw new Error(`Job is disabled: ${job.name}`);
        }
        try {
            const jobData = data || job.data;
            await queue_1.queueManager.addJob(job.queue, job.jobType, jobData, { priority: job.options?.priority || 0 });
            logger_1.logger.info(`Manually triggered job: ${job.name}`);
        }
        catch (error) {
            logger_1.logger.error(`Failed to trigger job ${job.name}:`, error);
            throw error;
        }
    }
    getStatus() {
        const totalJobs = this.scheduledJobs.size;
        const enabledJobs = Array.from(this.scheduledJobs.values()).filter(job => job.enabled).length;
        const disabledJobs = totalJobs - enabledJobs;
        return {
            isInitialized: this.isInitialized,
            totalJobs,
            enabledJobs,
            disabledJobs
        };
    }
    async stopAllJobs() {
        logger_1.logger.info('Stopping job scheduler...');
        this.isInitialized = false;
        logger_1.logger.info('Job scheduler stopped');
    }
}
exports.JobScheduler = JobScheduler;
exports.jobScheduler = JobScheduler.getInstance();
const startJobScheduler = () => {
    return exports.jobScheduler.startAllJobs();
};
exports.startJobScheduler = startJobScheduler;
