"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.processDailyReport = processDailyReport;
exports.processWeeklyReport = processWeeklyReport;
exports.processMonthlyReport = processMonthlyReport;
exports.processDisputeReport = processDisputeReport;
exports.getReportProcessor = getReportProcessor;
const logger_1 = require("../../config/logger");
const transactions_1 = require("../../database/repositories/transactions");
const dispute_1 = require("../../database/repositories/dispute");
const queue_1 = require("../../config/queue");
async function processDailyReport(job) {
    const startTime = Date.now();
    const { date, includeTransactions = true, includeDisputes = true, includeReconciliation = true } = job.data;
    logger_1.logger.info(`Starting daily report generation job ${job.id}`, { date, includeTransactions, includeDisputes, includeReconciliation });
    try {
        const reportDate = date ? new Date(date) : new Date();
        reportDate.setDate(reportDate.getDate() - 1);
        const startDate = new Date(reportDate);
        startDate.setHours(0, 0, 0, 0);
        const endDate = new Date(reportDate);
        endDate.setHours(23, 59, 59, 999);
        const startDateStr = startDate.toISOString();
        const endDateStr = endDate.toISOString();
        logger_1.logger.info(`Generating daily report for: ${startDate.toISOString().split('T')[0]}`);
        const reportData = {
            reportType: 'daily',
            date: startDate.toISOString().split('T')[0],
            period: {
                start: startDateStr,
                end: endDateStr
            },
            generatedAt: new Date()
        };
        if (includeTransactions) {
            const txStats = await transactions_1.transactionRepository.getTransactionStats({
                dateFrom: startDate,
                dateTo: endDate
            });
            reportData.transactions = {
                total: txStats.totalTransactions,
                totalAmount: txStats.totalAmount,
                totalFees: txStats.totalFees,
                totalVAT: txStats.totalVAT,
                averageAmount: txStats.avgAmount,
                minAmount: txStats.minAmount,
                maxAmount: txStats.maxAmount
            };
            const categoryBreakdown = await transactions_1.transactionRepository.getCategoryBreakdown(startDate, endDate);
            reportData.transactions.categoryBreakdown = categoryBreakdown;
        }
        if (includeDisputes) {
            const dateFilteredDisputes = await dispute_1.disputeRepository.find({
                openedAt: { $gte: startDate, $lte: endDate }
            });
            reportData.disputes = {
                total: dateFilteredDisputes.length,
                open: dateFilteredDisputes.filter(d => d.status === 'opened').length,
                underReview: dateFilteredDisputes.filter(d => d.status === 'under_review').length,
                resolved: dateFilteredDisputes.filter(d => d.status === 'resolved').length,
                closed: dateFilteredDisputes.filter(d => d.status === 'closed').length
            };
        }
        if (includeReconciliation) {
            reportData.reconciliation = {
                status: 'pending',
                note: 'Reconciliation data will be populated by reconciliation jobs'
            };
        }
        const duration = Date.now() - startTime;
        const result = {
            success: true,
            jobType: queue_1.JOB_TYPES.DAILY_REPORT,
            reportData,
            generatedAt: new Date(),
            duration
        };
        logger_1.logger.info(`Daily report generation job ${job.id} completed`, {
            duration,
            reportSize: JSON.stringify(reportData).length
        });
        return result;
    }
    catch (error) {
        const duration = Date.now() - startTime;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        logger_1.logger.error(`Daily report generation job ${job.id} failed:`, error);
        return {
            success: false,
            jobType: queue_1.JOB_TYPES.DAILY_REPORT,
            reportData: {},
            generatedAt: new Date(),
            duration,
            errors: [errorMessage]
        };
    }
}
async function processWeeklyReport(job) {
    const startTime = Date.now();
    const { weekStart, weekEnd, includeTransactions = true, includeDisputes = true, includeReconciliation = true } = job.data;
    logger_1.logger.info(`Starting weekly report generation job ${job.id}`, { weekStart, weekEnd, includeTransactions, includeDisputes, includeReconciliation });
    try {
        let startDate;
        let endDate;
        if (weekStart && weekEnd) {
            startDate = new Date(weekStart);
            endDate = new Date(weekEnd);
        }
        else {
            const now = new Date();
            const lastWeek = new Date(now);
            lastWeek.setDate(now.getDate() - 7);
            const dayOfWeek = lastWeek.getDay();
            const daysToMonday = dayOfWeek === 0 ? 6 : dayOfWeek - 1;
            startDate = new Date(lastWeek);
            startDate.setDate(lastWeek.getDate() - daysToMonday);
            startDate.setHours(0, 0, 0, 0);
            endDate = new Date(startDate);
            endDate.setDate(startDate.getDate() + 6);
            endDate.setHours(23, 59, 59, 999);
        }
        logger_1.logger.info(`Generating weekly report for: ${startDate.toISOString().split('T')[0]} to ${endDate.toISOString().split('T')[0]}`);
        const reportData = {
            reportType: 'weekly',
            period: {
                start: startDate.toISOString(),
                end: endDate.toISOString()
            },
            generatedAt: new Date()
        };
        if (includeTransactions) {
            const txStats = await transactions_1.transactionRepository.getTransactionStats({
                dateFrom: startDate,
                dateTo: endDate
            });
            reportData.transactions = {
                total: txStats.totalTransactions,
                totalAmount: txStats.totalAmount,
                totalFees: txStats.totalFees,
                totalVAT: txStats.totalVAT,
                averageAmount: txStats.avgAmount,
                minAmount: txStats.minAmount,
                maxAmount: txStats.maxAmount
            };
            const dailyBreakdown = [];
            const currentDate = new Date(startDate);
            while (currentDate <= endDate) {
                const dayStart = new Date(currentDate);
                dayStart.setHours(0, 0, 0, 0);
                const dayEnd = new Date(currentDate);
                dayEnd.setHours(23, 59, 59, 999);
                const dayStats = await transactions_1.transactionRepository.getTransactionStats({
                    dateFrom: dayStart,
                    dateTo: dayEnd
                });
                dailyBreakdown.push({
                    date: currentDate.toISOString().split('T')[0],
                    transactions: dayStats.totalTransactions,
                    amount: dayStats.totalAmount
                });
                currentDate.setDate(currentDate.getDate() + 1);
            }
            reportData.transactions.dailyBreakdown = dailyBreakdown;
        }
        if (includeDisputes) {
            const dateFilteredDisputes = await dispute_1.disputeRepository.find({
                openedAt: { $gte: startDate, $lte: endDate }
            });
            reportData.disputes = {
                total: dateFilteredDisputes.length,
                open: dateFilteredDisputes.filter(d => d.status === 'opened').length,
                underReview: dateFilteredDisputes.filter(d => d.status === 'under_review').length,
                resolved: dateFilteredDisputes.filter(d => d.status === 'resolved').length,
                closed: dateFilteredDisputes.filter(d => d.status === 'closed').length
            };
        }
        const duration = Date.now() - startTime;
        const result = {
            success: true,
            jobType: queue_1.JOB_TYPES.WEEKLY_REPORT,
            reportData,
            generatedAt: new Date(),
            duration
        };
        logger_1.logger.info(`Weekly report generation job ${job.id} completed`, {
            duration,
            reportSize: JSON.stringify(reportData).length
        });
        return result;
    }
    catch (error) {
        const duration = Date.now() - startTime;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        logger_1.logger.error(`Weekly report generation job ${job.id} failed:`, error);
        return {
            success: false,
            jobType: queue_1.JOB_TYPES.WEEKLY_REPORT,
            reportData: {},
            generatedAt: new Date(),
            duration,
            errors: [errorMessage]
        };
    }
}
async function processMonthlyReport(job) {
    const startTime = Date.now();
    const { month, includeTransactions = true, includeDisputes = true, includeReconciliation = true } = job.data;
    logger_1.logger.info(`Starting monthly report generation job ${job.id}`, { month, includeTransactions, includeDisputes, includeReconciliation });
    try {
        let startDate;
        let endDate;
        if (month) {
            const [year, monthNum] = month.split('-').map(Number);
            if (year && monthNum) {
                startDate = new Date(year, monthNum - 1, 1);
                endDate = new Date(year, monthNum, 0, 23, 59, 59, 999);
            }
            else {
                throw new Error('Invalid month format. Expected YYYY-MM');
            }
        }
        else {
            const now = new Date();
            startDate = new Date(now.getFullYear(), now.getMonth() - 1, 1);
            endDate = new Date(now.getFullYear(), now.getMonth(), 0, 23, 59, 59, 999);
        }
        logger_1.logger.info(`Generating monthly report for: ${startDate.toISOString().split('T')[0]} to ${endDate.toISOString().split('T')[0]}`);
        const reportData = {
            reportType: 'monthly',
            period: {
                start: startDate.toISOString(),
                end: endDate.toISOString()
            },
            generatedAt: new Date()
        };
        if (includeTransactions) {
            const txStats = await transactions_1.transactionRepository.getTransactionStats({
                dateFrom: startDate,
                dateTo: endDate
            });
            reportData.transactions = {
                total: txStats.totalTransactions,
                totalAmount: txStats.totalAmount,
                totalFees: txStats.totalFees,
                totalVAT: txStats.totalVAT,
                averageAmount: txStats.avgAmount,
                minAmount: txStats.minAmount,
                maxAmount: txStats.maxAmount
            };
        }
        if (includeDisputes) {
            const dateFilteredDisputes = await dispute_1.disputeRepository.find({
                openedAt: { $gte: startDate, $lte: endDate }
            });
            reportData.disputes = {
                total: dateFilteredDisputes.length,
                open: dateFilteredDisputes.filter(d => d.status === 'opened').length,
                underReview: dateFilteredDisputes.filter(d => d.status === 'under_review').length,
                resolved: dateFilteredDisputes.filter(d => d.status === 'resolved').length,
                closed: dateFilteredDisputes.filter(d => d.status === 'closed').length
            };
        }
        const duration = Date.now() - startTime;
        const result = {
            success: true,
            jobType: queue_1.JOB_TYPES.MONTHLY_REPORT,
            reportData,
            generatedAt: new Date(),
            duration
        };
        logger_1.logger.info(`Monthly report generation job ${job.id} completed`, {
            duration,
            reportSize: JSON.stringify(reportData).length
        });
        return result;
    }
    catch (error) {
        const duration = Date.now() - startTime;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        logger_1.logger.error(`Monthly report generation job ${job.id} failed:`, error);
        return {
            success: false,
            jobType: queue_1.JOB_TYPES.MONTHLY_REPORT,
            reportData: {},
            generatedAt: new Date(),
            duration,
            errors: [errorMessage]
        };
    }
}
async function processDisputeReport(job) {
    const startTime = Date.now();
    const { dateFrom, dateTo, includeEvidence = false, includeResolution = false } = job.data;
    logger_1.logger.info(`Starting dispute report generation job ${job.id}`, { dateFrom, dateTo, includeEvidence, includeResolution });
    try {
        let startDate;
        let endDate;
        if (dateFrom && dateTo) {
            startDate = new Date(dateFrom);
            endDate = new Date(dateTo);
        }
        else {
            endDate = new Date();
            startDate = new Date();
            startDate.setDate(endDate.getDate() - 30);
        }
        logger_1.logger.info(`Generating dispute report for: ${startDate.toISOString().split('T')[0]} to ${endDate.toISOString().split('T')[0]}`);
        const reportData = {
            reportType: 'dispute',
            period: {
                start: startDate.toISOString(),
                end: endDate.toISOString()
            },
            generatedAt: new Date()
        };
        const dateFilteredDisputes = await dispute_1.disputeRepository.find({
            createdAt: { $gte: startDate, $lte: endDate }
        });
        reportData.disputes = {
            total: dateFilteredDisputes.length,
            open: dateFilteredDisputes.filter(d => d.status === 'opened').length,
            underReview: dateFilteredDisputes.filter(d => d.status === 'under_review').length,
            resolved: dateFilteredDisputes.filter(d => d.status === 'resolved').length,
            closed: dateFilteredDisputes.filter(d => d.status === 'closed').length,
            urgent: dateFilteredDisputes.filter(d => d.priority === 'urgent').length,
            chargebacks: dateFilteredDisputes.filter(d => d.type === 'chargeback').length
        };
        if (includeEvidence || includeResolution) {
            const detailedDisputes = dateFilteredDisputes.map(dispute => {
                const detailed = {
                    id: dispute.id,
                    type: dispute.type,
                    status: dispute.status,
                    priority: dispute.priority,
                    openedAt: dispute.createdAt,
                    reason: dispute.reason
                };
                if (includeEvidence) {
                    detailed.evidence = dispute.evidence;
                }
                if (includeResolution && dispute.status === 'resolved') {
                    detailed.resolution = dispute.metadata?.['resolution'];
                    detailed.resolvedAt = dispute.resolution?.resolvedAt;
                }
                return detailed;
            });
            reportData.disputes.details = detailedDisputes;
        }
        const duration = Date.now() - startTime;
        const result = {
            success: true,
            jobType: queue_1.JOB_TYPES.DISPUTE_REPORT,
            reportData,
            generatedAt: new Date(),
            duration
        };
        logger_1.logger.info(`Dispute report generation job ${job.id} completed`, {
            duration,
            reportSize: JSON.stringify(reportData).length
        });
        return result;
    }
    catch (error) {
        const duration = Date.now() - startTime;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        logger_1.logger.error(`Dispute report generation job ${job.id} failed:`, error);
        return {
            success: false,
            jobType: queue_1.JOB_TYPES.DISPUTE_REPORT,
            reportData: {},
            generatedAt: new Date(),
            duration,
            errors: [errorMessage]
        };
    }
}
function getReportProcessor(jobType) {
    switch (jobType) {
        case queue_1.JOB_TYPES.DAILY_REPORT:
            return processDailyReport;
        case queue_1.JOB_TYPES.WEEKLY_REPORT:
            return processWeeklyReport;
        case queue_1.JOB_TYPES.MONTHLY_REPORT:
            return processMonthlyReport;
        case queue_1.JOB_TYPES.DISPUTE_REPORT:
            return processDisputeReport;
        default:
            throw new Error(`Unknown report job type: ${jobType}`);
    }
}
