"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.processDailyReconciliation = processDailyReconciliation;
exports.processProviderReconciliation = processProviderReconciliation;
exports.getReconciliationProcessor = getReconciliationProcessor;
const logger_1 = require("../../config/logger");
const payment_service_1 = require("../../modules/payments/payment.service");
const queue_1 = require("../../config/queue");
async function processDailyReconciliation(job) {
    const startTime = Date.now();
    const { date, providers = ['paystack', 'flutterwave', 'aspfiy'] } = job.data;
    logger_1.logger.info(`Starting daily reconciliation job ${job.id}`, { date, providers });
    const results = [];
    try {
        const endDate = date ? new Date(date) : new Date();
        const startDate = new Date(endDate);
        startDate.setDate(startDate.getDate() - 1);
        const startDateStr = startDate.toISOString().split('T')[0];
        const endDateStr = endDate.toISOString().split('T')[0];
        logger_1.logger.info(`Reconciliation date range: ${startDateStr} to ${endDateStr}`);
        for (const provider of providers) {
            try {
                logger_1.logger.info(`Processing reconciliation for provider: ${provider}`);
                const result = await payment_service_1.paymentService.reconcileTransactions({
                    provider: provider,
                    startDate: startDateStr,
                    endDate: endDateStr
                });
                const duration = Date.now() - startTime;
                results.push({
                    success: result.success,
                    provider,
                    reconciledCount: result.reconciledCount,
                    discrepancies: result.discrepancies,
                    duration
                });
                logger_1.logger.info(`Provider ${provider} reconciliation completed`, {
                    reconciledCount: result.reconciledCount,
                    discrepancies: result.discrepancies.length,
                    duration
                });
                if (result.discrepancies.length > 0) {
                    logger_1.logger.warn(`Found ${result.discrepancies.length} discrepancies for ${provider}:`, result.discrepancies);
                }
            }
            catch (error) {
                const duration = Date.now() - startTime;
                const errorMessage = error instanceof Error ? error.message : 'Unknown error';
                logger_1.logger.error(`Reconciliation failed for provider ${provider}:`, error);
                results.push({
                    success: false,
                    provider,
                    reconciledCount: 0,
                    discrepancies: [],
                    errors: [errorMessage],
                    duration
                });
            }
        }
        const successful = results.filter(r => r.success).length;
        const total = results.length;
        const totalDiscrepancies = results.reduce((sum, r) => sum + r.discrepancies.length, 0);
        const totalDuration = Date.now() - startTime;
        logger_1.logger.info(`Daily reconciliation job ${job.id} completed`, {
            successful: `${successful}/${total}`,
            totalDiscrepancies,
            totalDuration
        });
        return results;
    }
    catch (error) {
        const duration = Date.now() - startTime;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        logger_1.logger.error(`Daily reconciliation job ${job.id} failed:`, error);
        return providers.map(provider => ({
            success: false,
            provider,
            reconciledCount: 0,
            discrepancies: [],
            errors: [errorMessage],
            duration
        }));
    }
}
async function processProviderReconciliation(job) {
    const startTime = Date.now();
    const { provider, startDate, endDate } = job.data;
    logger_1.logger.info(`Starting provider reconciliation job ${job.id}`, { provider, startDate, endDate });
    try {
        const result = await payment_service_1.paymentService.reconcileTransactions({
            provider: provider,
            startDate: startDate,
            endDate: endDate
        });
        const duration = Date.now() - startTime;
        const reconciliationResult = {
            success: result.success,
            provider: provider,
            reconciledCount: result.reconciledCount,
            discrepancies: result.discrepancies,
            duration
        };
        logger_1.logger.info(`Provider reconciliation job ${job.id} completed`, {
            provider,
            reconciledCount: result.reconciledCount,
            discrepancies: result.discrepancies.length,
            duration
        });
        if (result.discrepancies.length > 0) {
            logger_1.logger.warn(`Found ${result.discrepancies.length} discrepancies for ${provider}:`, result.discrepancies);
        }
        return reconciliationResult;
    }
    catch (error) {
        const duration = Date.now() - startTime;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        logger_1.logger.error(`Provider reconciliation job ${job.id} failed:`, error);
        return {
            success: false,
            provider,
            reconciledCount: 0,
            discrepancies: [],
            errors: [errorMessage],
            duration
        };
    }
}
function getReconciliationProcessor(jobType) {
    switch (jobType) {
        case queue_1.JOB_TYPES.DAILY_RECONCILIATION:
            return processDailyReconciliation;
        case queue_1.JOB_TYPES.PROVIDER_RECONCILIATION:
            return processProviderReconciliation;
        default:
            throw new Error(`Unknown reconciliation job type: ${jobType}`);
    }
}
