"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.processOldTransactionsCleanup = processOldTransactionsCleanup;
exports.processOldLogsCleanup = processOldLogsCleanup;
exports.processOldSessionsCleanup = processOldSessionsCleanup;
exports.getCleanupProcessor = getCleanupProcessor;
const logger_1 = require("../../config/logger");
const transactions_1 = require("../../database/repositories/transactions");
const queue_1 = require("../../config/queue");
const mongoose_1 = __importDefault(require("mongoose"));
async function processOldTransactionsCleanup(job) {
    const startTime = Date.now();
    const { olderThanDays = 365, batchSize = 1000, dryRun = false } = job.data;
    logger_1.logger.info(`Starting old transactions cleanup job ${job.id}`, { olderThanDays, batchSize, dryRun });
    try {
        const cutoffDate = new Date();
        cutoffDate.setDate(cutoffDate.getDate() - olderThanDays);
        logger_1.logger.info(`Cleaning up transactions older than: ${cutoffDate.toISOString()}`);
        let itemsProcessed = 0;
        let itemsRemoved = 0;
        let hasMore = true;
        while (hasMore) {
            const oldTransactions = await transactions_1.transactionRepository.findOldTransactions(cutoffDate, batchSize);
            if (oldTransactions.length === 0) {
                hasMore = false;
                break;
            }
            itemsProcessed += oldTransactions.length;
            if (!dryRun) {
                const deleteResult = await transactions_1.transactionRepository.deleteOldTransactions(cutoffDate);
                itemsRemoved += deleteResult.deletedCount || 0;
                logger_1.logger.info(`Batch processed: ${oldTransactions.length} transactions, deleted: ${deleteResult.deletedCount}`);
            }
            else {
                logger_1.logger.info(`DRY RUN: Would delete ${oldTransactions.length} transactions`);
                itemsRemoved += oldTransactions.length;
            }
            if (oldTransactions.length < batchSize) {
                hasMore = false;
            }
        }
        const duration = Date.now() - startTime;
        const result = {
            success: true,
            jobType: queue_1.JOB_TYPES.OLD_TRANSACTIONS_CLEANUP,
            itemsProcessed,
            itemsRemoved,
            duration,
            dryRun
        };
        logger_1.logger.info(`Old transactions cleanup job ${job.id} completed`, {
            itemsProcessed,
            itemsRemoved,
            duration,
            dryRun
        });
        return result;
    }
    catch (error) {
        const duration = Date.now() - startTime;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        logger_1.logger.error(`Old transactions cleanup job ${job.id} failed:`, error);
        return {
            success: false,
            jobType: queue_1.JOB_TYPES.OLD_TRANSACTIONS_CLEANUP,
            itemsProcessed: 0,
            itemsRemoved: 0,
            errors: [errorMessage],
            duration,
            dryRun
        };
    }
}
async function processOldLogsCleanup(job) {
    const startTime = Date.now();
    const { olderThanDays = 90, logLevels = ['debug', 'info'], batchSize = 1000, dryRun = false } = job.data;
    logger_1.logger.info(`Starting old logs cleanup job ${job.id}`, { olderThanDays, logLevels, batchSize, dryRun });
    try {
        const cutoffDate = new Date();
        cutoffDate.setDate(cutoffDate.getDate() - olderThanDays);
        logger_1.logger.info(`Cleaning up logs older than: ${cutoffDate.toISOString()}`);
        const logsCollection = mongoose_1.default.connection.db?.collection('logs');
        let itemsProcessed = 0;
        let itemsRemoved = 0;
        let hasMore = true;
        while (hasMore) {
            const oldLogs = await logsCollection?.find({
                timestamp: { $lt: cutoffDate },
                level: { $in: logLevels }
            }).limit(batchSize).toArray() || [];
            if (oldLogs.length === 0) {
                hasMore = false;
                break;
            }
            itemsProcessed += oldLogs.length;
            if (!dryRun) {
                const deleteResult = await logsCollection?.deleteMany({
                    _id: { $in: oldLogs.map(l => l._id) }
                });
                itemsRemoved += deleteResult?.deletedCount || 0;
                logger_1.logger.info(`Batch processed: ${oldLogs.length} logs, deleted: ${deleteResult?.deletedCount}`);
            }
            else {
                logger_1.logger.info(`DRY RUN: Would delete ${oldLogs.length} logs`);
                itemsRemoved += oldLogs.length;
            }
            if (oldLogs.length < batchSize) {
                hasMore = false;
            }
        }
        const duration = Date.now() - startTime;
        const result = {
            success: true,
            jobType: queue_1.JOB_TYPES.OLD_LOGS_CLEANUP,
            itemsProcessed,
            itemsRemoved,
            duration,
            dryRun
        };
        logger_1.logger.info(`Old logs cleanup job ${job.id} completed`, {
            itemsProcessed,
            itemsRemoved,
            duration,
            dryRun
        });
        return result;
    }
    catch (error) {
        const duration = Date.now() - startTime;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        logger_1.logger.error(`Old logs cleanup job ${job.id} failed:`, error);
        return {
            success: false,
            jobType: queue_1.JOB_TYPES.OLD_LOGS_CLEANUP,
            itemsProcessed: 0,
            itemsRemoved: 0,
            errors: [errorMessage],
            duration,
            dryRun
        };
    }
}
async function processOldSessionsCleanup(job) {
    const startTime = Date.now();
    const { olderThanDays = 30, batchSize = 1000, dryRun = false } = job.data;
    logger_1.logger.info(`Starting old sessions cleanup job ${job.id}`, { olderThanDays, batchSize, dryRun });
    try {
        const cutoffDate = new Date();
        cutoffDate.setDate(cutoffDate.getDate() - olderThanDays);
        logger_1.logger.info(`Cleaning up sessions older than: ${cutoffDate.toISOString()}`);
        const sessionsCollection = mongoose_1.default.connection.db?.collection('sessions');
        let itemsProcessed = 0;
        let itemsRemoved = 0;
        let hasMore = true;
        while (hasMore) {
            const oldSessions = await sessionsCollection?.find({
                lastAccess: { $lt: cutoffDate }
            }).limit(batchSize).toArray() || [];
            if (oldSessions.length === 0) {
                hasMore = false;
                break;
            }
            itemsProcessed += oldSessions.length;
            if (!dryRun) {
                const deleteResult = await sessionsCollection?.deleteMany({
                    _id: { $in: oldSessions.map(s => s._id) }
                });
                itemsRemoved += deleteResult?.deletedCount || 0;
                logger_1.logger.info(`Batch processed: ${oldSessions.length} sessions, deleted: ${deleteResult?.deletedCount}`);
            }
            else {
                logger_1.logger.info(`DRY RUN: Would delete ${oldSessions.length} sessions`);
                itemsRemoved += oldSessions.length;
            }
            if (oldSessions.length < batchSize) {
                hasMore = false;
            }
        }
        const duration = Date.now() - startTime;
        const result = {
            success: true,
            jobType: queue_1.JOB_TYPES.OLD_SESSIONS_CLEANUP,
            itemsProcessed,
            itemsRemoved,
            duration,
            dryRun
        };
        logger_1.logger.info(`Old sessions cleanup job ${job.id} completed`, {
            itemsProcessed,
            itemsRemoved,
            duration,
            dryRun
        });
        return result;
    }
    catch (error) {
        const duration = Date.now() - startTime;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        logger_1.logger.error(`Old sessions cleanup job ${job.id} failed:`, error);
        return {
            success: false,
            jobType: queue_1.JOB_TYPES.OLD_SESSIONS_CLEANUP,
            itemsProcessed: 0,
            itemsRemoved: 0,
            errors: [errorMessage],
            duration,
            dryRun
        };
    }
}
function getCleanupProcessor(jobType) {
    switch (jobType) {
        case queue_1.JOB_TYPES.OLD_TRANSACTIONS_CLEANUP:
            return processOldTransactionsCleanup;
        case queue_1.JOB_TYPES.OLD_LOGS_CLEANUP:
            return processOldLogsCleanup;
        case queue_1.JOB_TYPES.OLD_SESSIONS_CLEANUP:
            return processOldSessionsCleanup;
        default:
            throw new Error(`Unknown cleanup job type: ${jobType}`);
    }
}
