export type Network = 'mtn' | 'airtel' | 'glo' | '9mobile';
export type VtuType = 'airtime' | 'data';
export type BillType = 'electricity' | 'cable' | 'education' | 'insurance' | 'transport';
export type MeterType = 'prepaid' | 'postpaid';
export type CableProvider = 'dstv' | 'gotv' | 'startimes' | 'showmax';
export type ElectricityProvider = 'ikeja' | 'eko' | 'ibadan' | 'enugu' | 'phcn' | 'jos' | 'kaduna' | 'kano' | 'portharcourt' | 'abuja';
export interface VtuRequest {
    amount: number;
    type: VtuType;
    network: Network;
    phone: string;
    planCode?: string;
    reference: string;
    metadata?: Record<string, any>;
}
export interface VtuResponse {
    success: boolean;
    providerReference?: string;
    transactionId?: string;
    status: 'pending' | 'processing' | 'successful' | 'failed';
    message?: string;
    meta?: Record<string, any>;
}
export interface DataPlan {
    id: string;
    name: string;
    network: Network;
    size: string;
    validity: string;
    price: number;
    planCode: string;
    description?: string;
}
export interface NetworkInfo {
    id: string;
    name: string;
    code: Network;
    logo?: string;
    status: 'active' | 'inactive';
}
export interface BillRequest {
    amount: number;
    type: BillType;
    provider: string;
    customerId: string;
    reference: string;
    metadata?: Record<string, any>;
}
export interface BillResponse {
    success: boolean;
    providerReference?: string;
    transactionId?: string;
    status: 'pending' | 'processing' | 'successful' | 'failed';
    message?: string;
    meta?: Record<string, any>;
}
export interface ElectricityBillRequest extends BillRequest {
    type: 'electricity';
    provider: ElectricityProvider;
    meterNumber: string;
    meterType: MeterType;
    customerName?: string;
    customerAddress?: string;
    units?: number;
}
export interface ElectricityValidationResponse {
    success: boolean;
    customerName?: string;
    customerAddress?: string;
    outstandingBalance?: number;
    dueDate?: string;
    meterNumber: string;
    provider: ElectricityProvider;
    message?: string;
}
export interface CableBillRequest extends BillRequest {
    type: 'cable';
    provider: CableProvider;
    smartCardNumber: string;
    customerName?: string;
    package?: string;
    months?: number;
}
export interface CableValidationResponse {
    success: boolean;
    customerName?: string;
    currentPackage?: string;
    dueDate?: string;
    outstandingBalance?: number;
    smartCardNumber: string;
    provider: CableProvider;
    message?: string;
}
export interface EducationBillRequest extends BillRequest {
    type: 'education';
    provider: string;
    studentId?: string;
    examType?: string;
    quantity?: number;
}
export interface ExamPinRequest {
    examType: string;
    quantity: number;
    reference: string;
    metadata?: Record<string, any>;
}
export interface ExamPinResponse {
    success: boolean;
    providerReference?: string;
    pins: string[];
    quantity: number;
    examType: string;
    message?: string;
    meta?: Record<string, any>;
}
export interface ProviderConfig {
    name: string;
    baseUrl: string;
    apiKey: string;
    secretKey?: string;
    timeout?: number;
    retryAttempts?: number;
    sandboxMode?: boolean;
    enabled: boolean;
}
export interface ProviderStatus {
    name: string;
    status: 'online' | 'offline' | 'maintenance';
    responseTime?: number;
    lastCheck: Date;
    errorCount: number;
    successRate: number;
}
export interface TransactionQuery {
    reference: string;
    providerReference?: string;
    startDate?: string;
    endDate?: string;
    status?: string;
    type?: VtuType | BillType;
}
export interface VtuBillsProviderAdapter {
    getProviderInfo(): {
        name: string;
        version: string;
        features: string[];
    };
    buyAirtime(request: VtuRequest): Promise<VtuResponse>;
    buyData(request: VtuRequest): Promise<VtuResponse>;
    getDataPlans(network: Network): Promise<DataPlan[]>;
    getNetworks(): Promise<NetworkInfo[]>;
    payElectricityBill(request: ElectricityBillRequest): Promise<BillResponse>;
    payCableBill(request: CableBillRequest): Promise<BillResponse>;
    payEducationBill(request: EducationBillRequest): Promise<BillResponse>;
    validateElectricityMeter(meterNumber: string, provider: ElectricityProvider, meterType: MeterType): Promise<ElectricityValidationResponse>;
    validateCableIUC(smartCardNumber: string, provider: CableProvider): Promise<CableValidationResponse>;
    buyExamPin(request: ExamPinRequest): Promise<ExamPinResponse>;
    generateRechargePins(amount: number, quantity: number, reference: string): Promise<ExamPinResponse>;
    queryTransaction(query: TransactionQuery): Promise<any>;
    getTransactionHistory(filters?: Record<string, any>): Promise<any[]>;
    checkHealth(): Promise<ProviderStatus>;
    updateConfig(config: Partial<ProviderConfig>): void;
    getConfig(): ProviderConfig;
}
export declare class VtuBillsError extends Error {
    code: string;
    statusCode: number;
    details?: any | undefined;
    constructor(message: string, code: string, statusCode?: number, details?: any | undefined);
}
export declare class ProviderError extends VtuBillsError {
    constructor(message: string, provider: string, details?: any);
}
export declare class ValidationError extends VtuBillsError {
    constructor(message: string, field?: string);
}
export declare class InsufficientBalanceError extends VtuBillsError {
    constructor(required: number, available: number);
}
export declare class ProviderInsufficientBalanceError extends VtuBillsError {
    constructor(provider: string, details?: any);
}
export type ProviderName = 'asbdata' | 'vtuplus' | 'vtucore' | 'mock';
export interface ProviderRegistry {
    [key: string]: VtuBillsProviderAdapter;
}
export interface SandboxConfig {
    enabled: boolean;
    providers: {
        [key in ProviderName]?: {
            enabled: boolean;
            mockResponses?: boolean;
            simulateDelays?: boolean;
            errorRate?: number;
        };
    };
}
