"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.vtuBillsProviderManager = exports.VtuBillsProviderManager = void 0;
const env_1 = require("../../config/env");
const asbdata_provider_1 = require("./asbdata.provider");
const mock_provider_1 = require("./mock.provider");
const vtu_bills_types_1 = require("./vtu-bills.types");
class VtuBillsProviderManager {
    static instance;
    providers = {};
    currentProvider = 'asbdata';
    sandboxConfig;
    providerCircuitUntil = {};
    circuitTtlMs = Number((env_1.env.VTU_PROVIDER_CIRCUIT_TTL_MS ?? process.env.VTU_PROVIDER_CIRCUIT_TTL_MS) || 300000);
    providerStats = {};
    constructor() {
        this.sandboxConfig = {
            enabled: env_1.env.NODE_ENV === 'development' || env_1.env.NODE_ENV === 'test',
            providers: {
                asbdata: {
                    enabled: true,
                    mockResponses: false,
                    simulateDelays: true,
                    errorRate: 0,
                },
                mock: {
                    enabled: true,
                    mockResponses: true,
                    simulateDelays: true,
                    errorRate: 0,
                },
            },
        };
        this.initializeProviders();
    }
    static getInstance() {
        if (!VtuBillsProviderManager.instance) {
            VtuBillsProviderManager.instance = new VtuBillsProviderManager();
        }
        return VtuBillsProviderManager.instance;
    }
    initializeProviders() {
        const asbDataConfig = {
            name: 'ASB Data',
            baseUrl: env_1.env.ASB_DATA_BASE_URL || 'https://asbdata.com/api',
            apiKey: env_1.env.ASB_DATA_API_KEY || '',
            timeout: 30000,
            retryAttempts: 3,
            sandboxMode: env_1.env.VTU_SANDBOX_MODE || false,
            enabled: !!env_1.env.ASB_DATA_API_KEY,
        };
        this.providers['asbdata'] = new asbdata_provider_1.ASBDataProvider(asbDataConfig);
        const mockConfig = {
            name: 'Mock Provider',
            baseUrl: 'https://mock.api',
            apiKey: 'mock-key',
            timeout: 5000,
            retryAttempts: 1,
            sandboxMode: true,
            enabled: true,
        };
        this.providers['mock'] = new mock_provider_1.MockProvider(mockConfig);
        this.currentProvider = this.sandboxConfig.enabled ? 'mock' : 'asbdata';
    }
    setProvider(providerName) {
        if (!this.providers[providerName]) {
            throw new vtu_bills_types_1.ProviderError(`Provider ${providerName} not found`, providerName);
        }
        if (!this.providers[providerName].getConfig().enabled) {
            throw new vtu_bills_types_1.ProviderError(`Provider ${providerName} is not enabled`, providerName);
        }
        this.currentProvider = providerName;
        console.log(`[ProviderManager] Switched to provider: ${providerName}`);
    }
    getCurrentProvider() {
        return this.currentProvider;
    }
    getProvider(providerName) {
        const provider = providerName || this.currentProvider;
        if (!this.providers[provider]) {
            throw new vtu_bills_types_1.ProviderError(`Provider ${provider} not found`, provider);
        }
        return this.providers[provider];
    }
    getAllProviders() {
        return { ...this.providers };
    }
    getProviderInfo(providerName) {
        const provider = this.getProvider(providerName);
        return provider.getProviderInfo();
    }
    enableSandboxMode() {
        this.sandboxConfig.enabled = true;
        this.currentProvider = 'mock';
        console.log('[ProviderManager] Sandbox mode enabled');
    }
    disableSandboxMode() {
        this.sandboxConfig.enabled = false;
        this.currentProvider = 'asbdata';
        console.log('[ProviderManager] Sandbox mode disabled');
    }
    isSandboxMode() {
        return this.sandboxConfig.enabled;
    }
    getSandboxConfig() {
        return { ...this.sandboxConfig };
    }
    updateSandboxConfig(config) {
        this.sandboxConfig = { ...this.sandboxConfig, ...config };
        console.log('[ProviderManager] Sandbox config updated:', this.sandboxConfig);
    }
    async checkAllProvidersHealth() {
        const healthStatus = {};
        for (const [name, provider] of Object.entries(this.providers)) {
            try {
                healthStatus[name] = await provider.checkHealth();
            }
            catch (error) {
                healthStatus[name] = {
                    name,
                    status: 'offline',
                    lastCheck: new Date(),
                    errorCount: 1,
                    successRate: 0,
                };
            }
        }
        return healthStatus;
    }
    async checkCurrentProviderHealth() {
        const provider = this.getProvider();
        return await provider.checkHealth();
    }
    _getEnabledProviderOrder() {
        const enabled = Object.keys(this.getAllProviders()).filter((n) => {
            const name = n;
            const isEnabled = this.getProvider(name).getConfig().enabled;
            return isEnabled && !this._isCircuitOpen(name);
        });
        const current = this.getCurrentProvider();
        const scored = enabled.sort((a, b) => {
            const sa = (this.providerStats[a]?.successes || 0) - (this.providerStats[a]?.failures || 0);
            const sb = (this.providerStats[b]?.successes || 0) - (this.providerStats[b]?.failures || 0);
            return sb - sa;
        });
        const ordered = [current, ...scored.filter((n) => n !== current)];
        return ordered;
    }
    _getProviderOrderWithPreferences(request) {
        const base = this._getEnabledProviderOrder();
        const metadata = (request && request.metadata) || {};
        const priorityList = metadata.providerPriority?.filter(Boolean) || [];
        const seen = new Set();
        const ordered = [];
        for (const p of priorityList) {
            if (base.includes(p) && !seen.has(p)) {
                ordered.push(p);
                seen.add(p);
            }
        }
        for (const p of base) {
            if (!seen.has(p))
                ordered.push(p);
        }
        return ordered;
    }
    _recordProviderSuccess(provider) {
        const stats = this.providerStats[provider] || { successes: 0, failures: 0 };
        stats.successes += 1;
        this.providerStats[provider] = stats;
        console.log(JSON.stringify({ event: 'provider_success', provider, timestamp: new Date().toISOString() }));
    }
    _recordProviderFailure(provider) {
        const stats = this.providerStats[provider] || { successes: 0, failures: 0 };
        stats.failures += 1;
        stats.lastFailureAt = Date.now();
        this.providerStats[provider] = stats;
        console.log(JSON.stringify({ event: 'provider_failure', provider, timestamp: new Date().toISOString() }));
    }
    _openCircuit(providerName, reason) {
        const until = Date.now() + this.circuitTtlMs;
        this.providerCircuitUntil[providerName] = until;
        console.warn(`[ProviderCircuit] Opened for ${providerName} until ${new Date(until).toISOString()} reason=${reason}`);
    }
    _isCircuitOpen(providerName) {
        const until = this.providerCircuitUntil[providerName];
        if (!until)
            return false;
        if (Date.now() >= until) {
            delete this.providerCircuitUntil[providerName];
            console.log(`[ProviderCircuit] Closed for ${providerName}`);
            return false;
        }
        return true;
    }
    async buyAirtime(request) {
        const providersInOrder = this._getProviderOrderWithPreferences(request);
        let lastError = null;
        const attempts = [];
        const maxAllowedCost = Number((request?.metadata?.maxAllowedCost));
        for (const providerName of providersInOrder) {
            const provider = this.getProvider(providerName);
            if (!isNaN(maxAllowedCost) && maxAllowedCost > 0) {
                const providerCost = request?.metadata?.providerCosts?.[providerName];
                if (typeof providerCost === 'number' && providerCost > maxAllowedCost) {
                    attempts.push({ provider: providerName, outcome: 'skipped', reason: 'cost_over_max' });
                    continue;
                }
            }
            try {
                console.log(`[ProviderManager] buyAirtime using provider=${providerName} ref=${request.reference}`);
                const result = await provider.buyAirtime(request);
                if (result?.success) {
                    attempts.push({ provider: providerName, outcome: 'success' });
                    this._recordProviderSuccess(providerName);
                    return result;
                }
                console.warn(`[ProviderManager] buyAirtime unsuccessful from ${providerName}: status=${result?.status}`);
                lastError = new Error(result?.message || 'Provider returned unsuccessful response');
                attempts.push({ provider: providerName, outcome: 'failed', message: result?.message });
                this._recordProviderFailure(providerName);
            }
            catch (error) {
                lastError = error;
                const isProviderLowBalance = error?.code === 'INSUFFICIENT_PROVIDER_BALANCE' ||
                    (error instanceof Error && error?.code === 'INSUFFICIENT_PROVIDER_BALANCE');
                if (isProviderLowBalance) {
                    console.warn(`[ProviderFallback] ${providerName} insufficient provider balance. Trying next provider...`);
                    console.log(JSON.stringify({
                        event: 'provider_low_balance',
                        operation: 'buyAirtime',
                        provider: providerName,
                        reference: request.reference,
                        timestamp: new Date().toISOString(),
                    }));
                    this._openCircuit(providerName, 'low_balance');
                    attempts.push({ provider: providerName, outcome: 'skipped', reason: 'low_balance' });
                    continue;
                }
                console.error(`[ProviderManager] buyAirtime error from ${providerName}:`, error?.message || error);
                attempts.push({ provider: providerName, outcome: 'failed', message: error?.message });
                this._recordProviderFailure(providerName);
            }
        }
        return {
            success: false,
            status: 'failed',
            message: `All airtime providers failed. Last error: ${lastError?.message || 'Unknown error'}`,
            meta: { error: lastError ? (lastError?.message || String(lastError)) : undefined, attempts },
        };
    }
    async buyData(request) {
        const providersInOrder = this._getProviderOrderWithPreferences(request);
        let lastError = null;
        const attempts = [];
        const maxAllowedCost = Number((request?.metadata?.maxAllowedCost));
        for (const providerName of providersInOrder) {
            const provider = this.getProvider(providerName);
            if (!isNaN(maxAllowedCost) && maxAllowedCost > 0) {
                const providerCost = request?.metadata?.providerCosts?.[providerName];
                if (typeof providerCost === 'number' && providerCost > maxAllowedCost) {
                    attempts.push({ provider: providerName, outcome: 'skipped', reason: 'cost_over_max' });
                    continue;
                }
            }
            try {
                console.log(`[ProviderManager] buyData using provider=${providerName} ref=${request.reference}`);
                const result = await provider.buyData(request);
                if (result?.success) {
                    attempts.push({ provider: providerName, outcome: 'success' });
                    this._recordProviderSuccess(providerName);
                    return result;
                }
                console.warn(`[ProviderManager] buyData unsuccessful from ${providerName}: status=${result?.status}`);
                lastError = new Error(result?.message || 'Provider returned unsuccessful response');
                attempts.push({ provider: providerName, outcome: 'failed', message: result?.message });
                this._recordProviderFailure(providerName);
            }
            catch (error) {
                lastError = error;
                const isProviderLowBalance = error?.code === 'INSUFFICIENT_PROVIDER_BALANCE' ||
                    (error instanceof Error && error?.code === 'INSUFFICIENT_PROVIDER_BALANCE');
                if (isProviderLowBalance) {
                    console.warn(`[ProviderFallback] ${providerName} insufficient provider balance. Trying next provider...`);
                    console.log(JSON.stringify({
                        event: 'provider_low_balance',
                        operation: 'buyData',
                        provider: providerName,
                        reference: request.reference,
                        timestamp: new Date().toISOString(),
                    }));
                    this._openCircuit(providerName, 'low_balance');
                    attempts.push({ provider: providerName, outcome: 'skipped', reason: 'low_balance' });
                    continue;
                }
                console.error(`[ProviderManager] buyData error from ${providerName}:`, error?.message || error);
                attempts.push({ provider: providerName, outcome: 'failed', message: error?.message });
                this._recordProviderFailure(providerName);
            }
        }
        return {
            success: false,
            status: 'failed',
            message: `All data providers failed. Last error: ${lastError?.message || 'Unknown error'}`,
            meta: { error: lastError ? (lastError?.message || String(lastError)) : undefined, attempts },
        };
    }
    async getDataPlans(network) {
        const provider = this.getProvider();
        return await provider.getDataPlans(network);
    }
    async getNetworks() {
        const provider = this.getProvider();
        return await provider.getNetworks();
    }
    async payElectricityBill(request) {
        const provider = this.getProvider();
        return await provider.payElectricityBill(request);
    }
    async payCableBill(request) {
        const provider = this.getProvider();
        return await provider.payCableBill(request);
    }
    async payEducationBill(request) {
        const provider = this.getProvider();
        return await provider.payEducationBill(request);
    }
    async validateElectricityMeter(meterNumber, provider, meterType) {
        const currentProvider = this.getProvider();
        return await currentProvider.validateElectricityMeter(meterNumber, provider, meterType);
    }
    async validateCableIUC(smartCardNumber, provider) {
        const currentProvider = this.getProvider();
        return await currentProvider.validateCableIUC(smartCardNumber, provider);
    }
    async buyExamPin(request) {
        const provider = this.getProvider();
        return await provider.buyExamPin(request);
    }
    async generateRechargePins(amount, quantity, reference) {
        const provider = this.getProvider();
        return await provider.generateRechargePins(amount, quantity, reference);
    }
    async queryTransaction(query) {
        const provider = this.getProvider();
        return await provider.queryTransaction(query);
    }
    async getTransactionHistory(filters) {
        const provider = this.getProvider();
        return await provider.getTransactionHistory(filters);
    }
    updateProviderConfig(providerName, config) {
        const provider = this.getProvider(providerName);
        provider.updateConfig(config);
        console.log(`[ProviderManager] Updated config for provider: ${providerName}`);
    }
    getProviderConfig(providerName) {
        const provider = this.getProvider(providerName);
        return provider.getConfig();
    }
    getAvailableProviders() {
        const available = [];
        for (const [name, provider] of Object.entries(this.providers)) {
            if (provider.getConfig().enabled) {
                available.push({
                    name: name,
                    info: provider.getProviderInfo(),
                });
            }
        }
        return available;
    }
    getProviderStatus() {
        return {
            currentProvider: this.currentProvider,
            sandboxMode: this.sandboxConfig.enabled,
            availableProviders: Object.keys(this.providers).filter(name => this.providers[name]?.getConfig().enabled),
            totalProviders: Object.keys(this.providers).length,
        };
    }
}
exports.VtuBillsProviderManager = VtuBillsProviderManager;
exports.vtuBillsProviderManager = VtuBillsProviderManager.getInstance();
