"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MockProvider = void 0;
const vtu_bills_types_1 = require("./vtu-bills.types");
class MockProvider {
    config;
    status;
    mockData = new Map();
    constructor(config) {
        this.config = {
            timeout: 5000,
            retryAttempts: 1,
            sandboxMode: true,
            ...config,
        };
        this.status = {
            name: this.config.name,
            status: 'online',
            lastCheck: new Date(),
            errorCount: 0,
            successRate: 100,
        };
        this.initializeMockData();
    }
    getProviderInfo() {
        return {
            name: 'Mock Provider (Sandbox)',
            version: '1.0.0',
            features: [
                'VTU Airtime (Mock)',
                'VTU Data (Mock)',
                'Electricity Bill Payment (Mock)',
                'Cable TV Bill Payment (Mock)',
                'Education Bill Payment (Mock)',
                'Exam Pin Generation (Mock)',
                'Recharge Card Generation (Mock)',
                'Meter Validation (Mock)',
                'IUC Validation (Mock)',
            ],
        };
    }
    async buyAirtime(request) {
        await this.simulateDelay();
        this.validateVtuRequest(request);
        const mockResponse = {
            success: true,
            providerReference: `MOCK-AIR-${Date.now()}`,
            transactionId: `TXN-${Date.now()}`,
            status: 'successful',
            message: 'Mock airtime purchase successful',
            meta: {
                provider: 'mock',
                sandbox: true,
                request,
                timestamp: new Date().toISOString(),
            },
        };
        this.storeMockTransaction(request.reference, mockResponse);
        return mockResponse;
    }
    async buyData(request) {
        await this.simulateDelay();
        this.validateVtuRequest(request);
        if (!request.planCode) {
            throw new vtu_bills_types_1.ValidationError('Plan code is required for data purchase');
        }
        const mockResponse = {
            success: true,
            providerReference: `MOCK-DATA-${Date.now()}`,
            transactionId: `TXN-${Date.now()}`,
            status: 'successful',
            message: 'Mock data purchase successful',
            meta: {
                provider: 'mock',
                sandbox: true,
                request,
                planCode: request.planCode,
                timestamp: new Date().toISOString(),
            },
        };
        this.storeMockTransaction(request.reference, mockResponse);
        return mockResponse;
    }
    async getDataPlans(network) {
        await this.simulateDelay();
        const mockPlans = [
            {
                id: '1',
                name: '1GB Daily',
                network,
                size: '1GB',
                validity: '1 Day',
                price: 100,
                planCode: `${network}_1gb_daily`,
                description: '1GB data valid for 1 day',
            },
            {
                id: '2',
                name: '2GB Weekly',
                network,
                size: '2GB',
                validity: '7 Days',
                price: 200,
                planCode: `${network}_2gb_weekly`,
                description: '2GB data valid for 7 days',
            },
            {
                id: '3',
                name: '5GB Monthly',
                network,
                size: '5GB',
                validity: '30 Days',
                price: 500,
                planCode: `${network}_5gb_monthly`,
                description: '5GB data valid for 30 days',
            },
        ];
        return mockPlans;
    }
    async getNetworks() {
        await this.simulateDelay();
        return [
            { id: '1', name: 'MTN', code: 'mtn', status: 'active' },
            { id: '2', name: 'Airtel', code: 'airtel', status: 'active' },
            { id: '3', name: 'Glo', code: 'glo', status: 'active' },
            { id: '4', name: '9mobile', code: '9mobile', status: 'active' },
        ];
    }
    async payElectricityBill(request) {
        await this.simulateDelay();
        this.validateElectricityRequest(request);
        const mockResponse = {
            success: true,
            providerReference: `MOCK-ELEC-${Date.now()}`,
            transactionId: `TXN-${Date.now()}`,
            status: 'successful',
            message: 'Mock electricity bill payment successful',
            meta: {
                provider: 'mock',
                sandbox: true,
                request,
                meterNumber: request.meterNumber,
                meterType: request.meterType,
                timestamp: new Date().toISOString(),
            },
        };
        this.storeMockTransaction(request.reference, mockResponse);
        return mockResponse;
    }
    async payCableBill(request) {
        await this.simulateDelay();
        this.validateCableRequest(request);
        const mockResponse = {
            success: true,
            providerReference: `MOCK-CABLE-${Date.now()}`,
            transactionId: `TXN-${Date.now()}`,
            status: 'successful',
            message: 'Mock cable bill payment successful',
            meta: {
                provider: 'mock',
                sandbox: true,
                request,
                smartCardNumber: request.smartCardNumber,
                timestamp: new Date().toISOString(),
            },
        };
        this.storeMockTransaction(request.reference, mockResponse);
        return mockResponse;
    }
    async payEducationBill(request) {
        await this.simulateDelay();
        this.validateEducationRequest(request);
        const mockResponse = {
            success: true,
            providerReference: `MOCK-EDU-${Date.now()}`,
            transactionId: `TXN-${Date.now()}`,
            status: 'successful',
            message: 'Mock education bill payment successful',
            meta: {
                provider: 'mock',
                sandbox: true,
                request,
                timestamp: new Date().toISOString(),
            },
        };
        this.storeMockTransaction(request.reference, mockResponse);
        return mockResponse;
    }
    async validateElectricityMeter(meterNumber, provider, _meterType) {
        await this.simulateDelay();
        const mockResponse = {
            success: true,
            customerName: 'Mock Customer',
            customerAddress: '123 Mock Street, Mock City',
            outstandingBalance: 0,
            dueDate: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString(),
            meterNumber,
            provider,
            message: 'Mock meter validation successful',
        };
        return mockResponse;
    }
    async validateCableIUC(smartCardNumber, provider) {
        await this.simulateDelay();
        const mockResponse = {
            success: true,
            customerName: 'Mock Customer',
            currentPackage: 'Premium',
            dueDate: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString(),
            outstandingBalance: 0,
            smartCardNumber,
            provider,
            message: 'Mock IUC validation successful',
        };
        return mockResponse;
    }
    async buyExamPin(request) {
        await this.simulateDelay();
        const mockPins = Array.from({ length: request.quantity }, (_, i) => `MOCK-PIN-${Date.now()}-${i + 1}`);
        const mockResponse = {
            success: true,
            providerReference: `MOCK-EPIN-${Date.now()}`,
            pins: mockPins,
            quantity: request.quantity,
            examType: request.examType,
            message: 'Mock exam pin purchase successful',
            meta: {
                provider: 'mock',
                sandbox: true,
                request,
                timestamp: new Date().toISOString(),
            },
        };
        this.storeMockTransaction(request.reference, mockResponse);
        return mockResponse;
    }
    async generateRechargePins(amount, quantity, reference) {
        await this.simulateDelay();
        const mockPins = Array.from({ length: quantity }, (_, i) => `MOCK-RECHARGE-${amount}-${Date.now()}-${i + 1}`);
        const mockResponse = {
            success: true,
            providerReference: `MOCK-RECHARGE-${Date.now()}`,
            pins: mockPins,
            quantity,
            examType: 'recharge_card',
            message: 'Mock recharge pin generation successful',
            meta: {
                provider: 'mock',
                sandbox: true,
                amount,
                quantity,
                reference,
                timestamp: new Date().toISOString(),
            },
        };
        this.storeMockTransaction(reference, mockResponse);
        return mockResponse;
    }
    async queryTransaction(query) {
        await this.simulateDelay();
        const mockTransaction = this.mockData.get(query.reference || query.providerReference || '');
        if (mockTransaction) {
            return {
                status: 'success',
                data: mockTransaction,
                message: 'Mock transaction found',
            };
        }
        return {
            status: 'error',
            message: 'Mock transaction not found',
        };
    }
    async getTransactionHistory(filters) {
        await this.simulateDelay();
        const transactions = Array.from(this.mockData.values());
        if (filters?.['type']) {
            return transactions.filter(t => t.meta?.request?.['type'] === filters['type']);
        }
        return transactions;
    }
    async checkHealth() {
        this.status.lastCheck = new Date();
        return this.status;
    }
    updateConfig(config) {
        this.config = { ...this.config, ...config };
    }
    getConfig() {
        return { ...this.config };
    }
    validateVtuRequest(request) {
        if (!request.amount || request.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!request.phone || !/^\+?234\d{10}$/.test(request.phone)) {
            throw new vtu_bills_types_1.ValidationError('Invalid phone number format');
        }
        if (!request.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
    }
    validateElectricityRequest(request) {
        if (!request.amount || request.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!request.meterNumber) {
            throw new vtu_bills_types_1.ValidationError('Meter number is required');
        }
        if (!request.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
    }
    validateCableRequest(request) {
        if (!request.amount || request.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!request.smartCardNumber) {
            throw new vtu_bills_types_1.ValidationError('Smart card number is required');
        }
        if (!request.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
    }
    validateEducationRequest(request) {
        if (!request.amount || request.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!request.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
    }
    async simulateDelay() {
        if (this.config.sandboxMode) {
            const delay = Math.random() * 2000 + 500;
            await new Promise(resolve => setTimeout(resolve, delay));
        }
    }
    storeMockTransaction(reference, data) {
        this.mockData.set(reference, {
            ...data,
            createdAt: new Date().toISOString(),
        });
    }
    initializeMockData() {
        const sampleData = [
            {
                reference: 'SAMPLE-REF-001',
                type: 'airtime',
                amount: 100,
                status: 'successful',
                createdAt: new Date().toISOString(),
            },
            {
                reference: 'SAMPLE-REF-002',
                type: 'data',
                amount: 200,
                status: 'successful',
                createdAt: new Date().toISOString(),
            },
        ];
        sampleData.forEach(item => {
            this.mockData.set(item.reference, item);
        });
    }
}
exports.MockProvider = MockProvider;
