"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ASBDataProvider = void 0;
const axios_1 = __importDefault(require("axios"));
const vtu_bills_types_1 = require("./vtu-bills.types");
class ASBDataProvider {
    client;
    config;
    status;
    constructor(config) {
        this.config = {
            timeout: 30000,
            retryAttempts: 3,
            sandboxMode: false,
            ...config,
        };
        this.client = axios_1.default.create({
            baseURL: this.config.baseUrl,
            timeout: this.config.timeout || 30000,
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'Authorization': `Token ${this.config.apiKey}`,
            },
        });
        this.client.interceptors.request.use((config) => {
            console.log(`[ASBData] ${config.method?.toUpperCase()} ${config.url}`);
            return config;
        }, (error) => {
            console.error('[ASBData] Request error:', error);
            return Promise.reject(error);
        });
        this.client.interceptors.response.use((response) => response, (error) => {
            const status = error.response?.status;
            const data = error.response?.data;
            const isHtml404 = status === 404 && typeof data === 'string' && data.includes('<!doctype html>');
            const debug = process.env['DEBUG_ASBDATA'] === '1';
            if (!isHtml404 || debug) {
                console.error('[ASBData] Response error:', data || error.message);
            }
            return Promise.reject(error);
        });
        this.status = {
            name: this.config.name,
            status: 'offline',
            lastCheck: new Date(),
            errorCount: 0,
            successRate: 0,
        };
    }
    getProviderInfo() {
        return {
            name: 'ASB Data',
            version: '1.0.0',
            features: [
                'VTU Airtime',
                'VTU Data',
                'Electricity Bill Payment',
                'Cable TV Bill Payment',
                'Education Bill Payment',
                'Exam Pin Generation',
                'Recharge Card Generation',
                'Meter Validation',
                'IUC Validation',
            ],
        };
    }
    async buyAirtime(request) {
        try {
            this.validateVtuRequest(request);
            const payload = {
                amount: request.amount,
                network: this.mapNetworkToProviderId(request.network),
                mobile_number: this.formatPhoneNumber(request.phone),
                reference: request.reference,
                Ported_number: true,
                airtime_type: "VTU",
                ...request.metadata,
            };
            console.log(`[ASBData] Airtime purchase request:`, {
                amount: payload.amount,
                network: payload.network,
                mobile_number: payload.mobile_number,
                reference: payload.reference,
                baseUrl: this.config.baseUrl,
                sandboxMode: this.config.sandboxMode
            });
            const response = await this.client.post('/topup/', payload);
            const data = response.data;
            console.log(`[ASBData] Airtime purchase response:`, {
                id: data.id,
                Status: data.Status,
                ident: data.ident,
                network: data.network,
                amount: data.amount,
                mobile_number: data.mobile_number,
                plan_network: data.plan_network,
                balance_before: data.balance_before,
                balance_after: data.balance_after,
                create_date: data.create_date,
                fullResponse: data
            });
            const status = data.Status?.toLowerCase();
            const isSuccessful = data.Status === 'successful' ||
                data.status === 'success' ||
                data.status === 'successful' ||
                data.Status === 'success' ||
                (data.Status && data.Status.toLowerCase() === 'successful') ||
                (data.status && data.status.toLowerCase() === 'successful') ||
                data.id;
            if (isSuccessful) {
                return {
                    success: true,
                    providerReference: data.ident || data.id?.toString(),
                    transactionId: data.id?.toString(),
                    status: 'successful',
                    message: data.api_response || data.message || 'Airtime purchase successful',
                    meta: data,
                };
            }
            else if (data.Status === 'pending' || data.status === 'pending') {
                return {
                    success: false,
                    providerReference: data.ident || data.id?.toString(),
                    transactionId: data.id?.toString(),
                    status: 'pending',
                    message: data.api_response || data.message || 'Airtime purchase pending',
                    meta: data,
                };
            }
            else {
                return {
                    success: false,
                    status: 'failed',
                    message: data.api_response || data.message || data.error || 'Airtime purchase failed',
                    meta: data,
                };
            }
        }
        catch (error) {
            console.error(`[ASBData] Airtime purchase error:`, {
                message: error.message,
                response: error.response?.data,
                status: error.response?.status,
                request: {
                    amount: request.amount,
                    network: request.network,
                    phone: request.phone,
                    reference: request.reference
                }
            });
            if (error.response?.data) {
                const errorData = error.response.data;
                const rawMsg = String(errorData.message || errorData.error || '').toLowerCase();
                const providerLowBalance = rawMsg.includes('insufficient') ||
                    rawMsg.includes('low balance') ||
                    rawMsg.includes('insufficient balance') ||
                    rawMsg.includes('not enough balance') ||
                    rawMsg.includes('wallet balance');
                if (providerLowBalance) {
                    throw new vtu_bills_types_1.ProviderInsufficientBalanceError('asbdata', { response: errorData });
                }
                return {
                    success: false,
                    status: 'failed',
                    message: errorData.message || errorData.error || 'Airtime purchase failed',
                    meta: errorData,
                };
            }
            throw new vtu_bills_types_1.ProviderError(`Failed to buy airtime: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { request, error });
        }
    }
    async buyData(request) {
        try {
            this.validateVtuRequest(request);
            if (!request.planCode) {
                throw new vtu_bills_types_1.ValidationError('Plan code is required for data purchase');
            }
            const payload = {
                network: this.mapNetworkToProviderId(request.network),
                mobile_number: this.formatPhoneNumber(request.phone),
                plan: request.planCode,
                Ported_number: true,
                ...request.metadata,
            };
            const response = await this.client.post('/data/', payload);
            const data = response.data;
            const isSuccessful = data.Status === 'successful' ||
                data.status === 'success' ||
                data.status === 'successful' ||
                data.Status === 'success' ||
                (data.Status && data.Status.toLowerCase() === 'successful') ||
                (data.status && data.status.toLowerCase() === 'successful');
            if (isSuccessful) {
                return {
                    success: true,
                    providerReference: data.ident || data.transaction_id || data.reference,
                    transactionId: data.id?.toString() || data.transaction_id,
                    status: 'successful',
                    message: data.api_response || data.message || 'Data purchase successful',
                    meta: data,
                };
            }
            else {
                return {
                    success: false,
                    status: 'failed',
                    message: data.api_response || data.message || 'Data purchase failed',
                    meta: data,
                };
            }
        }
        catch (error) {
            if (error?.response?.data) {
                const errorData = error.response.data;
                const rawMsg = String(errorData.message || errorData.error || '').toLowerCase();
                const providerLowBalance = rawMsg.includes('insufficient') ||
                    rawMsg.includes('low balance') ||
                    rawMsg.includes('insufficient balance') ||
                    rawMsg.includes('not enough balance') ||
                    rawMsg.includes('wallet balance');
                if (providerLowBalance) {
                    throw new vtu_bills_types_1.ProviderInsufficientBalanceError('asbdata', { response: errorData });
                }
            }
            throw new vtu_bills_types_1.ProviderError(`Failed to buy data: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { request, error });
        }
    }
    async getDataPlans(network) {
        try {
            console.log(`[ASBData] Getting data plans for network: ${network}`);
            const networkIdString = this.mapNetworkToProvider(network);
            const networkIdNumber = parseInt(networkIdString);
            const candidateRequests = [];
            candidateRequests.push(this.client.get(`/get/dataplan/${networkIdString}`).catch(() => null));
            candidateRequests.push(this.client.get(`/get/dataplan/${networkIdNumber}`).catch(() => null));
            candidateRequests.push(this.client.get(`/get/dataplan/`, { params: { network: networkIdString } }).catch(() => null));
            candidateRequests.push(this.client.get(`/get/dataplan/`, { params: { network: networkIdNumber } }).catch(() => null));
            candidateRequests.push(this.client.get('/network/').catch(() => null));
            let plansPayload = null;
            for (const req of candidateRequests) {
                const resp = await req;
                if (!resp)
                    continue;
                const data = resp.data;
                if (!data)
                    continue;
                let candidatePlans = [];
                if (Array.isArray(data)) {
                    candidatePlans = data;
                }
                else if (Array.isArray(data?.data)) {
                    candidatePlans = data.data;
                }
                else if (data.MTN_PLAN || data.AIRTEL_PLAN || data.GLO_PLAN || data['9MOBILE_PLAN']) {
                    const keyed = {
                        mtn: data.MTN_PLAN,
                        airtel: data.AIRTEL_PLAN,
                        glo: data.GLO_PLAN,
                        '9mobile': data['9MOBILE_PLAN']
                    };
                    candidatePlans = keyed[network] || [];
                }
                if (candidatePlans.length > 0) {
                    plansPayload = candidatePlans;
                    break;
                }
            }
            let plans = Array.isArray(plansPayload) ? plansPayload : [];
            if (plans.length > 0) {
                console.log(`[ASBData] Raw plans fetched: ${plans.length}. Filtering for ${network} using IDs ${networkIdString}/${networkIdNumber}`);
                const filteredPlans = plans.filter((plan) => {
                    const net = plan.network ?? plan.network_id ?? plan.provider_id ?? plan.networkId;
                    if (net == null)
                        return true;
                    const netStr = String(net).toLowerCase();
                    return (net === networkIdNumber ||
                        net === networkIdString ||
                        netStr === String(networkIdNumber) ||
                        netStr === String(networkIdString) ||
                        netStr === network ||
                        netStr.includes(network));
                });
                const finalPlans = (filteredPlans.length > 0 ? filteredPlans : plans).map((plan) => ({
                    id: plan.id || plan.dataplan_id || plan.plan_id || plan.code,
                    name: plan.plan || plan.name || plan.plan_name || plan.description,
                    network,
                    size: plan.plan || plan.size || plan.data_size || plan.volume,
                    validity: plan.month_validate || plan.validity || plan.duration || plan.validity_month,
                    price: parseFloat((plan.plan_amount ?? plan.price ?? plan.amount ?? plan.selling_price ?? plan.buy_price)?.toString() || '0'),
                    planCode: plan.dataplan_id || plan.plan_code || plan.code || plan.id,
                    description: plan.plan_type || plan.description || plan.category,
                }));
                const cleaned = finalPlans.filter(p => !isNaN(p.price) && p.price > 0);
                if (cleaned.length > 0)
                    return cleaned;
            }
            console.log('[ASBData] No data plans found or invalid response format; using fallback');
            return this.getFallbackDataPlans(network);
        }
        catch (error) {
            console.error('[ASBData] Failed to get data plans:', error);
            return this.getFallbackDataPlans(network);
        }
    }
    async getNetworks() {
        try {
            const response = await this.client.get('/get/network/');
            const data = response.data;
            if (data.status === 'success' && Array.isArray(data.data)) {
                return data.data.map((network) => ({
                    id: network.id,
                    name: network.name,
                    code: this.mapProviderToNetwork(network.code || network.network_code),
                    logo: network.logo,
                    status: network.status === 'active' ? 'active' : 'inactive',
                }));
            }
            return [
                { id: '1', name: 'MTN', code: 'mtn', status: 'active' },
                { id: '2', name: 'Airtel', code: 'airtel', status: 'active' },
                { id: '3', name: 'Glo', code: 'glo', status: 'active' },
                { id: '4', name: '9mobile', code: '9mobile', status: 'active' },
            ];
        }
        catch (error) {
            console.error('[ASBData] Failed to get networks:', error);
            return [
                { id: '1', name: 'MTN', code: 'mtn', status: 'active' },
                { id: '2', name: 'Airtel', code: 'airtel', status: 'active' },
                { id: '3', name: 'Glo', code: 'glo', status: 'active' },
                { id: '4', name: '9mobile', code: '9mobile', status: 'active' },
            ];
        }
    }
    async payElectricityBill(request) {
        try {
            this.validateElectricityRequest(request);
            const payload = {
                amount: request.amount,
                provider: request.provider,
                meter_number: request.meterNumber,
                meter_type: request.meterType,
                customer_name: request.customerName,
                customer_address: request.customerAddress,
                units: request.units,
                reference: request.reference,
                ...request.metadata,
            };
            const response = await this.client.post('/billpayment/', payload);
            const data = response.data;
            if (data.status === 'success') {
                return {
                    success: true,
                    providerReference: data.transaction_id || data.reference,
                    transactionId: data.transaction_id,
                    status: 'successful',
                    message: data.message || 'Electricity bill payment successful',
                    meta: data,
                };
            }
            else {
                return {
                    success: false,
                    status: 'failed',
                    message: data.message || 'Electricity bill payment failed',
                    meta: data,
                };
            }
        }
        catch (error) {
            throw new vtu_bills_types_1.ProviderError(`Failed to pay electricity bill: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { request, error });
        }
    }
    async payCableBill(request) {
        try {
            this.validateCableRequest(request);
            const payload = {
                amount: request.amount,
                provider: request.provider,
                smart_card_number: request.smartCardNumber,
                customer_name: request.customerName,
                package: request.package,
                months: request.months || 1,
                reference: request.reference,
                ...request.metadata,
            };
            const response = await this.client.post('/cablesub/', payload);
            const data = response.data;
            if (data.status === 'success') {
                return {
                    success: true,
                    providerReference: data.transaction_id || data.reference,
                    transactionId: data.transaction_id,
                    status: 'successful',
                    message: data.message || 'Cable bill payment successful',
                    meta: data,
                };
            }
            else {
                return {
                    success: false,
                    status: 'failed',
                    message: data.message || 'Cable bill payment failed',
                    meta: data,
                };
            }
        }
        catch (error) {
            throw new vtu_bills_types_1.ProviderError(`Failed to pay cable bill: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { request, error });
        }
    }
    async payEducationBill(request) {
        try {
            this.validateEducationRequest(request);
            const payload = {
                amount: request.amount,
                provider: request.provider,
                student_id: request.studentId,
                exam_type: request.examType,
                quantity: request.quantity || 1,
                reference: request.reference,
                ...request.metadata,
            };
            const response = await this.client.post('/epin/', payload);
            const data = response.data;
            if (data.status === 'success') {
                return {
                    success: true,
                    providerReference: data.transaction_id || data.reference,
                    transactionId: data.transaction_id,
                    status: 'successful',
                    message: data.message || 'Education bill payment successful',
                    meta: data,
                };
            }
            else {
                return {
                    success: false,
                    status: 'failed',
                    message: data.message || 'Education bill payment failed',
                    meta: data,
                };
            }
        }
        catch (error) {
            throw new vtu_bills_types_1.ProviderError(`Failed to pay education bill: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { request, error });
        }
    }
    async validateElectricityMeter(meterNumber, provider, meterType) {
        try {
            const response = await this.client.get('/validatemeter', {
                params: {
                    meternumber: meterNumber,
                    disconame: provider,
                    mtype: meterType,
                },
            });
            const data = response.data;
            if (data.status === 'success') {
                return {
                    success: true,
                    customerName: data.customer_name,
                    customerAddress: data.customer_address,
                    outstandingBalance: parseFloat(data.outstanding_balance || '0'),
                    dueDate: data.due_date,
                    meterNumber,
                    provider,
                    message: data.message,
                };
            }
            else {
                return {
                    success: false,
                    meterNumber,
                    provider,
                    message: data.message || 'Meter validation failed',
                };
            }
        }
        catch (error) {
            throw new vtu_bills_types_1.ProviderError(`Failed to validate electricity meter: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { meterNumber, provider, meterType, error });
        }
    }
    async validateCableIUC(smartCardNumber, provider) {
        try {
            const response = await this.client.get('/validateiuc', {
                params: {
                    smart_card_number: smartCardNumber,
                    cablename: provider,
                },
            });
            const data = response.data;
            if (data.status === 'success') {
                return {
                    success: true,
                    customerName: data.customer_name,
                    currentPackage: data.current_package,
                    dueDate: data.due_date,
                    outstandingBalance: parseFloat(data.outstanding_balance || '0'),
                    smartCardNumber,
                    provider,
                    message: data.message,
                };
            }
            else {
                return {
                    success: false,
                    smartCardNumber,
                    provider,
                    message: data.message || 'IUC validation failed',
                };
            }
        }
        catch (error) {
            throw new vtu_bills_types_1.ProviderError(`Failed to validate cable IUC: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { smartCardNumber, provider, error });
        }
    }
    async buyExamPin(request) {
        try {
            const payload = {
                exam_type: request.examType,
                quantity: request.quantity,
                reference: request.reference,
                ...request.metadata,
            };
            const response = await this.client.post('/epin/', payload);
            const data = response.data;
            if (data.status === 'success') {
                return {
                    success: true,
                    providerReference: data.transaction_id || data.reference,
                    pins: data.pins || [],
                    quantity: request.quantity,
                    examType: request.examType,
                    message: data.message || 'Exam pin purchase successful',
                    meta: data,
                };
            }
            else {
                return {
                    success: false,
                    pins: [],
                    quantity: request.quantity,
                    examType: request.examType,
                    message: data.message || 'Exam pin purchase failed',
                    meta: data,
                };
            }
        }
        catch (error) {
            throw new vtu_bills_types_1.ProviderError(`Failed to buy exam pin: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { request, error });
        }
    }
    async generateRechargePins(amount, quantity, reference) {
        try {
            const payload = {
                amount,
                quantity,
                reference,
            };
            const response = await this.client.post('/rechargepin/', payload);
            const data = response.data;
            if (data.status === 'success') {
                return {
                    success: true,
                    providerReference: data.transaction_id || data.reference,
                    pins: data.pins || [],
                    quantity,
                    examType: 'recharge_card',
                    message: data.message || 'Recharge pin generation successful',
                    meta: data,
                };
            }
            else {
                return {
                    success: false,
                    pins: [],
                    quantity,
                    examType: 'recharge_card',
                    message: data.message || 'Recharge pin generation failed',
                    meta: data,
                };
            }
        }
        catch (error) {
            throw new vtu_bills_types_1.ProviderError(`Failed to generate recharge pins: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { amount, quantity, reference, error });
        }
    }
    async queryTransaction(query) {
        try {
            let endpoint = '/data/';
            if (query.providerReference) {
                endpoint += query.providerReference;
            }
            else if (query.reference) {
                endpoint += query.reference;
            }
            else {
                throw new vtu_bills_types_1.ValidationError('Either reference or providerReference is required');
            }
            const response = await this.client.get(endpoint);
            return response.data;
        }
        catch (error) {
            throw new vtu_bills_types_1.ProviderError(`Failed to query transaction: ${error instanceof Error ? error.message : 'Unknown error'}`, 'asbdata', { query, error });
        }
    }
    async getTransactionHistory(filters) {
        try {
            const response = await this.client.get('/data/', { params: filters });
            const data = response.data;
            if (data.status === 'success' && Array.isArray(data.data)) {
                return data.data;
            }
            return [];
        }
        catch (error) {
            console.error('[ASBData] Failed to get transaction history:', error);
            return [];
        }
    }
    async checkHealth() {
        try {
            const startTime = Date.now();
            const response = await this.client.get('/user/');
            const responseTime = Date.now() - startTime;
            if (response.data.status === 'success') {
                this.status.status = 'online';
                this.status.responseTime = responseTime;
                this.status.successRate = Math.min(100, this.status.successRate + 10);
            }
            else {
                this.status.status = 'offline';
                this.status.errorCount++;
                this.status.successRate = Math.max(0, this.status.successRate - 10);
            }
            this.status.lastCheck = new Date();
            return this.status;
        }
        catch (error) {
            this.status.status = 'offline';
            this.status.errorCount++;
            this.status.successRate = Math.max(0, this.status.successRate - 10);
            this.status.lastCheck = new Date();
            return this.status;
        }
    }
    updateConfig(config) {
        this.config = { ...this.config, ...config };
        if (config.baseUrl || config.apiKey) {
            this.client.defaults.baseURL = this.config.baseUrl;
            this.client.defaults.headers['Authorization'] = `Bearer ${this.config.apiKey}`;
        }
    }
    getConfig() {
        return { ...this.config };
    }
    validateVtuRequest(request) {
        if (!request.amount || request.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!request.phone || !/^\+?234\d{10}$/.test(request.phone)) {
            throw new vtu_bills_types_1.ValidationError('Invalid phone number format');
        }
        if (!request.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
    }
    validateElectricityRequest(request) {
        if (!request.amount || request.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!request.meterNumber) {
            throw new vtu_bills_types_1.ValidationError('Meter number is required');
        }
        if (!request.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
    }
    validateCableRequest(request) {
        if (!request.amount || request.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!request.smartCardNumber) {
            throw new vtu_bills_types_1.ValidationError('Smart card number is required');
        }
        if (!request.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
    }
    validateEducationRequest(request) {
        if (!request.amount || request.amount <= 0) {
            throw new vtu_bills_types_1.ValidationError('Amount must be greater than 0');
        }
        if (!request.reference) {
            throw new vtu_bills_types_1.ValidationError('Reference is required');
        }
    }
    mapNetworkToProvider(network) {
        const mapping = {
            mtn: '1',
            glo: '2',
            '9mobile': '3',
            airtel: '4',
        };
        return mapping[network] || network;
    }
    mapNetworkToProviderId(network) {
        const mapping = {
            mtn: 1,
            glo: 2,
            '9mobile': 3,
            airtel: 4,
        };
        return mapping[network] || 1;
    }
    formatPhoneNumber(phone) {
        const digits = phone.replace(/\D/g, '');
        if (digits.startsWith('234')) {
            return '0' + digits.substring(3);
        }
        if (phone.startsWith('+234')) {
            return '0' + digits.substring(3);
        }
        if (digits.startsWith('0')) {
            return digits;
        }
        if (digits.length === 11 && !digits.startsWith('0')) {
            return '0' + digits;
        }
        if (digits.length === 10) {
            return '0' + digits;
        }
        return digits;
    }
    mapProviderToNetwork(providerCode) {
        const mapping = {
            '1': 'mtn',
            '2': 'glo',
            '3': '9mobile',
            '4': 'airtel',
        };
        return mapping[providerCode] || 'mtn';
    }
    getFallbackDataPlans(network) {
        const fallbackPlans = {
            mtn: [
                {
                    id: 'mtn-sme-500mb',
                    name: 'MTN SME 500MB',
                    network: 'mtn',
                    size: '500MB',
                    validity: '30 days',
                    price: 130,
                    planCode: 'mtn_500mb_sme',
                    description: 'MTN SME Data 500MB - 30 days validity',
                },
                {
                    id: 'mtn-sme-1gb',
                    name: 'MTN SME 1GB',
                    network: 'mtn',
                    size: '1GB',
                    validity: '30 days',
                    price: 250,
                    planCode: 'mtn_1gb_sme',
                    description: 'MTN SME Data 1GB - 30 days validity',
                },
                {
                    id: 'mtn-sme-2gb',
                    name: 'MTN SME 2GB',
                    network: 'mtn',
                    size: '2GB',
                    validity: '30 days',
                    price: 500,
                    planCode: 'mtn_2gb_sme',
                    description: 'MTN SME Data 2GB - 30 days validity',
                },
                {
                    id: 'mtn-sme-5gb',
                    name: 'MTN SME 5GB',
                    network: 'mtn',
                    size: '5GB',
                    validity: '30 days',
                    price: 1250,
                    planCode: 'mtn_5gb_sme',
                    description: 'MTN SME Data 5GB - 30 days validity',
                },
            ],
            airtel: [
                {
                    id: 'airtel-sme-500mb',
                    name: 'Airtel SME 500MB',
                    network: 'airtel',
                    size: '500MB',
                    validity: '30 days',
                    price: 130,
                    planCode: 'airtel_500mb_sme',
                    description: 'Airtel SME Data 500MB - 30 days validity',
                },
                {
                    id: 'airtel-sme-1gb',
                    name: 'Airtel SME 1GB',
                    network: 'airtel',
                    size: '1GB',
                    validity: '30 days',
                    price: 250,
                    planCode: 'airtel_1gb_sme',
                    description: 'Airtel SME Data 1GB - 30 days validity',
                },
                {
                    id: 'airtel-sme-2gb',
                    name: 'Airtel SME 2GB',
                    network: 'airtel',
                    size: '2GB',
                    validity: '30 days',
                    price: 500,
                    planCode: 'airtel_2gb_sme',
                    description: 'Airtel SME Data 2GB - 30 days validity',
                },
            ],
            glo: [
                {
                    id: 'glo-sme-500mb',
                    name: 'Glo SME 500MB',
                    network: 'glo',
                    size: '500MB',
                    validity: '30 days',
                    price: 130,
                    planCode: 'glo_500mb_sme',
                    description: 'Glo SME Data 500MB - 30 days validity',
                },
                {
                    id: 'glo-sme-1gb',
                    name: 'Glo SME 1GB',
                    network: 'glo',
                    size: '1GB',
                    validity: '30 days',
                    price: 250,
                    planCode: 'glo_1gb_sme',
                    description: 'Glo SME Data 1GB - 30 days validity',
                },
            ],
            '9mobile': [
                {
                    id: '9mobile-sme-500mb',
                    name: '9mobile SME 500MB',
                    network: '9mobile',
                    size: '500MB',
                    validity: '30 days',
                    price: 130,
                    planCode: '9mobile_500mb_sme',
                    description: '9mobile SME Data 500MB - 30 days validity',
                },
                {
                    id: '9mobile-sme-1gb',
                    name: '9mobile SME 1GB',
                    network: '9mobile',
                    size: '1GB',
                    validity: '30 days',
                    price: 250,
                    planCode: '9mobile_1gb_sme',
                    description: '9mobile SME Data 1GB - 30 days validity',
                },
            ],
        };
        return fallbackPlans[network] || [];
    }
}
exports.ASBDataProvider = ASBDataProvider;
