"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnifyProvider = void 0;
const axios_1 = __importDefault(require("axios"));
const verification_types_1 = require("./verification.types");
const circuit_breaker_1 = require("./circuit-breaker");
const retry_strategy_1 = require("./retry-strategy");
class UnifyProvider {
    client;
    config;
    circuitBreaker;
    retryStrategy;
    constructor(config) {
        this.config = config;
        this.circuitBreaker = new circuit_breaker_1.CircuitBreaker({
            failureThreshold: 5,
            recoveryTimeout: 30000,
            monitoringPeriod: 60000,
            successThreshold: 3
        });
        this.retryStrategy = new retry_strategy_1.RetryStrategy({
            maxRetries: 3,
            baseDelay: 1000,
            maxDelay: 5000,
            backoffMultiplier: 2,
            jitter: true
        });
        const providerTimeout = Math.min(config.timeout || 25000, 25000);
        this.client = axios_1.default.create({
            baseURL: config.baseUrl,
            timeout: providerTimeout,
            headers: {
                'Content-Type': 'application/json',
                'User-Agent': 'curl/7.68.0',
            },
        });
        console.log('[UnifyProvider] Initialized without API key, will be set dynamically');
        this.client.interceptors.request.use((config) => {
            if (!config.headers['x-api-key']) {
                const apiKeyFromDefaults = this.client.defaults.headers.common['x-api-key'];
                const apiKeyString = typeof apiKeyFromDefaults === 'string' ? apiKeyFromDefaults : '';
                if (apiKeyString) {
                    config.headers['x-api-key'] = apiKeyString;
                    console.log('[UnifyProvider] Added API key from axios defaults:', {
                        apiKey: `${apiKeyString.substring(0, 8)}...`,
                        length: apiKeyString.length,
                        source: 'axios-defaults'
                    });
                }
                else {
                    const instanceApiKey = this.config.apiKey;
                    if (instanceApiKey && instanceApiKey.length > 0) {
                        config.headers['x-api-key'] = instanceApiKey;
                        console.log('[UnifyProvider] Added API key from instance config:', {
                            apiKey: `${instanceApiKey.substring(0, 8)}...`,
                            length: instanceApiKey.length,
                            source: 'instance-config'
                        });
                    }
                    else {
                        console.warn('[UnifyProvider] No API key found in defaults or instance config');
                    }
                }
            }
            console.log(`[UnifyProvider] ${config.method?.toUpperCase()} ${config.url}`);
            console.log(`[UnifyProvider] Headers:`, JSON.stringify(config.headers, null, 2));
            console.log(`[UnifyProvider] Payload:`, JSON.stringify(config.data, null, 2));
            const apiKey = config.headers['x-api-key'];
            console.log(`[UnifyProvider] API Key Debug:`, {
                raw: apiKey,
                length: apiKey ? apiKey.length : 0,
                first8: apiKey ? apiKey.substring(0, 8) : 'MISSING',
                last8: apiKey ? apiKey.substring(apiKey.length - 8) : 'MISSING'
            });
            return config;
        }, (error) => {
            console.error('[UnifyProvider] Request error:', error);
            return Promise.reject(error);
        });
        this.client.interceptors.response.use((response) => {
            return response;
        }, (error) => {
            const message = error?.message || '';
            const status = error.response?.status || 500;
            const errorData = error.response?.data;
            if (error?.code === 'ECONNABORTED' || /timeout/i.test(message)) {
                throw new verification_types_1.ProviderError('Provider timeout', 504);
            }
            console.error('[UnifyProvider] Response error:', errorData || message);
            if (status === 401) {
                throw new verification_types_1.AuthenticationError('Invalid API key or authentication failed');
            }
            if (status === 400) {
                throw new verification_types_1.ValidationError(errorData?.error || 'Invalid request parameters');
            }
            if (status === 500) {
                const errorMessage = errorData?.error || 'Internal server error';
                const rawError = String(errorMessage).toLowerCase();
                if (rawError.includes('overloaded') || rawError.includes('overload')) {
                    throw new verification_types_1.ProviderError('Service temporarily overloaded. Please try again in a few moments.', 503);
                }
                if (rawError.includes('internal error') || rawError.includes('server error')) {
                    throw new verification_types_1.ProviderError('Verification service is temporarily unavailable. Please try again later.', 503);
                }
                throw new verification_types_1.ProviderError('Verification service is temporarily unavailable. Please try again later.', 503);
            }
            if (status === 403 || (errorData?.error && String(errorData.error).toLowerCase().includes('permission denied'))) {
                throw new verification_types_1.AuthenticationError('API key is invalid or has insufficient permissions. Please check your API credentials.');
            }
            if (status >= 500) {
                throw new verification_types_1.ProviderError('Verification service is temporarily unavailable. Please try again later.', 503);
            }
            throw new verification_types_1.ProviderError(errorData?.error || message, status);
        });
    }
    getProviderInfo() {
        return {
            name: 'Unify Verification',
            version: '4.0.0',
            features: ['nin', 'bvn', 'bvn_phone', 'ipe', 'personalization'],
            baseUrl: this.config.baseUrl,
        };
    }
    updateConfig(config) {
        console.log('[UnifyProvider] Updating config:', {
            apiKey: config.apiKey ? `${config.apiKey.substring(0, 8)}...` : 'NOT_PROVIDED',
            apiKeyLength: config.apiKey ? config.apiKey.length : 0,
            baseUrl: config.baseUrl || 'NOT_PROVIDED'
        });
        this.config = { ...this.config, ...config };
        console.log('[UnifyProvider] Config after merge:', {
            apiKey: this.config.apiKey ? `${this.config.apiKey.substring(0, 8)}...` : 'MISSING',
            apiKeyLength: this.config.apiKey ? this.config.apiKey.length : 0
        });
        if (config.baseUrl) {
            this.client.defaults.baseURL = this.config.baseUrl;
        }
        if (config.apiKey && config.apiKey.length > 0) {
            if (config.apiKey.length < 16) {
                console.warn('[UnifyProvider] WARNING: API key appears to be truncated!', {
                    received: config.apiKey.length,
                    expected: '>= 16',
                    value: config.apiKey.substring(0, 8) + '...'
                });
            }
            this.client.defaults.headers.common['x-api-key'] = this.config.apiKey;
            this.client.defaults.headers['x-api-key'] = this.config.apiKey;
            this.client.defaults.headers.post['x-api-key'] = this.config.apiKey;
            console.log('[UnifyProvider] Updated axios headers:', {
                'x-api-key': this.client.defaults.headers.common['x-api-key'] ? `${this.client.defaults.headers.common['x-api-key'].substring(0, 8)}...` : 'MISSING',
                'x-api-key-length': this.client.defaults.headers.common['x-api-key'] ? this.client.defaults.headers.common['x-api-key'].length : 0
            });
            console.log('[UnifyProvider] Header verification:', {
                common: this.client.defaults.headers.common['x-api-key'] ? `${this.client.defaults.headers.common['x-api-key'].substring(0, 8)}...` : 'MISSING',
                request: this.client.defaults.headers['x-api-key'] ? `${this.client.defaults.headers['x-api-key'].substring(0, 8)}...` : 'MISSING'
            });
            console.log('[UnifyProvider] Config debug after update:', {
                configApiKey: this.config.apiKey ? `${this.config.apiKey.substring(0, 8)}...` : 'MISSING',
                configApiKeyLength: this.config.apiKey ? this.config.apiKey.length : 0
            });
        }
        else if (config.apiKey === '') {
            delete this.client.defaults.headers.common['x-api-key'];
            delete this.client.defaults.headers['x-api-key'];
            delete this.client.defaults.headers.post['x-api-key'];
            console.log('[UnifyProvider] Cleared API key from axios headers');
        }
        if (typeof config.timeout === 'number') {
            const providerTimeout = Math.min(config.timeout, 25000);
            this.client.defaults.timeout = providerTimeout;
        }
    }
    async verifyNin(request) {
        try {
            this.validateNinRequest(request);
            this.ensureApiKeyAvailable();
            let endpoint = '/api/v4/search/nin';
            let payload = {};
            if (request.nin) {
                payload = { NIN: request.nin };
            }
            else if (request.first_name && request.last_name && request.gender && request.dob) {
                payload = {
                    first_name: request.first_name,
                    last_name: request.last_name,
                    gender: request.gender,
                    dob: request.dob,
                };
            }
            else if (request.phone) {
                payload = { phone: request.phone };
            }
            else {
                throw new verification_types_1.ValidationError('Invalid NIN verification request. Provide either NIN, name/gender/dob, or phone number.');
            }
            return await this.circuitBreaker.execute(async () => {
                return await this.retryStrategy.execute(async () => {
                    const response = await this.client.post(endpoint, payload);
                    if (response.data.error) {
                        const raw = String(response.data.error || '').toLowerCase();
                        if (raw.includes('no record')) {
                            throw new verification_types_1.ProviderError('No record found', 404);
                        }
                        if (raw.includes('try again later') || raw.includes('temporary') || raw.includes('currently unavailable')) {
                            throw new verification_types_1.ProviderError('Provider temporarily unavailable, try again later', 503);
                        }
                        throw new verification_types_1.ProviderError(response.data.error, 400);
                    }
                    return response.data;
                }, (error) => {
                    const status = error?.statusCode || error?.status || error?.response?.status;
                    return (status >= 500) || status === 504 || status === 503 || status === 429 || status === 408;
                });
            });
        }
        catch (error) {
            if (error instanceof verification_types_1.ValidationError || error instanceof verification_types_1.AuthenticationError || error instanceof verification_types_1.ProviderError) {
                throw error;
            }
            throw new verification_types_1.ProviderError(`NIN verification failed: ${error instanceof Error ? error.message : String(error)}`);
        }
    }
    async verifyBvn(request) {
        try {
            if (!request.BVN || request.BVN.length !== 11) {
                throw new verification_types_1.ValidationError('BVN must be exactly 11 digits');
            }
            this.ensureApiKeyAvailable();
            const endpoint = '/api/v4/search/bvn';
            const payload = { BVN: request.BVN };
            return await this.circuitBreaker.execute(async () => {
                return await this.retryStrategy.execute(async () => {
                    const response = await this.client.post(endpoint, payload);
                    if (response.data.error) {
                        const raw = String(response.data.error || '').toLowerCase();
                        if (raw.includes('no record')) {
                            throw new verification_types_1.ProviderError('No record found', 404);
                        }
                        if (raw.includes('try again later') || raw.includes('temporary') || raw.includes('currently unavailable')) {
                            throw new verification_types_1.ProviderError('Provider temporarily unavailable, try again later', 503);
                        }
                        if (raw.includes('nonetype') || raw.includes('startswith')) {
                            throw new verification_types_1.ProviderError('BVN verification service is temporarily unavailable due to provider issues. Please try again later.', 503);
                        }
                        throw new verification_types_1.ProviderError(response.data.error, 400);
                    }
                    return response.data;
                }, (error) => {
                    const status = error?.statusCode || error?.status || error?.response?.status;
                    return (status >= 500) || status === 504 || status === 503 || status === 429 || status === 408;
                });
            });
        }
        catch (error) {
            if (error instanceof verification_types_1.ValidationError || error instanceof verification_types_1.AuthenticationError || error instanceof verification_types_1.ProviderError) {
                throw error;
            }
            throw new verification_types_1.ProviderError(`BVN verification failed: ${error instanceof Error ? error.message : String(error)}`);
        }
    }
    async verifyBvnByPhone(request) {
        try {
            if (!request.phone || !/^\d{10,11}$/.test(request.phone)) {
                throw new verification_types_1.ValidationError('Phone number must be 10-11 digits');
            }
            this.ensureApiKeyAvailable();
            const endpoint = '/api/v4/search/bvn';
            const payload = { phone: request.phone };
            if (request.firstName)
                payload.firstName = request.firstName;
            if (request.lastName)
                payload.lastName = request.lastName;
            const response = await this.client.post(endpoint, payload);
            if (response.data?.error) {
                const raw = String(response.data.error || '').toLowerCase();
                if (raw.includes('no record')) {
                    throw new verification_types_1.ProviderError('No record found', 404);
                }
                if (raw.includes('try again later') || raw.includes('temporary') || raw.includes('currently unavailable')) {
                    throw new verification_types_1.ProviderError('Provider temporarily unavailable, try again later', 503);
                }
                throw new verification_types_1.ProviderError(response.data.error, 400);
            }
            return response.data;
        }
        catch (error) {
            if (error instanceof verification_types_1.ValidationError || error instanceof verification_types_1.AuthenticationError || error instanceof verification_types_1.ProviderError) {
                throw error;
            }
            throw new verification_types_1.ProviderError(`BVN-by-phone verification failed: ${error instanceof Error ? error.message : String(error)}`);
        }
    }
    async createIpeClearance(request) {
        try {
            if (!request.type || !request.tracking_ids || !Array.isArray(request.tracking_ids) || request.tracking_ids.length === 0) {
                throw new verification_types_1.ValidationError('Invalid IPE clearance request. Type and tracking_ids array are required.');
            }
            this.ensureApiKeyAvailable();
            const endpoint = '/api/v4/transaction/create';
            const response = await this.client.post(endpoint, request);
            if (response.data.error) {
                throw new verification_types_1.ProviderError(response.data.error, 400);
            }
            return response.data;
        }
        catch (error) {
            if (error instanceof verification_types_1.ValidationError || error instanceof verification_types_1.AuthenticationError || error instanceof verification_types_1.ProviderError) {
                throw error;
            }
            throw new verification_types_1.ProviderError(`IPE clearance creation failed: ${error instanceof Error ? error.message : String(error)}`);
        }
    }
    async getIpeClearanceStatus(transactionId) {
        try {
            if (!transactionId || transactionId <= 0) {
                throw new verification_types_1.ValidationError('Invalid transaction ID');
            }
            this.ensureApiKeyAvailable();
            const endpoint = `/api/v4/transactions/${transactionId}`;
            const response = await this.client.get(endpoint);
            if (response.data.error) {
                throw new verification_types_1.ProviderError(response.data.error, 400);
            }
            return response.data;
        }
        catch (error) {
            if (error instanceof verification_types_1.ValidationError || error instanceof verification_types_1.AuthenticationError || error instanceof verification_types_1.ProviderError) {
                throw error;
            }
            throw new verification_types_1.ProviderError(`IPE clearance status check failed: ${error instanceof Error ? error.message : String(error)}`);
        }
    }
    async createPersonalization(request) {
        try {
            if (!request.type || !request.tracking_id) {
                throw new verification_types_1.ValidationError('Invalid personalization request. Type and tracking_id are required.');
            }
            this.ensureApiKeyAvailable();
            const endpoint = '/api/v4/transaction/create';
            const response = await this.client.post(endpoint, request);
            if (response.data.error) {
                throw new verification_types_1.ProviderError(response.data.error, 400);
            }
            return response.data;
        }
        catch (error) {
            if (error instanceof verification_types_1.ValidationError || error instanceof verification_types_1.AuthenticationError || error instanceof verification_types_1.ProviderError) {
                throw error;
            }
            throw new verification_types_1.ProviderError(`Personalization creation failed: ${error instanceof Error ? error.message : String(error)}`);
        }
    }
    async healthCheck() {
        try {
            this.ensureApiKeyAvailable();
            const endpoint = '/api/v4/profile';
            const response = await this.client.get(endpoint);
            return response.status === 200 && !response.data.error;
        }
        catch (error) {
            console.error('[UnifyProvider] Health check failed:', error);
            return false;
        }
    }
    getCircuitBreakerStatus() {
        return this.circuitBreaker.getStats();
    }
    resetCircuitBreaker() {
        this.circuitBreaker.reset();
    }
    validateNinRequest(request) {
        const hasNin = request.nin && request.nin.length === 11;
        const hasNameAndDetails = request.first_name && request.last_name && request.gender && request.dob;
        const hasPhone = request.phone && request.phone.length >= 10;
        if (!hasNin && !hasNameAndDetails && !hasPhone) {
            throw new verification_types_1.ValidationError('NIN verification requires either: NIN (11 digits), name/gender/dob combination, or phone number');
        }
        if (request.nin && !/^\d{11}$/.test(request.nin)) {
            throw new verification_types_1.ValidationError('NIN must be exactly 11 digits');
        }
        if (request.phone && !/^\d{10,11}$/.test(request.phone)) {
            throw new verification_types_1.ValidationError('Phone number must be 10-11 digits');
        }
        if (request.gender && !['male', 'female'].includes(request.gender)) {
            throw new verification_types_1.ValidationError('Gender must be either "male" or "female"');
        }
        if (request.dob && !/^\d{2}-\d{2}-\d{4}$/.test(request.dob)) {
            throw new verification_types_1.ValidationError('Date of birth must be in format: DD-MM-YYYY');
        }
    }
    ensureApiKeyAvailable() {
        const apiKeyFromDefaults = this.client.defaults.headers.common['x-api-key'];
        const apiKeyFromConfig = this.config.apiKey;
        const hasApiKey = (apiKeyFromDefaults && typeof apiKeyFromDefaults === 'string' && apiKeyFromDefaults.length > 0) ||
            (apiKeyFromConfig && apiKeyFromConfig.length > 0);
        if (!hasApiKey) {
            throw new verification_types_1.AuthenticationError('API key is not configured. Please ensure the verification provider API key is properly set.');
        }
    }
}
exports.UnifyProvider = UnifyProvider;
